import type { DataSource, DataQualityMetrics, DataQualityIssue } from './types';

interface ValidationResult {
  validRecords: any[];
  invalidRecords: Array<{ record: any; errors: string[] }>;
}

interface ValidationRule {
  field: string;
  type: 'required' | 'type' | 'range' | 'format' | 'unique' | 'custom';
  config: any;
}

export class DataValidationService {
  private static instance: DataValidationService;

  private constructor() {}

  static getInstance(): DataValidationService {
    if (!DataValidationService.instance) {
      DataValidationService.instance = new DataValidationService();
    }
    return DataValidationService.instance;
  }

  async validateData(records: any[], source: DataSource): Promise<ValidationResult> {
    const result: ValidationResult = {
      validRecords: [],
      invalidRecords: [],
    };

    const validationRules = this.getValidationRules(source);

    for (const record of records) {
      const errors = this.validateRecord(record, validationRules);

      if (errors.length === 0) {
        result.validRecords.push(record);
      } else {
        result.invalidRecords.push({ record, errors });
      }
    }

    return result;
  }

  private getValidationRules(source: DataSource): ValidationRule[] {
    const baseRules: ValidationRule[] = [];

    // Domain-specific validation rules
    switch (source.domain) {
      case 'sports':
        baseRules.push(
          { field: 'date', type: 'required', config: {} },
          { field: 'team', type: 'required', config: {} },
          { field: 'result', type: 'format', config: { pattern: /^(win|loss)$/i } },
          { field: 'odds', type: 'type', config: { expectedType: 'number' } }
        );
        break;

      case 'crypto':
        baseRules.push(
          { field: 'timestamp', type: 'required', config: {} },
          { field: 'asset', type: 'required', config: {} },
          { field: 'open', type: 'type', config: { expectedType: 'number' } },
          { field: 'close', type: 'type', config: { expectedType: 'number' } },
          { field: 'volume', type: 'type', config: { expectedType: 'number' } }
        );
        break;

      case 'stocks':
        baseRules.push(
          { field: 'date', type: 'required', config: {} },
          { field: 'ticker', type: 'required', config: {} },
          { field: 'open', type: 'type', config: { expectedType: 'number' } },
          { field: 'close', type: 'type', config: { expectedType: 'number' } }
        );
        break;

      case 'forex':
        baseRules.push(
          { field: 'timestamp', type: 'required', config: {} },
          { field: 'pair', type: 'required', config: {} },
          { field: 'open', type: 'type', config: { expectedType: 'number' } },
          { field: 'close', type: 'type', config: { expectedType: 'number' } },
          { field: 'high', type: 'type', config: { expectedType: 'number' } },
          { field: 'low', type: 'type', config: { expectedType: 'number' } },
          { field: 'spread', type: 'type', config: { expectedType: 'number' } }
        );
        break;
    }

    // Add custom rules from source config
    if (source.config.schemaValidation && source.config.transformations) {
      // Extract validation rules from transformations
      for (const transform of source.config.transformations) {
        if (transform.operation === 'validate') {
          baseRules.push({
            field: transform.field,
            type: 'custom',
            config: transform.config,
          });
        }
      }
    }

    return baseRules;
  }

  private validateRecord(record: any, rules: ValidationRule[]): string[] {
    const errors: string[] = [];

    for (const rule of rules) {
      const value = record[rule.field];
      const error = this.validateField(value, rule);

      if (error) {
        errors.push(`${rule.field}: ${error}`);
      }
    }

    return errors;
  }

  private validateField(value: any, rule: ValidationRule): string | null {
    switch (rule.type) {
      case 'required':
        if (value == null || value === '' || value === undefined) {
          return 'Field is required';
        }
        break;

      case 'type':
        const expectedType = rule.config.expectedType;
        if (value != null && typeof value !== expectedType) {
          if (expectedType === 'number' && isNaN(Number(value))) {
            return `Expected ${expectedType}, got ${typeof value}`;
          }
        }
        break;

      case 'range':
        if (typeof value === 'number') {
          const { min, max } = rule.config;
          if ((min !== undefined && value < min) || (max !== undefined && value > max)) {
            return `Value ${value} is outside range [${min || '-∞'}, ${max || '∞'}]`;
          }
        }
        break;

      case 'format':
        if (typeof value === 'string') {
          const pattern = new RegExp(rule.config.pattern);
          if (!pattern.test(value)) {
            return `Value "${value}" does not match required format`;
          }
        }
        break;

      case 'custom':
        if (rule.config.validator && !rule.config.validator(value)) {
          return rule.config.message || 'Custom validation failed';
        }
        break;
    }

    return null;
  }

  async checkDataQuality(records: any[], source: DataSource): Promise<DataQualityMetrics> {
    const metrics: DataQualityMetrics = {
      sourceId: source.id,
      totalRecords: records.length,
      duplicateRecords: 0,
      invalidRecords: 0,
      missingValues: {},
      dataCompleteness: 0,
      lastCheck: new Date(),
      issues: [],
    };

    // Check for duplicates
    const seen = new Set<string>();
    const duplicates = new Set<any>();

    for (const record of records) {
      const key = this.createRecordKey(record, source);
      if (seen.has(key)) {
        duplicates.add(record);
      }
      seen.add(key);
    }

    metrics.duplicateRecords = duplicates.size;

    // Check missing values and data completeness
    const fieldCounts: Record<string, number> = {};
    const totalFields = records.length * this.getExpectedFields(source).length;

    for (const record of records) {
      for (const field of this.getExpectedFields(source)) {
        if (record[field] == null || record[field] === '') {
          fieldCounts[field] = (fieldCounts[field] || 0) + 1;
        }
      }
    }

    metrics.missingValues = fieldCounts;
    const totalMissing = Object.values(fieldCounts).reduce((sum, count) => sum + count, 0);
    metrics.dataCompleteness = (totalFields - totalMissing) / totalFields;

    // Check for invalid records
    const validation = await this.validateData(records, source);
    metrics.invalidRecords = validation.invalidRecords.length;

    // Generate issues
    metrics.issues = this.generateQualityIssues(metrics, source);

    return metrics;
  }

  private createRecordKey(record: any, source: DataSource): string {
    // Create a unique key based on primary fields for duplicate detection
    const keyFields = this.getKeyFields(source);
    return keyFields.map(field => record[field]).join('|');
  }

  private getKeyFields(source: DataSource): string[] {
    switch (source.domain) {
      case 'sports':
        return ['date', 'team', 'opponent'];
      case 'crypto':
        return ['timestamp', 'asset'];
      case 'stocks':
        return ['date', 'ticker'];
      case 'forex':
        return ['timestamp', 'pair'];
      default:
        return ['id'];
    }
  }

  private getExpectedFields(source: DataSource): string[] {
    switch (source.domain) {
      case 'sports':
        return ['date', 'team', 'opponent', 'result', 'odds'];
      case 'crypto':
        return ['timestamp', 'asset', 'open', 'close', 'volume'];
      case 'stocks':
        return ['date', 'ticker', 'open', 'close'];
      case 'forex':
        return ['timestamp', 'pair', 'open', 'close', 'high', 'low', 'spread'];
      default:
        return [];
    }
  }

  private generateQualityIssues(metrics: DataQualityMetrics, source: DataSource): DataQualityIssue[] {
    const issues: DataQualityIssue[] = [];

    // Missing values issues
    for (const [field, count] of Object.entries(metrics.missingValues)) {
      const percentage = (count / metrics.totalRecords) * 100;
      if (percentage > 10) {
        issues.push({
          type: 'missing_values',
          severity: percentage > 50 ? 'critical' : percentage > 25 ? 'high' : 'medium',
          field,
          description: `${percentage.toFixed(1)}% of records missing ${field}`,
          affectedRecords: count,
          suggestedAction: `Review data source for ${field} field collection`,
        });
      }
    }

    // Duplicate records
    if (metrics.duplicateRecords > 0) {
      const percentage = (metrics.duplicateRecords / metrics.totalRecords) * 100;
      issues.push({
        type: 'duplicates',
        severity: percentage > 5 ? 'high' : 'medium',
        description: `${metrics.duplicateRecords} duplicate records found (${percentage.toFixed(1)}%)`,
        affectedRecords: metrics.duplicateRecords,
        suggestedAction: 'Implement deduplication logic or review data source',
      });
    }

    // Invalid records
    if (metrics.invalidRecords > 0) {
      const percentage = (metrics.invalidRecords / metrics.totalRecords) * 100;
      issues.push({
        type: 'invalid_format',
        severity: percentage > 20 ? 'high' : 'medium',
        description: `${metrics.invalidRecords} records failed validation (${percentage.toFixed(1)}%)`,
        affectedRecords: metrics.invalidRecords,
        suggestedAction: 'Review validation rules or clean data source',
      });
    }

    // Data completeness
    if (metrics.dataCompleteness < 0.8) {
      issues.push({
        type: 'missing_values',
        severity: metrics.dataCompleteness < 0.5 ? 'critical' : 'high',
        description: `Data completeness: ${(metrics.dataCompleteness * 100).toFixed(1)}%`,
        affectedRecords: Math.round(metrics.totalRecords * (1 - metrics.dataCompleteness)),
        suggestedAction: 'Improve data collection or add default values for missing fields',
      });
    }

    return issues;
  }
}

