import fs from 'fs/promises';
import path from 'path';
import { parse } from 'csv-parse/sync';
import crypto from 'crypto';

import type {
  DataSource,
  IngestionResult,
  DataQualityMetrics,
  DataVersion,
} from './types';
import { DataValidationService } from './DataValidationService';
import { DataStorageService } from './DataStorageService';

export class DataIngestionService {
  private static instance: DataIngestionService;
  private validationService: DataValidationService;
  private storageService: DataStorageService;
  private activeIngestions: Map<string, Promise<IngestionResult>> = new Map();

  private constructor() {
    this.validationService = DataValidationService.getInstance();
    this.storageService = DataStorageService.getInstance();
  }

  static getInstance(): DataIngestionService {
    if (!DataIngestionService.instance) {
      DataIngestionService.instance = new DataIngestionService();
    }
    return DataIngestionService.instance;
  }

  async ingestFromSource(source: DataSource): Promise<IngestionResult> {
    const startTime = Date.now();

    // Prevent concurrent ingestion of the same source
    if (this.activeIngestions.has(source.id)) {
      throw new Error(`Ingestion already in progress for source ${source.id}`);
    }

    const ingestionPromise = this.performIngestion(source);
    this.activeIngestions.set(source.id, ingestionPromise);

    try {
      const result = await ingestionPromise;
      result.duration = Date.now() - startTime;
      return result;
    } finally {
      this.activeIngestions.delete(source.id);
    }
  }

  private async performIngestion(source: DataSource): Promise<IngestionResult> {
    const result: IngestionResult = {
      sourceId: source.id,
      success: false,
      recordsProcessed: 0,
      recordsInserted: 0,
      recordsUpdated: 0,
      recordsFailed: 0,
      errors: [],
      duration: 0,
      timestamp: new Date(),
    };

    try {
      let rawData: any[];

      // Extract data based on source type
      switch (source.type) {
        case 'csv':
          rawData = await this.loadCSV(source);
          break;
        case 'api':
          rawData = await this.fetchAPI(source);
          break;
        case 'database':
          rawData = await this.queryDatabase(source);
          break;
        default:
          throw new Error(`Unsupported source type: ${source.type}`);
      }

      result.recordsProcessed = rawData.length;

      // Validate data
      const validationResult = await this.validationService.validateData(rawData, source);
      result.recordsFailed = validationResult.invalidRecords.length;

      // Apply transformations
      const transformedData = await this.applyTransformations(validationResult.validRecords, source);

      // Store data with versioning
      const storageResult = await this.storageService.storeData(source, transformedData);

      result.recordsInserted = storageResult.inserted;
      result.recordsUpdated = storageResult.updated;
      result.success = true;

      // Update source metadata
      await this.updateSourceMetadata(source, result);

    } catch (error) {
      result.errors.push(error instanceof Error ? error.message : 'Unknown error');
      console.error(`Ingestion failed for source ${source.id}:`, error);
    }

    return result;
  }

  private async loadCSV(source: DataSource): Promise<any[]> {
    if (!source.filePath) {
      throw new Error('CSV source must have filePath configured');
    }

    const filePath = path.resolve(source.filePath);
    const content = await fs.readFile(filePath, (source.config.encoding || 'utf8') as BufferEncoding);

    const records = parse(content, {
      delimiter: source.config.delimiter || ',',
      skip_empty_lines: true,
      trim: true,
      from_line: source.config.hasHeaders ? 2 : 1,
      columns: source.config.hasHeaders ? true : false,
    });

    return Array.isArray(records) ? records : [records];
  }

  private async fetchAPI(source: DataSource): Promise<any[]> {
    if (!source.endpoint) {
      throw new Error('API source must have endpoint configured');
    }

    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
      ...source.config.headers,
    };

    if (source.config.apiKey) {
      headers['Authorization'] = `Bearer ${source.config.apiKey}`;
    }

    const response = await fetch(source.endpoint, {
      method: 'GET',
      headers,
      signal: AbortSignal.timeout(source.config.timeout || 30000),
    });

    if (!response.ok) {
      throw new Error(`API request failed: ${response.status} ${response.statusText}`);
    }

    const data = await response.json();

    // Handle different API response formats
    if (Array.isArray(data)) {
      return data;
    } else if (data.data && Array.isArray(data.data)) {
      return data.data;
    } else if (data.results && Array.isArray(data.results)) {
      return data.results;
    } else {
      // Single object response
      return [data];
    }
  }

  private async queryDatabase(source: DataSource): Promise<any[]> {
    // Placeholder for database queries
    // In a real implementation, this would use a database client
    throw new Error('Database ingestion not yet implemented');
  }

  private async applyTransformations(data: any[], source: DataSource): Promise<any[]> {
    if (!source.config.transformations || source.config.transformations.length === 0) {
      return data;
    }

    let transformedData = [...data];

    for (const transformation of source.config.transformations) {
      transformedData = this.applyTransformation(transformedData, transformation);
    }

    return transformedData;
  }

  private applyTransformation(data: any[], transformation: any): any[] {
    switch (transformation.operation) {
      case 'rename':
        return data.map(record => ({
          ...record,
          [transformation.config.newName]: record[transformation.field],
          [transformation.field]: undefined,
        }));

      case 'cast':
        return data.map(record => ({
          ...record,
          [transformation.field]: this.castValue(record[transformation.field], transformation.config.type),
        }));

      case 'transform':
        return data.map(record => ({
          ...record,
          [transformation.field]: transformation.config.function(record[transformation.field], record),
        }));

      case 'validate':
        return data.filter(record => {
          const value = record[transformation.field];
          return transformation.config.validator(value);
        });

      case 'filter':
        return data.filter(record => transformation.config.condition(record));

      default:
        return data;
    }
  }

  private castValue(value: any, type: string): any {
    if (value == null || value === '') return null;

    switch (type) {
      case 'string':
        return String(value);
      case 'number':
        const num = Number(value);
        return isNaN(num) ? null : num;
      case 'boolean':
        return Boolean(value);
      case 'date':
        return new Date(value);
      default:
        return value;
    }
  }

  private async updateSourceMetadata(source: DataSource, result: IngestionResult): Promise<void> {
    // In a real implementation, this would update a database
    source.lastUpdated = result.timestamp;
    source.recordCount = (source.recordCount || 0) + result.recordsInserted;
    source.status = result.success ? 'active' : 'error';
  }

  async getIngestionStatus(sourceId: string): Promise<IngestionResult | null> {
    return this.activeIngestions.get(sourceId) || null;
  }

  async getAllActiveIngestions(): Promise<string[]> {
    return Array.from(this.activeIngestions.keys());
  }
}

