import fs from 'node:fs/promises';
import path from 'node:path';
import { parse } from 'csv-parse/sync';

import {
  CryptoBarRecord,
  SportsGameRecord,
  StockBarRecord,
  ForexBarRecord,
} from './types';

const csvRoot = path.join(process.cwd(), 'data', 'csv');
const dataRoot = path.join(process.cwd(), 'data');

async function readCsv(fileName: string) {
  const filePath = path.join(csvRoot, fileName);
  return fs.readFile(filePath, 'utf-8');
}

async function readJson(filePath: string): Promise<any[]> {
  try {
    const fullPath = path.join(dataRoot, filePath);
    const content = await fs.readFile(fullPath, 'utf-8');
    return JSON.parse(content);
  } catch (error) {
    console.error(`Failed to read JSON file: ${filePath}`, error);
    return [];
  }
}

function asRecords<T>(raw: string, mapper: (row: any, idx: number) => T): T[] {
  const rows = parse(raw, {
    columns: true,
    skip_empty_lines: true,
    trim: true,
  }) as Record<string, string>[];

  return rows.map(mapper);
}

// Load games and odds from BallDontLie JSON data
async function loadBallDontLieData(sport: string): Promise<{ games: any[], odds: any[] }> {
  const sportLower = sport.toLowerCase();
  let gamesPath = '';
  let oddsPath = '';

  switch (sportLower) {
    case 'nba':
      gamesPath = 'nba/games.json';
      oddsPath = 'nba/odds.json';
      break;
    case 'nfl':
      gamesPath = 'nfl/games.json';
      oddsPath = 'odds/nfl_odds.json';
      break;
    case 'nhl':
      gamesPath = 'nhl/games.json';
      oddsPath = 'odds/nhl_odds.json';
      break;
    case 'mlb':
      gamesPath = 'mlb/games.json';
      oddsPath = 'mlb/odds.json';
      break;
    default:
      gamesPath = 'nba/games.json';
      oddsPath = 'nba/odds.json';
  }

  const [games, odds] = await Promise.all([
    readJson(gamesPath),
    readJson(oddsPath)
  ]);

  return { games, odds };
}

// Convert BallDontLie game format to our SportsGameRecord format
function convertBallDontLieGame(game: any, odds: any[], sport: string, idx: number): SportsGameRecord {
  const homeTeam = game.home_team?.full_name || game.home_team?.name || 'Home Team';
  const awayTeam = game.visitor_team?.full_name || game.visitor_team?.name || 'Away Team';
  const homeScore = game.home_team_score || 0;
  const awayScore = game.visitor_team_score || 0;
  const homeWon = homeScore > awayScore;

  // Find matching odds for this game - prefer 2way (moneyline) type
  const gameOddsList = odds.filter(o => o.game_id === game.id);
  const moneylineOdds = gameOddsList.find(o => o.type === '2way') || gameOddsList[0];
  const spreadOdds = gameOddsList.find(o => o.type === 'spread');
  
  // Calculate spread from odds if available
  let spread = 0;
  let oddsValue = '+100';
  let awayOddsValue = '+100';
  let spreadOddsHome = '-110';
  let spreadOddsAway = '-110';
  let isFavorite = false;
  
  if (moneylineOdds) {
    // Use moneyline odds - negative means favorite
    const mlHome = moneylineOdds.odds_american_home || moneylineOdds.moneyline_home_odds;
    const mlAway = moneylineOdds.odds_american_visitor || moneylineOdds.odds_american_away || moneylineOdds.moneyline_away_odds;
    
    if (mlHome) {
      const mlValue = typeof mlHome === 'string' ? parseInt(mlHome) : mlHome;
      oddsValue = mlValue > 0 ? `+${mlValue}` : `${mlValue}`;
      isFavorite = mlValue < 0; // Negative odds = favorite
      
      // Estimate spread from moneyline if no spread data
      // Rough conversion: every 20 points of ML ≈ 1 point spread
      if (!spreadOdds) {
        spread = isFavorite ? Math.round(Math.abs(mlValue) / 20) * -1 : Math.round(Math.abs(mlValue) / 20);
      }
    }
    
    if (mlAway) {
      const mlAwayValue = typeof mlAway === 'string' ? parseInt(mlAway) : mlAway;
      awayOddsValue = mlAwayValue > 0 ? `+${mlAwayValue}` : `${mlAwayValue}`;
    }
  }
  
  // Use actual spread if available
  if (spreadOdds) {
    const awaySpread = parseFloat(spreadOdds.away_spread) || 0;
    spread = -awaySpread; // If away spread is +2.5, home spread is -2.5
    
    // NFL format has different field names
    if (spreadOdds.spread_home_value) {
      spread = parseFloat(spreadOdds.spread_home_value) || 0;
    }
    
    // Get spread odds (vig) - typically -110 each side
    const spreadHomeOdds = spreadOdds.odds_american_home || spreadOdds.spread_home_odds;
    const spreadAwayOddsVal = spreadOdds.odds_american_visitor || spreadOdds.spread_away_odds;
    
    if (spreadHomeOdds) {
      const sho = typeof spreadHomeOdds === 'string' ? parseInt(spreadHomeOdds) : spreadHomeOdds;
      spreadOddsHome = sho > 0 ? `+${sho}` : `${sho}`;
    }
    if (spreadAwayOddsVal) {
      const sao = typeof spreadAwayOddsVal === 'string' ? parseInt(spreadAwayOddsVal) : spreadAwayOddsVal;
      spreadOddsAway = sao > 0 ? `+${sao}` : `${sao}`;
    }
  }

  return {
    id: `${sport}-${game.id || idx}`,
    type: 'sports' as const,
    sport: sport.toLowerCase(),
    date: game.date || game.datetime?.split('T')[0] || new Date().toISOString().split('T')[0],
    team: homeTeam,
    opponent: awayTeam,
    result: homeWon ? 'win' : 'loss',
    margin: homeScore - awayScore,
    odds: oddsValue,
    awayOdds: awayOddsValue,
    spread: spread,
    spreadOddsHome: spreadOddsHome,
    spreadOddsAway: spreadOddsAway,
    injuries: 'N/A',
    weather: 'Indoor',
    losingStreakBefore: 0,
    // Extended fields from BallDontLie
    homeScore,
    awayScore,
    season: game.season,
    isPostseason: game.postseason || false,
    gameId: game.id,
  };
}

export async function loadSportsGames(sportFilter?: string): Promise<SportsGameRecord[]> {
  // Determine which sport to load
  const sport = sportFilter || 'nba';
  
  try {
    // Try to load from BallDontLie JSON data first
    const { games, odds } = await loadBallDontLieData(sport);
    
    if (games.length > 0) {
      console.log(`[csvLoader] Loaded ${games.length} ${sport.toUpperCase()} games and ${odds.length} odds from BallDontLie data`);
      
      // Convert to our format
      const records = games
        .filter((g: any) => g.status === 'Final' || g.home_team_score > 0) // Only completed games
        .map((game: any, idx: number) => convertBallDontLieGame(game, odds, sport, idx));

      // Sort by date
      const sorted = records.sort(
        (a, b) => new Date(a.date).getTime() - new Date(b.date).getTime(),
      );

      // Calculate losing streaks
      const streaks: Record<string, number> = {};
      
      return sorted.map((record) => {
        const currentStreak = streaks[record.team] ?? 0;
        const updated: SportsGameRecord = {
          ...record,
          losingStreakBefore: currentStreak,
        };

        if (record.result === 'loss') {
          streaks[record.team] = currentStreak + 1;
        } else {
          streaks[record.team] = 0;
        }

        return updated;
      });
    }
  } catch (error) {
    console.error(`[csvLoader] Failed to load BallDontLie data for ${sport}, falling back to CSV:`, error);
  }

  // Fallback to CSV if JSON not available
  console.log('[csvLoader] Falling back to CSV data');
  const raw = await readCsv('sports_games.csv');

  const rows = asRecords(raw, (row, idx) => ({
    id: `sports-${idx}`,
    type: 'sports' as const,
    sport: row.sport || 'unknown',
    date: row.date,
    team: row.team,
    opponent: row.opponent,
    result: (row.result?.toLowerCase() === 'win' ? 'win' : 'loss') as 'win' | 'loss',
    margin: Number(row.team_score || 0) - Number(row.opponent_score || 0),
    odds: row.odds || '+100',
    spread: Number(row.spread || 0),
    injuries: row.injuries || 'None',
    weather: row.weather || 'Indoor',
    losingStreakBefore: 0,
  }));

  // Filter by sport if specified
  const filtered = sportFilter
    ? rows.filter(r => r.sport.toLowerCase() === sportFilter.toLowerCase())
    : rows;

  const sorted = filtered.sort(
    (a, b) => new Date(a.date).getTime() - new Date(b.date).getTime(),
  );

  const streaks: Record<string, number> = {};

  return sorted.map((record) => {
    const currentStreak = streaks[record.team] ?? 0;
    const updated: SportsGameRecord = {
      ...record,
      losingStreakBefore: currentStreak,
    };

    if (record.result === 'loss') {
      streaks[record.team] = currentStreak + 1;
    } else {
      streaks[record.team] = 0;
    }

    return updated;
  });
}


export async function loadCryptoPrices(): Promise<CryptoBarRecord[]> {
  const raw = await readCsv('crypto_prices.csv');

  return asRecords(raw, (row, idx) => {
    const open = Number(row.open);
    const close = Number(row.close);

    return {
      id: `crypto-${idx}`,
      type: 'crypto' as const,
      timestamp: row.timestamp,
      asset: row.asset,
      open,
      close,
      changePct: (close - open) / open,
      volume: Number(row.volume),
      newsSentiment: row.news_sentiment,
    };
  });
}

export async function loadStockPrices(): Promise<StockBarRecord[]> {
  const raw = await readCsv('stock_prices.csv');

  return asRecords(raw, (row, idx) => {
    const open = Number(row.open);
    const close = Number(row.close);

    return {
      id: `stock-${idx}`,
      type: 'stocks' as const,
      date: row.date,
      ticker: row.ticker,
      open,
      close,
      changePct: (close - open) / open,
      volume: Number(row.volume),
      earningSurprise: Number(row.earning_surprise),
      macroSignal: row.macro_signal,
    };
  });
}

export async function loadForexPairs(): Promise<ForexBarRecord[]> {
  const raw = await readCsv('forex_pairs.csv');

  return asRecords(raw, (row, idx) => {
    const open = Number(row.open);
    const close = Number(row.close);
    const high = Number(row.high);
    const low = Number(row.low);

    return {
      id: `forex-${idx}`,
      type: 'forex' as const,
      timestamp: row.timestamp,
      pair: row.pair,
      open,
      close,
      high,
      low,
      changePct: (close - open) / open,
      volume: Number(row.volume),
      spread: Number(row.spread),
      newsImpact: row.news_impact,
    };
  });
}

