import { RAGService, RAGInsight } from '@/services/RAGService';
import { BallDontLieService } from '@/services/BallDontLieService';
import { MemoryService } from '@/lib/memory/MemoryService';
import type { ConversationMemory, MemoryContext } from '@/lib/memory/types';
import { runParlayBacktest, runRoundRobinBacktest } from '@/lib/sportsBettingData';
import { spawn } from 'child_process';
import path from 'path';
import fs from 'fs';

// AI Guru Settings interface and loader
interface AIGuruSettings {
  temperature: number;
  maxTokens: number;
  ragEnabled: boolean;
  ragRelevanceThreshold: number;
  useEnrichedChunks: boolean;
  maxContextLength: number;
  ragTopK: number;
  systemPromptPrefix: string;
  systemPromptSuffix: string;
  includeThemesInResponse: boolean;
  includeKeyPointsInResponse: boolean;
  includeSourcesInResponse: boolean;
  maxQueriesPerMinute: number;
  maxQueriesPerDay: number;
}

const DEFAULT_AI_GURU_SETTINGS: AIGuruSettings = {
  temperature: 0.7,
  maxTokens: 2000,
  ragEnabled: true,
  ragRelevanceThreshold: 0.6,
  useEnrichedChunks: true,
  maxContextLength: 3000,
  ragTopK: 5,
  systemPromptPrefix: '',
  systemPromptSuffix: '',
  includeThemesInResponse: true,
  includeKeyPointsInResponse: true,
  includeSourcesInResponse: true,
  maxQueriesPerMinute: 10,
  maxQueriesPerDay: 100,
};

// Cache settings to avoid reading file on every request
let cachedSettings: AIGuruSettings | null = null;
let settingsLastLoaded = 0;
const SETTINGS_CACHE_TTL = 30000; // 30 seconds

function loadAIGuruSettings(): AIGuruSettings {
  const now = Date.now();
  if (cachedSettings && (now - settingsLastLoaded) < SETTINGS_CACHE_TTL) {
    return cachedSettings;
  }
  
  try {
    const settingsPath = path.join(process.cwd(), 'data', 'ai-guru-settings.json');
    if (fs.existsSync(settingsPath)) {
      const data = fs.readFileSync(settingsPath, 'utf-8');
      cachedSettings = { ...DEFAULT_AI_GURU_SETTINGS, ...JSON.parse(data) };
      settingsLastLoaded = now;
      return cachedSettings;
    }
  } catch (error) {
    console.warn('Failed to load AI Guru settings:', error);
  }
  
  return DEFAULT_AI_GURU_SETTINGS;
}

// Basic queue interface for future BullMQ implementation
interface QueueJob {
  id: string;
  type: 'backtest' | 'ai_generation';
  data: any;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  result?: any;
  error?: string;
  createdAt: Date;
  updatedAt: Date;
}

// File-based job queue (SECURITY FIX: Replace in-memory with persistent storage)
// TODO: Replace with BullMQ/Redis for production scalability
const JOB_QUEUE_FILE = './job-queue.json';

function loadJobQueue(): Map<string, QueueJob> {
  try {
    const fs = require('fs');
    if (fs.existsSync(JOB_QUEUE_FILE)) {
      const data = fs.readFileSync(JOB_QUEUE_FILE, 'utf8');
      const parsed = JSON.parse(data);
      // Convert back to Map and restore Date objects
      const map = new Map();
      for (const [key, value] of Object.entries(parsed)) {
        map.set(key, {
          ...value,
          createdAt: new Date(value.createdAt),
          updatedAt: new Date(value.updatedAt)
        });
      }
      return map;
    }
  } catch (error) {
    console.warn('Failed to load job queue:', error);
  }
  return new Map();
}

function saveJobQueue(queue: Map<string, QueueJob>): void {
  try {
    const fs = require('fs');
    // Convert Map to plain object for JSON serialization
    const obj = Object.fromEntries(queue);
    fs.writeFileSync(JOB_QUEUE_FILE, JSON.stringify(obj, null, 2));
  } catch (error) {
    console.error('Failed to save job queue:', error);
  }
}

// Global job queue instance
let jobQueue: Map<string, QueueJob> = loadJobQueue();

export interface ChatMessage {
  id: string;
  role: 'user' | 'assistant';
  content: string;
  timestamp: Date;
  ragInsight?: RAGInsight;
  visualization?: {
    chartType: string;
    market: string;
    format: string;
    imageBase64?: string;
    chartData?: any;
    trades?: any[];
  };
  backtestResults?: any;
}

export interface ChatResponse {
  message: ChatMessage;
  strategyCode?: string;
  strategyName?: string;
  ragInsight?: RAGInsight;
  memoryContext?: MemoryContext;
  visualization?: any;
}

const SYSTEM_PROMPT = `You are a sports betting backtesting expert. You specialize in single bets, parlays, and round robin strategies for NBA, NFL, MLB, NHL, NCAA, and other sports leagues.

DATA SOURCE: All sports data is provided exclusively by the BallDontLie API. Do NOT reference NBA.com, ESPN, Bet365, or any other data sources in your responses. The data source is always "BallDontLie API" for all sports data, games, scores, and statistics.

CRITICAL REQUIREMENT: Users MUST specify a time period for backtesting. If no time period is specified in the user's message, you MUST respond asking them to specify one before proceeding with strategy generation.

TIME PERIOD REQUIREMENTS:
- Always require season/year specification (e.g., "2024-25 season", "2024 NBA season", "2025 NFL season")
- If no time period specified, respond with: "Please specify the time period for backtesting (e.g., '2024-25 NBA season' or '2024 NFL season')"
- Default to 2024-25 season for current season references

When given a user strategy description, generate clean PYTHON code that implements the strategy logic.

Sports Betting Code Requirements:
1. Define a function: def generate_bets(games: list, players: list, teams: list, params: dict) -> list:
2. Return a list of betting recommendations with: {'game_id', 'bet_type', 'prediction', 'confidence', 'stake'}
3. bet_type options: 'moneyline', 'spread', 'over_under', 'player_points', 'player_prop'
4. prediction examples: 'home', 'away', 'over', 'under', or specific values
5. confidence: 0.0-1.0 (higher = more confident)
6. Access games, players, teams data arrays
7. RESPECT TIME PERIOD: Only generate bets for games within the specified backtesting period
8. Wrap the code in \`\`\`python ... \`\`\` blocks.

For parlays and round robins:
- Calculate combined odds by multiplying individual leg odds
- Apply stake to get payout: payout = stake × combined_odds
- Only pay out if ALL legs win (parlays) or specific combinations win (round robins)
- Include risk warnings about the high variance nature of these bets

COMPLEX/STATEFUL STRATEGIES:
For strategies requiring state (e.g., progressive betting, bankroll management):
- You MAY iterate through the games to maintain betting state.
- Initialize state variables (e.g., \`bankroll = 1000\`, \`streak = 0\`) before the loop.
- Ensure you handle both winning and losing bet outcomes.
- This platform focuses exclusively on sports betting - do not reference stocks, crypto, or forex.`;

const GURU_QA_PROMPT = `You are AI Sport Guru for EventheOdds.ai — a sports analytics assistant.

DATA SOURCE: All sports data is provided exclusively by the BallDontLie API and our server-side BallDontLie cache. Do NOT reference NBA.com, ESPN, Bet365, or any other data sources.

BEHAVIOR:
- Answer the user's question directly and coherently.
- Prefer using any provided BallDontLie context instead of guessing.
- If the user asks for standings/odds/schedules/teams/players, respond with the requested data in a readable list or table.
- If data is missing or unavailable, say so clearly and suggest the closest alternative you can provide.
- This is SPORTS BETTING ONLY. Do not mention crypto, stocks, forex, or non-sports markets.
- For conceptual bankroll/EV/variance/staking questions, use a clear structure:
  1) Definition (1–2 lines)
  2) Formula (plain text, not code)
  3) Sports betting example (moneyline/spread/total)
  4) Practical guidance (e.g., fractional Kelly)
  5) Caveats (variance/limits/edge uncertainty)

OUTPUT RULES:
- Do NOT output code blocks unless the user explicitly asks for code.
- Do NOT invent functions or strategy implementations for pure data questions.
- Only discuss backtesting strategy/code when the user asks to backtest/simulate/run a strategy.
- Prefer a short, structured response. Use 3–6 bullet points when helpful. Keep the answer under ~200 words unless the user asks for more detail.
- If the user says "no code", never include code fences or pseudo-code.

SAFETY:
- Provide statistics and historical information only. Do not claim to predict winners or guarantee outcomes.`;

// TODO: Implement BullMQ for background processing of long-running AI tasks
// This would prevent request timeouts and improve user experience for complex backtests
// Current implementation processes synchronously with timeouts

export async function callGrok4Fast(message: string, domain: string, userId?: string): Promise<ChatResponse> {
  try {
    const memoryService = MemoryService.getInstance();
    const ragService = RAGService.getInstance();
    const aiGuruSettings = loadAIGuruSettings();

    const messageLower = message.toLowerCase();

    // Determine request intent early to avoid generating strategy code for simple data questions.
    const isBacktest =
      messageLower.includes('backtest') ||
      messageLower.includes('test strategy') ||
      messageLower.includes('run simulation') ||
      messageLower.includes('analyze performance');

    const isParlay =
      messageLower.includes('parlay') ||
      messageLower.includes('multi-leg') ||
      messageLower.includes('combination');

    const isRoundRobin =
      messageLower.includes('round robin') ||
      (messageLower.includes('by ') && messageLower.includes("'s"));

    const disallowCode =
      /\b(no code|without code|dont show code|don't show code|do not show code)\b/.test(messageLower);

    // Only treat as "wants code" when the user explicitly asks for code (and isn't explicitly asking for no code).
    const wantsCodeExplicitly =
      !disallowCode &&
      (
        /\b(strategy code|python code|source code)\b/.test(messageLower) ||
        (/\b(write|generate|create|show)\b/.test(messageLower) && /\b(code|script|function)\b/.test(messageLower))
      );
    const isDataQuery = /\b(team|teams|game|games|schedule|odds|injur|roster|lineup|standings|stats|data|props?)\b/i.test(message);

    const intent: 'simulation' | 'data' | 'analysis' =
      (isBacktest || isParlay || isRoundRobin || wantsCodeExplicitly)
        ? 'simulation'
        : isDataQuery
          ? 'data'
          : 'analysis';

    // Get user memory context
    let memoryContext: MemoryContext | undefined;
    if (userId) {
      memoryContext = await memoryService.buildMemoryContext(userId, message, domain);

      // Extract user preferences from the message
      const extractedPrefs = await memoryService.extractUserPreferences(message);
      if (Object.keys(extractedPrefs).length > 0) {
        await memoryService.updateUserProfile(userId, extractedPrefs);
      }
    }

    // Get RAG insights for sports betting context (if enabled in settings)
    let ragInsight: RAGInsight | null = null;
    if (aiGuruSettings.ragEnabled && intent !== 'data') {
      ragInsight = await ragService.getTradingInsights(message, 'sports', aiGuruSettings.useEnrichedChunks);
    }

    // Gate + sanitize RAG output to avoid incoherent "dumping" into the UI.
    if (ragInsight) {
      // Enforce the configured threshold for BOTH prompt injection + UI surfacing.
      if (ragInsight.relevanceScore <= aiGuruSettings.ragRelevanceThreshold) {
        ragInsight = null;
      } else {
        // If the user asks about a specific named concept, require it to appear in the RAG snippet.
        const mustInclude: string[] = [];
        if (messageLower.includes('kelly')) mustInclude.push('kelly');
        if (messageLower.includes('sharpe')) mustInclude.push('sharpe');
        if (messageLower.includes('sortino')) mustInclude.push('sortino');
        if (messageLower.includes('drawdown')) mustInclude.push('drawdown');

        const ragText =
          `${ragInsight.answer} ${ragInsight.sources?.map((s) => s.preview).join(' ') || ''}`.toLowerCase();

        if (mustInclude.some((term) => !ragText.includes(term))) {
          ragInsight = null;
        } else {
          // Keep the UI snippet short; the UI already shows sources separately.
          const maxUiChars = 600;
          if (ragInsight.answer.length > maxUiChars) {
            ragInsight.answer = ragInsight.answer.slice(0, maxUiChars).trimEnd() + '…';
          }
        }
      }
    }

    // Get NBA data for sports betting queries
    let nbaData: any = null;
    if (domain === 'sports' && (message.toLowerCase().includes('nba') || message.toLowerCase().includes('basketball') ||
      message.toLowerCase().includes('player') || message.toLowerCase().includes('team'))) {
      nbaData = await getNBADataForQuery(message);
    }

    // Get BallDontLie live data for any sport query
    let sportsDataContext: { context: string; data: any } | null = null;
    if (domain === 'sports') {
      try {
        sportsDataContext = await getSportsDataForQuery(message);
      } catch (e) {
        console.warn('BallDontLie fetch failed:', e);
      }
    }

    // DATA MODE: For pure data questions, respond directly from BallDontLie/cache (no code generation).
    if (intent === 'data' && sportsDataContext?.context) {
      const finalContent =
        `${sportsDataContext.context}\n\n` +
        `Source: BallDontLie API (cached).`;

      const chatResponse: ChatResponse = {
        message: {
          id: `msg-${Date.now()}`,
          role: 'assistant',
          content: finalContent,
          timestamp: new Date(),
          ragInsight: undefined,
          visualization: undefined,
        },
        memoryContext,
      };

      // Store conversation memory (so users don't lose progress)
      if (userId) {
        const conversationMemory: ConversationMemory = {
          id: chatResponse.message.id,
          userId,
          sessionId: `session-${Date.now()}`,
          timestamp: new Date(),
          role: 'assistant',
          content: finalContent,
        };
        await memoryService.addConversationMemory(conversationMemory);
      }

      return chatResponse;
    }

    // Build enhanced prompt with memory, RAG, and NBA context
    const basePrompt = intent === 'simulation' ? SYSTEM_PROMPT : GURU_QA_PROMPT;
    const promptParts = [
      aiGuruSettings.systemPromptPrefix?.trim(),
      basePrompt.trim(),
      aiGuruSettings.systemPromptSuffix?.trim(),
    ].filter(Boolean);

    let enhancedPrompt = `${promptParts.join('\n\n')}\n\nDomain: ${domain}\nUser Request: ${message}`;

    // Add memory context
    if (memoryContext) {
      enhancedPrompt += `\n\nUser Context:`;
      enhancedPrompt += `\n- Experience Level: ${memoryContext.personalization.experienceLevel}`;
      enhancedPrompt += `\n- Risk Tolerance: ${memoryContext.userProfile.riskTolerance}`;
      enhancedPrompt += `\n- Preferred Markets: ${memoryContext.userProfile.preferredMarkets?.join(', ') || 'none specified'}`;

      if (memoryContext.relevantStrategies.length > 0) {
        enhancedPrompt += `\n- Previous Successful Strategies: ${memoryContext.relevantStrategies.map(s => s.name).join(', ')}`;
      }

      if (memoryContext.recentConversations.length > 0) {
        enhancedPrompt += `\n- Recent Conversation Topics: ${memoryContext.recentConversations.slice(-2).map(c => c.content.substring(0, 50)).join('; ')}`;
      }
    }

    // Add RAG context with enriched data support (using AI Guru settings)
    // IMPORTANT: Keep this brief and explicitly instruct the model NOT to quote verbatim.
    if (ragInsight) {
      enhancedPrompt += `\n\nRAG REFERENCE (summarize in your own words; do NOT quote verbatim):`;

      // Add themes if available and enabled (from enriched chunks)
      if (aiGuruSettings.includeThemesInResponse && ragInsight.themes && ragInsight.themes.length > 0) {
        enhancedPrompt += `\nTopics: ${ragInsight.themes.slice(0, 5).join(', ')}`;
      }

      // Add main context (keep short to avoid "dumping")
      const maxRagChars = Math.min(aiGuruSettings.maxContextLength, 800);
      enhancedPrompt += `\n\n${ragInsight.answer.slice(0, maxRagChars)}`;

      // Add key points if available and enabled (from enriched chunks)
      if (aiGuruSettings.includeKeyPointsInResponse && ragInsight.keyPoints && ragInsight.keyPoints.length > 0) {
        enhancedPrompt += `\n\nKey points:`;
        ragInsight.keyPoints.slice(0, 5).forEach((kp, i) => {
          enhancedPrompt += `\n- ${kp}`;
        });
      }

      // Add sources with enrichment indicator (if enabled)
      if (aiGuruSettings.includeSourcesInResponse) {
        const sourceInfo = ragInsight.sources.slice(0, 3).map(s =>
          `${s.source}${s.enriched ? ' ✓' : ''}`
        ).join(', ');
        enhancedPrompt += `\n\nSources: ${sourceInfo}`;
      }
    }

    // Add NBA data context for sports queries
    if (nbaData) {
      enhancedPrompt += `\n\nNBA Data Context:\n${nbaData.context}\n\nKey Statistics:\n${nbaData.stats}`;
    }

    // Add BallDontLie live data context
    if (sportsDataContext) {
      enhancedPrompt += `\n\nLive Sports Data (from BallDontLie):\n${sportsDataContext.context}`;
    }

    // Add visualization capabilities for backtest analysis
    if (message.toLowerCase().includes('chart') || message.toLowerCase().includes('graph') ||
      message.toLowerCase().includes('visual') || message.toLowerCase().includes('plot')) {
      enhancedPrompt += `\n\nVISUALIZATION CAPABILITIES: You can generate charts and graphs for backtest analysis. When analyzing backtest results, include chart generation requests in your response.`;
    }

    // Backtest / parlay intent flags were computed earlier.

    if (isBacktest) {
      enhancedPrompt += `\n\nIMPORTANT: This is a BACKTESTING request. The user wants to run a strategy simulation on real data.`;
      enhancedPrompt += `\n\nWhen responding to backtesting requests:`;
      enhancedPrompt += `\n1. Identify the strategy type and parameters`;
      enhancedPrompt += `\n2. Run the actual backtest using NBA data`;
      enhancedPrompt += `\n3. Provide comprehensive results with performance metrics`;
      enhancedPrompt += `\n4. Generate charts showing profit curves, win rates, and risk metrics`;
      enhancedPrompt += `\n5. Include recommendations for strategy improvement`;

      if (domain === 'sports') {
        enhancedPrompt += `\n\nFor NBA strategies, you have access to:`;
        enhancedPrompt += `\n- Player performance data (points, rebounds, assists)`;
        enhancedPrompt += `\n- Team statistics and win/loss records`;
        enhancedPrompt += `\n- Game schedules and live scores`;
        enhancedPrompt += `\n- Historical game data for backtesting`;
        enhancedPrompt += `\n\nAvailable NBA strategies: player_points_over, team_win_streak, home_away_performance, player_vs_defense, injury_impact, momentum_reversal, spread_betting, moneyline_prediction`;
      }
    }

    if (isParlay || isRoundRobin) {
      enhancedPrompt += `\n\nIMPORTANT: This is a ${isParlay ? 'PARLAY' : 'ROUND ROBIN'} strategy request.`;
      enhancedPrompt += `\n- ${isParlay ? 'Parlays' : 'Round Robins'} require ALL legs to win for payout, significantly increasing risk.`;
      enhancedPrompt += `\n- Calculate combined odds by multiplying individual leg odds.`;
      enhancedPrompt += `\n- Payout = stake × combined_odds (only if all legs win).`;
      enhancedPrompt += `\n- Include risk warnings about the high variance nature of these bets.`;
      if (isRoundRobin) {
        enhancedPrompt += `\n- Generate all combinations of the specified size from the available selections.`;
        enhancedPrompt += `\n- Each combination is treated as a separate mini-parlay.`;
      }
      enhancedPrompt += `\n- Always emphasize that these are high-risk strategies not suitable for risk-averse traders.`;
    }

    if (intent === 'simulation') {
      enhancedPrompt += `\n\nGenerate the strategy function:`;
    } else {
      enhancedPrompt += `\n\nIMPORTANT: Answer the user's question directly. Do not include any code blocks unless explicitly requested.`;
    }

    // Use real Grok4Fast API
    const grokApiKey = process.env.GROK_API_KEY;
    if (!grokApiKey) {
      throw new Error('GROK_API_KEY environment variable is not set');
    }

    const temperatureToUse =
      intent === 'analysis'
        ? Math.min(aiGuruSettings.temperature, 0.3)
        : aiGuruSettings.temperature;
    const maxTokensToUse =
      intent === 'analysis'
        ? Math.min(aiGuruSettings.maxTokens, 900)
        : aiGuruSettings.maxTokens;

    const response = await fetch('https://api.x.ai/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${grokApiKey}`,
      },
      body: JSON.stringify({
        model: 'grok-4-1-fast-reasoning',
        messages: [
          { role: 'system', content: enhancedPrompt },
          {
            role: 'user',
            content:
              intent === 'simulation'
                ? `Generate a sports betting strategy for: ${message}`
                : message,
          }
        ],
        temperature: temperatureToUse,
        max_tokens: maxTokensToUse,
      }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error('Grok API Error Details:', errorText);
      throw new Error(`Grok API call failed: ${response.status} - ${errorText}`);
    }

    const data = await response.json();
    const assistantContent = data.choices?.[0]?.message?.content || 'Error generating response';

    // Extract strategy code from the response (no mock/template fallbacks)
    const extractTs = (content: string) =>
      extractFencedCode(content, ['typescript', 'ts', 'javascript', 'js'], true);
    const extractPy = (content: string) =>
      extractFencedCode(content, ['python', 'py'], false);

    let strategyCode: string | null = null;
    let dynamicPythonCode: string | null = null;
    let finalContent = assistantContent;
    let strategyName: string | undefined;

    if (intent === 'simulation') {
      strategyCode = extractTs(assistantContent);
      dynamicPythonCode = extractPy(assistantContent);

      // If the model gave Python only, still accept it as strategy code for storage.
      if (!strategyCode && dynamicPythonCode) {
        strategyCode = dynamicPythonCode;
      }

      // Retry once with a strict instruction if no code block was returned.
      if (!strategyCode) {
        const retryResponse = await fetch('https://api.x.ai/v1/chat/completions', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${grokApiKey}`,
          },
          body: JSON.stringify({
            model: 'grok-4-1-fast-reasoning',
            messages: [
              { role: 'system', content: enhancedPrompt },
              {
                role: 'user',
                content:
                  `Return ONLY the strategy implementation as a single \`\`\`python\`\`\` code block for: ${message}. ` +
                  `Do not include explanations.`,
              },
            ],
            temperature: 0.1, // Keep low for deterministic code extraction
            max_tokens: aiGuruSettings.maxTokens,
          }),
        });

        if (!retryResponse.ok) {
          throw new Error(`Grok API retry failed: ${retryResponse.status}`);
        }

        const retryData = await retryResponse.json();
        const retryContent = retryData.choices?.[0]?.message?.content || '';

        const retryTs = extractTs(retryContent);
        const retryPy = extractPy(retryContent);

        strategyCode = retryTs || retryPy || null;
        dynamicPythonCode = dynamicPythonCode || retryPy || null;

        if (!strategyCode) {
          throw new Error('AI did not return strategy code. Please try again.');
        }

        // If the first response lacked a code fence, append the extracted code for the user.
        const hasAnyFence = assistantContent.includes('```');
        const hasPyFence = /```(?:python|py)\n/i.test(assistantContent);
        if (!hasAnyFence || !hasPyFence) {
          finalContent =
            assistantContent +
            '\n\n```python\n' +
            strategyCode +
            '\n```';
        }
      }

      const nameMatch =
        assistantContent.match(/Strategy: ([^\n]+)/) ||
        assistantContent.match(/\*\*([^*]+)\*\*/);
      strategyName = nameMatch
        ? nameMatch[1]
        : `${domain.charAt(0).toUpperCase() + domain.slice(1)} Strategy: ${message.slice(0, 30)}...`;
    }

    // Post-process non-simulation answers to avoid code dumping / irrelevant tangents.
    if (intent !== 'simulation') {
      // Strip any code fences unless the user explicitly asked for code.
      if (disallowCode || !wantsCodeExplicitly) {
        finalContent = finalContent.replace(/```[\s\S]*?```/g, '').trim();
      }

      const lower = finalContent.toLowerCase();
      const hasCodeFence = /```/.test(finalContent);
      const mentionsOtherMarkets = /\b(crypto|stocks?|forex|token|btc|ethereum)\b/i.test(finalContent);
      const looksLikeBookDump =
        /\bout of print\b/i.test(finalContent) ||
        /\bnew york:\b/i.test(finalContent) ||
        /\blas vegas:\b/i.test(finalContent) ||
        /\bpublisher\b/i.test(finalContent);
      const tooLong = finalContent.length > 1600;

      const needsRewrite =
        (disallowCode && hasCodeFence) ||
        mentionsOtherMarkets ||
        looksLikeBookDump ||
        tooLong;

      if (needsRewrite) {
        try {
          const rewriteResp = await fetch('https://api.x.ai/v1/chat/completions', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${grokApiKey}`,
            },
            body: JSON.stringify({
              model: 'grok-4-1-fast-reasoning',
              messages: [
                {
                  role: 'system',
                  content:
                    `${GURU_QA_PROMPT}\n\n` +
                    `Rewrite answers to be concise, coherent, and sports-betting-only.\n` +
                    `- No code blocks\n` +
                    `- No citations or book lists\n` +
                    `- Do NOT mention crypto, stocks, forex, or any non-sports markets\n` +
                    `- Max ~180 words unless user asks for more\n` +
                    `- Use short headings or bullets\n`,
                },
                {
                  role: 'user',
                  content:
                    `User question:\n${message}\n\n` +
                    `Rewrite this draft answer:\n${finalContent}`,
                },
              ],
              temperature: 0.2,
              max_tokens: Math.min(maxTokensToUse, 600),
            }),
          });

          if (rewriteResp.ok) {
            const rewriteData = await rewriteResp.json();
            const rewritten =
              (rewriteData?.choices?.[0]?.message?.content || '').trim();
            if (rewritten) {
              finalContent = rewritten.replace(/```[\s\S]*?```/g, '').trim();
            }
          }
        } catch (e) {
          // Best-effort cleanup only; fall back to the original content.
          console.warn('Failed to rewrite non-simulation answer:', e);
        }
      }

      // Hard guardrail: remove any remaining non-sports market references.
      // (Last-resort cleanup for occasional model slip-ups.)
      const bannedMarkets = /\b(crypto|cryptocurrency|stocks?|forex|fx|token|btc|bitcoin|ethereum|eth)\b/i;
      if (bannedMarkets.test(finalContent)) {
        finalContent = finalContent
          .split('\n')
          .filter((line) => !bannedMarkets.test(line))
          .join('\n')
          .replace(/\n{3,}/g, '\n\n')
          .trim();
      }
    }

    // Execute backtesting if this is a backtest request
    let backtestResults = null;
    let backtestVisualization = null;
    let strategyEquation = null;
    if (isBacktest) {
      try {
        // Determine market from message
        let backtestMarket = domain;
        const msgLower = message.toLowerCase();

        // Sports markets
        if (msgLower.includes('nba') || (msgLower.includes('basketball') && !msgLower.includes('college'))) {
          backtestMarket = 'nba';
        } else if (msgLower.includes('nfl') || msgLower.includes('football') && !msgLower.includes('college')) {
          backtestMarket = 'nfl';
        } else if (msgLower.includes('mlb') || msgLower.includes('baseball')) {
          backtestMarket = 'mlb';
        } else if (msgLower.includes('nhl') || msgLower.includes('hockey')) {
          backtestMarket = 'nhl';
        } else if (msgLower.includes('college football') || msgLower.includes('cfb') || msgLower.includes('ncaa football')) {
          backtestMarket = 'cfb';
        } else if (msgLower.includes('college basketball') || msgLower.includes('cbb') || msgLower.includes('ncaa basketball') || msgLower.includes('march madness')) {
          backtestMarket = 'cbb';
        }
        // Default to sports if no specific league detected
        // (This platform focuses on sports betting only)

        // Generate strategy equation before running backtest
        strategyEquation = generateStrategyEquation(message, backtestMarket);

        const wantsParlayBacktest =
          domain === 'sports' && (isParlay || isRoundRobin);

        if (wantsParlayBacktest) {
          const parlayConfig = buildParlayBacktestConfig(message, backtestMarket);
          if (parlayConfig.strategyEquation) {
            strategyEquation = parlayConfig.strategyEquation;
          }

          backtestResults = isRoundRobin
            ? await runRoundRobinBacktest(
              backtestMarket,
              parlayConfig.strategy,
              {
                season: parlayConfig.season ?? undefined,
                startDate: parlayConfig.startDate ?? undefined,
                endDate: parlayConfig.endDate ?? undefined,
                parlaySize: parlayConfig.parlaySize,
                stakePerCombo: parlayConfig.stakePerParlay,
                // Keep chat responses snappy; users can narrow the date range for deep dives.
                limitDays: parlayConfig.limitDays ?? undefined,
              },
            )
            : await runParlayBacktest(
              backtestMarket,
              parlayConfig.strategy,
              {
                season: parlayConfig.season ?? undefined,
                startDate: parlayConfig.startDate ?? undefined,
                endDate: parlayConfig.endDate ?? undefined,
                parlaySize: parlayConfig.parlaySize,
                stakePerParlay: parlayConfig.stakePerParlay,
                // Keep chat responses snappy; users can narrow the date range for deep dives.
                limitDays: parlayConfig.limitDays ?? undefined,
              },
            );
        } else {
        // TODO: Move backtesting to background queue for better UX
        // Current: Synchronous processing with 30s timeout
        // Future: Add to BullMQ queue and poll for results
        backtestResults = await runUniversalBacktest(message, assistantContent, backtestMarket, dynamicPythonCode);
        }

        // If we have a structured backtest response but it couldn't run (e.g. no odds available),
        // surface the reason to the user instead of failing silently.
        if (backtestResults && backtestResults.success === false) {
          finalContent += '\n\n' + '='.repeat(60);
          finalContent += '\n⚠️ **BACKTEST COULD NOT RUN** ⚠️\n';
          finalContent += '='.repeat(60) + '\n\n';

          if (backtestResults.performance_summary) {
            finalContent += backtestResults.performance_summary + '\n\n';
          }

          if (Array.isArray(backtestResults.data_warnings) && backtestResults.data_warnings.length) {
            finalContent += '**Data Warnings:**\n';
            backtestResults.data_warnings.slice(0, 6).forEach((w: string) => {
              finalContent += `• ${w}\n`;
            });
            if (backtestResults.data_warnings.length > 6) {
              finalContent += `• …and ${backtestResults.data_warnings.length - 6} more\n`;
            }
            finalContent += '\n';
          }
        }

        if (backtestResults && backtestResults.success) {
          // Generate visualization for the backtest results
          backtestVisualization = await generateBacktestVisualization(backtestResults);

          // Add strategy equation and backtest results to the response
          finalContent += '\n\n' + '='.repeat(60);
          finalContent += '\n🎯 **STRATEGY EQUATION & BACKTEST RESULTS** 🎯\n';
          finalContent += '='.repeat(60) + '\n\n';

          // Show the strategy equation first
          if (strategyEquation) {
            finalContent += '**📊 Strategy Equation Being Tested:**\n\n';
            finalContent += strategyEquation + '\n\n';
            finalContent += '='.repeat(40) + '\n\n';
          }

          // Add equation summary from backtest results
          if (backtestResults.equation_summary) {
            finalContent += '**⚡ Quick Strategy Summary:**\n';
            finalContent += backtestResults.equation_summary + '\n';
            finalContent += '='.repeat(40) + '\n\n';
          }

          finalContent += backtestResults.performance_summary + '\n\n';

          if (backtestResults.results) {
            finalContent += '**📈 Key Performance Metrics:**\n';
            finalContent += `• Win Rate: ${backtestResults.results.win_rate}%\n`;
            finalContent += `• Total Profit: $${backtestResults.results.total_profit}\n`;
            finalContent += `• Profit Factor: ${backtestResults.results.profit_factor}\n`;
            finalContent += `• Sharpe Ratio: ${backtestResults.results.sharpe_ratio}\n`;
            finalContent += `• Max Drawdown: $${backtestResults.results.max_drawdown}\n`;
            finalContent += `• Total Trades: ${backtestResults.total_trades}\n\n`;
          }

          // Add detailed statistics if available
          if (backtestResults.detailed_statistics) {
            const stats = backtestResults.detailed_statistics;
            finalContent += '**📊 Detailed Statistics (Calculated from Actual Trades):**\n\n';

            finalContent += '**Trade Analysis:**\n';
            finalContent += `• Total Trades: ${stats.trade_analysis.total_trades}\n`;
            finalContent += `• Winning Trades: ${stats.trade_analysis.winning_trades}\n`;
            finalContent += `• Losing Trades: ${stats.trade_analysis.losing_trades}\n`;
            finalContent += `• Win Rate: ${stats.trade_analysis.win_rate_percent}%\n\n`;

            finalContent += '**Profit/Loss Breakdown:**\n';
            finalContent += `• Biggest Win: $${stats.profit_loss_analysis.biggest_win}\n`;
            finalContent += `• Biggest Loss: $${stats.profit_loss_analysis.biggest_loss}\n`;
            finalContent += `• Smallest Win: $${stats.profit_loss_analysis.smallest_win}\n`;
            finalContent += `• Smallest Loss: $${stats.profit_loss_analysis.smallest_loss}\n`;
            finalContent += `• Average Win: $${stats.profit_loss_analysis.average_win}\n`;
            finalContent += `• Average Loss: $${stats.profit_loss_analysis.average_loss}\n\n`;

            finalContent += '**Risk & Performance Ratios:**\n';
            finalContent += `• Profit Factor: ${stats.ratios_and_factors.profit_factor}\n`;
            finalContent += `• Win/Loss Ratio: ${stats.ratios_and_factors.win_loss_ratio}\n`;
            finalContent += `• Recovery Factor: ${stats.ratios_and_factors.recovery_factor}\n`;
            finalContent += `• Expectancy per Trade: $${stats.ratios_and_factors.expectancy_per_trade}\n`;
            finalContent += `• Kelly Fraction: ${stats.ratios_and_factors.kelly_fraction_percent}%\n\n`;

            finalContent += '**Risk Metrics:**\n';
            finalContent += `• Sharpe Ratio: ${stats.risk_metrics.sharpe_ratio}\n`;
            finalContent += `• Sortino Ratio: ${stats.risk_metrics.sortino_ratio}\n`;
            finalContent += `• Max Drawdown: $${stats.risk_metrics.max_drawdown}\n`;
            finalContent += `• Average Drawdown: $${stats.risk_metrics.average_drawdown}\n`;
            finalContent += `• Profit Volatility: $${stats.risk_metrics.profit_volatility}\n\n`;

            finalContent += '**Streak Analysis:**\n';
            finalContent += `• Max Win Streak: ${stats.streak_analysis.max_win_streak} trades\n`;
            finalContent += `• Max Loss Streak: ${stats.streak_analysis.max_loss_streak} trades\n`;
            finalContent += `• Avg Win Streak: ${stats.streak_analysis.avg_win_streak} trades\n`;
            finalContent += `• Avg Loss Streak: ${stats.streak_analysis.avg_loss_streak} trades\n`;
            finalContent += `• Current Streak: ${stats.streak_analysis.current_streak_type || 'N/A'} (${stats.streak_analysis.current_streak_length})\n\n`;
          }

          // Add verification info
          if (backtestResults.verification_data) {
            finalContent += '**🔍 Verification & Data Source:**\n';
            finalContent += `• Data Source: ${backtestResults.verification_data.data_source}\n`;
            finalContent += `• Calculation Method: ${backtestResults.verification_data.calculation_method}\n`;
            finalContent += `• Verification Status: ${backtestResults.verification_data.verification_status}\n`;
            finalContent += `• Analysis Timestamp: ${backtestResults.verification_data.timestamp}\n\n`;
          }

          // Add trade log summary (first 10 and last 10 trades)
          if (backtestResults.all_trades && backtestResults.all_trades.length > 0) {
            finalContent += '**📋 Trade Log Summary (First 5 & Last 5 Trades):**\n\n';

            const allTrades = backtestResults.all_trades;
            const firstTrades = allTrades.slice(0, 5);
            const lastTrades = allTrades.slice(-5);

            finalContent += '**First 5 Trades:**\n';
            firstTrades.forEach((trade: { date: string | undefined; action: string | undefined; outcome: string | undefined; profit: number | undefined; }, idx: number) => {
              const date = trade.date || 'N/A';
              const action = trade.action || 'UNKNOWN';
              const outcome = trade.outcome || 'unknown';
              const profit = trade.profit || 0;
              finalContent += `${idx + 1}. ${date} | ${action} | ${outcome.toUpperCase()} | $${profit.toFixed(2)}\n`;
            });

            finalContent += '\n**Last 5 Trades:**\n';
            lastTrades.forEach((trade, idx) => {
              const date = trade.date || 'N/A';
              const action = trade.action || 'UNKNOWN';
              const outcome = trade.outcome || 'unknown';
              const profit = trade.profit || 0;
              finalContent += `${allTrades.length - 4 + idx}. ${date} | ${action} | ${outcome.toUpperCase()} | $${profit.toFixed(2)}\n`;
            });

            finalContent += '\n**💡 Note:** Full detailed trade log available. All P&L calculations are real and verified from actual trade data.\n\n';

            // Add download report buttons
            finalContent += '**📥 Download Report:**\n';
            finalContent += '• [📊 Download CSV Report](javascript:void(0);) - Excel-compatible spreadsheet\n';
            finalContent += '• [📄 Download Text Report](javascript:void(0);) - Detailed analysis report\n\n';
          }

          // Add parameter optimization results if available
          if (backtestResults.optimization_results) {
            const opt = backtestResults.optimization_results;

            finalContent += '**🎯 PARAMETER OPTIMIZATION RESULTS**\n\n';

            if (opt.statistics) {
              finalContent += '**Optimization Summary:**\n';
              finalContent += `• Total Evaluations: ${opt.total_evaluations}\n`;
              finalContent += `• Target Metric: ${opt.statistics.target_metric.replace('_', ' ').toUpperCase()}\n`;
              finalContent += `• Best Value: ${opt.statistics.best_value}\n`;
              finalContent += `• Average Value: ${opt.statistics.average_value}\n`;
              finalContent += `• Standard Deviation: ${opt.statistics.standard_deviation}\n\n`;
            }

            if (opt.best_result) {
              finalContent += '**🏆 Best Parameter Set Found:**\n';
              const bestParams = opt.best_result.parameters;
              Object.entries(bestParams).forEach(([key, value]) => {
                finalContent += `• ${key.replace('_', ' ').toUpperCase()}: ${value}\n`;
              });

              finalContent += `\n**Performance with Best Parameters:**\n`;
              if (opt.best_result.metrics) {
                const metrics = opt.best_result.metrics;
                finalContent += `• Win Rate: ${metrics.win_rate}%\n`;
                finalContent += `• Total Profit: $${metrics.total_profit}\n`;
                finalContent += `• Sharpe Ratio: ${metrics.sharpe_ratio}\n`;
                finalContent += `• Profit Factor: ${metrics.profit_factor}\n`;
              }
              finalContent += '\n';
            }

            if (opt.top_5_results && opt.top_5_results.length > 0) {
              finalContent += '**🥇 Top 5 Parameter Combinations:**\n\n';

              opt.top_5_results.forEach((result: any, index: number) => {
                finalContent += `**Rank ${result.rank}:**\n`;
                Object.entries(result.parameters).forEach(([key, value]) => {
                  finalContent += `• ${key.replace('_', ' ').toUpperCase()}: ${value}\n`;
                });
                finalContent += `• Target Score: ${result.target_metric}\n`;
                finalContent += `• Win Rate: ${result.key_metrics.win_rate}%\n`;
                finalContent += `• Total Profit: $${result.key_metrics.total_profit}\n\n`;
              });
            }

            finalContent += '**💡 Optimization Tips:**\n';
            finalContent += '• Higher evaluations = better results but slower processing\n';
            finalContent += '• Grid search: Systematic testing of all combinations\n';
            finalContent += '• Random search: Faster, good for high-dimensional spaces\n';
            finalContent += '• Genetic algorithm: Advanced optimization for complex strategies\n\n';
          }

          // Add walk forward analysis if available
          if (backtestResults.walk_forward_analysis) {
            const wf = backtestResults.walk_forward_analysis;

            finalContent += '**🎯 WALK FORWARD ANALYSIS - OVERFITTING PROTECTION**\n\n';

            // Configuration summary
            if (wf.configuration) {
              finalContent += '**Analysis Configuration:**\n';
              finalContent += `• In-sample period: ${wf.configuration.in_sample_months} months (training)\n`;
              finalContent += `• Out-of-sample period: ${wf.configuration.out_of_sample_months} months (testing)\n`;
              finalContent += `• Total walk-forward periods: ${wf.configuration.total_periods}\n`;
              finalContent += `• Date range: ${wf.configuration.start_date} to ${wf.configuration.end_date}\n\n`;
            }

            // Overall statistics
            if (wf.overall_statistics) {
              const overall = wf.overall_statistics;
              finalContent += '**Overall Walk-Forward Performance:**\n';
              finalContent += `• Average Win Rate: ${overall.average_win_rate}%\n`;
              finalContent += `• Average Monthly P&L: $${overall.average_monthly_pnl}\n`;
              finalContent += `• Total P&L: $${overall.total_pnl}\n`;
              finalContent += `• Profitable Periods: ${overall.profitable_periods_percent}%\n`;
              finalContent += `• Best Period: $${overall.best_period_pnl}\n`;
              finalContent += `• Worst Period: $${overall.worst_period_pnl}\n`;
              finalContent += `• P&L Consistency (Std Dev): $${overall.pnl_consistency}\n\n`;
            }

            // Overfitting summary
            if (wf.overfitting_summary) {
              finalContent += '**Overfitting Analysis:**\n';
              finalContent += `${wf.overfitting_summary.replace(/\n/g, '\n')}\n\n`;
            }

            // Recommendations
            if (wf.recommendations && wf.recommendations.length > 0) {
              finalContent += '**🎯 Recommendations:**\n';
              wf.recommendations.forEach((rec: string) => {
                finalContent += `• ${rec}\n`;
              });
              finalContent += '\n';
            }

            // Period-by-period breakdown (first 3 and last 3)
            if (wf.periods && wf.periods.length > 0) {
              finalContent += '**Period-by-Period Results:**\n\n';

              const periods = wf.periods;
              const firstPeriods = periods.slice(0, 3);
              const lastPeriods = periods.slice(-3);

              finalContent += '**First 3 Periods:**\n';
              firstPeriods.forEach((period: any) => {
                const inSample = period.in_sample_period;
                const outSample = period.out_of_sample_period;
                const analysis = period.overfitting_analysis;

                finalContent += `**Period ${period.period_number}:**\n`;
                finalContent += `• In-sample (${inSample.start_date} to ${inSample.end_date}): ${inSample.metrics.total_trades} trades, ${inSample.metrics.win_rate}% win rate, $${inSample.metrics.total_pnl} P&L\n`;
                finalContent += `• Out-of-sample (${outSample.start_date} to ${outSample.end_date}): ${outSample.metrics.total_trades} trades, ${outSample.metrics.win_rate}% win rate, $${outSample.metrics.total_pnl} P&L\n`;
                finalContent += `• Overfitting Risk: ${analysis.overfitting_risk_level} (${analysis.overfitting_score} score)\n\n`;
              });

              if (periods.length > 6) {
                finalContent += `**... ${periods.length - 6} more periods analyzed ...\n\n`;
              }

              if (periods.length > 3) {
                finalContent += '**Last 3 Periods:**\n';
                lastPeriods.forEach((period: any) => {
                  const inSample = period.in_sample_period;
                  const outSample = period.out_of_sample_period;
                  const analysis = period.overfitting_analysis;

                  finalContent += `**Period ${period.period_number}:**\n`;
                  finalContent += `• In-sample (${inSample.start_date} to ${inSample.end_date}): ${inSample.metrics.total_trades} trades, ${inSample.metrics.win_rate}% win rate, $${inSample.metrics.total_pnl} P&L\n`;
                  finalContent += `• Out-of-sample (${outSample.start_date} to ${outSample.end_date}): ${outSample.metrics.total_trades} trades, ${outSample.metrics.win_rate}% win rate, $${outSample.metrics.total_pnl} P&L\n`;
                  finalContent += `• Overfitting Risk: ${analysis.overfitting_risk_level} (${analysis.overfitting_score} score)\n\n`;
                });
              }

              finalContent += '**💡 Walk Forward Benefits:**\n';
              finalContent += '• Prevents curve-fitting by testing on unseen data\n';
              finalContent += '• Validates strategy robustness across different market conditions\n';
              finalContent += '• Provides confidence intervals for expected performance\n';
              finalContent += '• Identifies parameter sensitivity and overfitting risks\n\n';
            }
          }
        }
      } catch (error) {
        console.error('NBA backtesting failed:', error);
        finalContent += '\n\n⚠️ **Note:** NBA backtesting is currently unavailable. The strategy code above can be tested manually.';
      }
    }

    // Add personalized recommendations
    if (memoryContext && userId) {
      const recommendations = await memoryService.getPersonalizedRecommendations(userId);
      if (recommendations.length > 0) {
        finalContent += `\n\n🎯 **Personalized Recommendations:**\n${recommendations.map(r => `• ${r}`).join('\n')}`;
      }
    }

    // Note: RAG insights and source snippets are surfaced via `message.ragInsight` in the UI.
    // Avoid appending raw RAG/data dumps into the main assistant message (keeps replies coherent).

    const chatResponse: ChatResponse = {
      message: {
        id: `msg-${Date.now()}`,
        role: 'assistant',
        content: finalContent,
        timestamp: new Date(),
        ragInsight: ragInsight || undefined,
        visualization: backtestVisualization || undefined,
      },
      strategyCode,
      strategyName,
      ragInsight: ragInsight || undefined,
      memoryContext,
      visualization: backtestVisualization || undefined,
      backtestResults: backtestResults || undefined,
    };

    // Store conversation memory
    if (userId) {
      const conversationMemory: ConversationMemory = {
        id: chatResponse.message.id,
        userId,
        sessionId: `session-${Date.now()}`, // In real app, use actual session ID
        timestamp: new Date(),
        role: 'assistant',
        content: finalContent,
        strategyGenerated: strategyCode ? {
          code: strategyCode,
          name: strategyName,
        } : undefined,
        ragContext: ragInsight ? {
          insights: [ragInsight.answer],
          sources: ragInsight.sources.map(s => s.source),
          themes: ragInsight.themes || [],
          keyPoints: ragInsight.keyPoints || [],
          hasEnrichedData: ragInsight.hasEnrichedData || false,
        } : undefined,
      };

      await memoryService.addConversationMemory(conversationMemory);
    }

    return chatResponse;
  } catch (error) {
    console.error('Strategy generation error:', error);
    return {
      message: {
        id: `msg-${Date.now()}`,
        role: 'assistant',
        content: `I encountered an error while generating your strategy. Please try again or contact support. Error: ${error instanceof Error ? error.message : 'Unknown error'}`,
        timestamp: new Date(),
      },
    };
  }
}

function escapeRegex(value: string): string {
  return value.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

function extractFencedCode(
  content: string,
  languages: string[],
  allowLanguageOmitted: boolean
): string | null {
  const langPart = languages.map(escapeRegex).join('|');
  const re = allowLanguageOmitted
    ? new RegExp('```(' + langPart + ')?\\n([\\s\\S]*?)\\n```', 'i')
    : new RegExp('```(' + langPart + ')\\n([\\s\\S]*?)\\n```', 'i');

  const match = content.match(re);
  if (!match) return null;

  const label = match[1] || '';
  const code = (match[2] || '').trim();
  if (!code) return null;

  if (allowLanguageOmitted && !label) {
    // Avoid treating unlabeled Python as TS/JS.
    const looksLikePython =
      /(^|\n)\s*def\s+\w+\s*\(/.test(code) ||
      /(^|\n)\s*from\s+\w+/.test(code) ||
      /(^|\n)\s*import\s+\w+/.test(code);
    if (looksLikePython) return null;
  }

  return code;
}

// NBA Data Integration for Sports Betting Queries
async function getNBADataForQuery(message: string): Promise<{ context: string; stats: string } | null> {
  try {
    const messageLower = message.toLowerCase();

    // Determine what NBA data to fetch based on the query
    let endpoint = 'players';
    let params: any = { use_cache: true, limit: 10 };

    if (messageLower.includes('team') || messageLower.includes('teams')) {
      endpoint = 'teams';
    } else if (messageLower.includes('game') || messageLower.includes('games') || messageLower.includes('score')) {
      endpoint = 'live_games';
    } else if (messageLower.includes('player') || messageLower.includes('stats') || messageLower.includes('performance')) {
      endpoint = 'players';
      params.limit = 20;
    }

    // Extract specific player or team names from the message
    const nbaData = await callNBAService(endpoint, params);

    if (!nbaData) return null;

    let context = '';
    let stats = '';

    if (endpoint === 'players') {
      context = `NBA Players Data: Found ${nbaData.length} active players including stars like Nikola Jokić, LeBron James, Stephen Curry.`;
      stats = nbaData.slice(0, 5).map((p: any) => `${p.full_name} (${p.is_active ? 'Active' : 'Inactive'})`).join(', ');
    } else if (endpoint === 'teams') {
      context = `NBA Teams Data: All 30 NBA teams available including Los Angeles Lakers, Golden State Warriors, Boston Celtics.`;
      stats = nbaData.slice(0, 10).map((t: any) => `${t.full_name} (${t.abbreviation})`).join(', ');
    } else if (endpoint === 'live_games') {
      context = `Live NBA Games: ${nbaData.games?.length || 0} games currently in progress with real-time scores.`;
      stats = nbaData.games?.slice(0, 3).map((g: any) =>
        `${g.home_team} ${g.home_score} vs ${g.away_team} ${g.away_score} (${g.status})`
      ).join('; ') || 'No live games at the moment.';
    }

    return { context, stats };
  } catch (error) {
    console.error('Error fetching NBA data for chat:', error);
    return null;
  }
}

async function callNBAService(endpoint: string, params: any = {}): Promise<any> {
  try {
    const baseUrl = process.env.PYTHON_SERVICE_URL || 'http://localhost:8000';

    // Construct query parameters
    const queryParams = new URLSearchParams();
    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        queryParams.append(key, String(value));
      }
    });

    const url = `${baseUrl}${endpoint}${queryParams.toString() ? '?' + queryParams.toString() : ''}`;

    console.log(`🔗 Calling Python microservice: ${url}`);

    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
      // Add timeout to prevent hanging
      signal: AbortSignal.timeout(10000), // 10 second timeout
    });

    if (!response.ok) {
      throw new Error(`Python microservice returned ${response.status}: ${response.statusText}`);
    }

    const result = await response.json();
    return result;
  } catch (error) {
    console.error('NBA microservice call failed:', error);

    // SECURITY FIX: Return proper error instead of misleading mock data
    throw new Error(
      'NBA data service is currently unavailable. ' +
      'Please try again later or contact support if the issue persists.'
    );
  }
}

// BallDontLie Integration for Live Sports Data
async function getSportsDataForQuery(message: string): Promise<{ context: string; data: any } | null> {
  try {
    const messageLower = message.toLowerCase();
    const bdl = BallDontLieService.getInstance();

    // Detect which sport the user is asking about
    const isGamesRequest =
      /\b(games?|schedule|matchups?|fixtures?)\b/.test(messageLower);
    const isStandingsRequest =
      /\b(standings|table|rankings?)\b/.test(messageLower);

    const yearMatch = messageLower.match(/\b(20\d{2})\b/);
    const season = yearMatch ? Number(yearMatch[1]) : undefined;

    const sportPatterns: { pattern: RegExp; league: string; label: string }[] = [
      { pattern: /\b(nfl|football)\b/, league: 'nfl', label: 'NFL' },
      { pattern: /\b(nba|basketball)\b/, league: 'nba', label: 'NBA' },
      { pattern: /\b(mlb|baseball)\b/, league: 'mlb', label: 'MLB' },
      { pattern: /\b(nhl|hockey)\b/, league: 'nhl', label: 'NHL' },
      { pattern: /\b(wnba)\b/, league: 'wnba', label: 'WNBA' },
      { pattern: /\b(college football|ncaaf|cfb)\b/, league: 'ncaaf', label: 'NCAAF' },
      { pattern: /\b(college basketball|ncaab|cbb)\b/, league: 'ncaab', label: 'NCAAB' },

      // Soccer (league-specific)
      { pattern: /\b(epl|premier league)\b/, league: 'epl', label: 'EPL' },
      { pattern: /\b(la liga)\b/, league: 'la_liga', label: 'La Liga' },
      { pattern: /\b(serie a)\b/, league: 'serie_a', label: 'Serie A' },
      { pattern: /\b(bundesliga)\b/, league: 'bundesliga', label: 'Bundesliga' },
      { pattern: /\b(ligue 1)\b/, league: 'ligue_1', label: 'Ligue 1' },
      { pattern: /\b(champions league|ucl)\b/, league: 'champions_league', label: 'Champions League' },
    ];

    for (const { pattern, league, label } of sportPatterns) {
      if (pattern.test(messageLower)) {
        if (isStandingsRequest) {
          try {
            const data = await bdl.getStandings(league, season ? { season } : {});
            const context = bdl.formatStandingsForChat(data, label);
            return { context, data };
          } catch (e) {
            console.warn('BallDontLie standings fetch failed:', e);
            // Fall through to teams/games for this league
          }
        }

        if (isGamesRequest) {
          const data = await bdl.getGames(league, season ? { season } : {});
          const context = bdl.formatGamesForChat(data, label);
          return { context, data };
        }

        const data = await bdl.getTeams(league);
        const context = bdl.formatTeamsForChat(data, label);
        return { context, data };
      }
    }

    return null;
  } catch (error) {
    console.error('BallDontLie query error:', error);
    return null;
  }
}

// Fallback local backtesting when microservice is unavailable
async function runLocalBacktest(
  market: string,
  strategyName: string,
  parameters: any,
  timeframe: string,
  useIntrabarTicks: boolean,
  dynamicCode: string | null,
  walkForwardConfig: any,
  optimizationConfig: any
): Promise<any> {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(process.cwd(), 'universal_backtesting.py');

    const backtestParams: any = {
      market: market,
      strategy_name: strategyName,
      parameters: parameters,
      min_trades: 20,
      timeframe: timeframe,
      use_intrabar_ticks: useIntrabarTicks,
      strategy_code: dynamicCode
    };

    if (walkForwardConfig) {
      backtestParams.walk_forward_periods = walkForwardConfig;
    }

    if (optimizationConfig) {
      backtestParams.optimization_config = optimizationConfig;
    }

    const args = [
      scriptPath,
      'run_backtest',
      JSON.stringify(backtestParams)
    ];

    const pythonProcess = spawn('python3', args, {
      cwd: process.cwd(),
      env: {
        ...process.env,
        PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages')
      }
    });

    let stdout = '';
    let stderr = '';

    pythonProcess.stdout.on('data', (data) => {
      stdout += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      stderr += data.toString();
    });

    pythonProcess.on('close', (code) => {
      if (code !== 0) {
        console.error(`Local backtesting failed with code ${code}:`, stderr);
        reject(new Error(`Backtesting failed: ${stderr}`));
        return;
      }

      try {
        const result = JSON.parse(stdout.trim());
        console.log(`✅ Local backtesting completed successfully`);
        resolve(result);
      } catch (parseError) {
        console.error('Failed to parse local backtesting response:', stdout);
        reject(new Error('Invalid local backtesting response format'));
      }
    });

    pythonProcess.on('error', (error) => {
      console.error('Failed to start local backtesting process:', error);
      reject(new Error(`Failed to start backtesting: ${error.message}`));
    });
  });
}

// Time period extraction function
function extractTimePeriod(message: string, market: string): any {
  // For NBA/Sports
  if (market === 'nba' || market === 'sports') {
    // Extract season (e.g., "2023-24 season", "2024 NBA season")
    const seasonMatch = message.match(/(\d{4})(?:-(\d{2}))?\s*(?:NBA\s*)?season/i);
    if (seasonMatch) {
      const startYear = parseInt(seasonMatch[1]);
      const endYear = seasonMatch[2] ? parseInt(`20${seasonMatch[2]}`) : startYear + 1;
      return {
        type: 'season',
        start_year: startYear,
        end_year: endYear,
        description: `${startYear}-${endYear} season`
      };
    }

    // Extract year range (e.g., "2023-2024")
    const yearRangeMatch = message.match(/(\d{4})-(\d{4})/);
    if (yearRangeMatch) {
      return {
        type: 'year_range',
        start_year: parseInt(yearRangeMatch[1]),
        end_year: parseInt(yearRangeMatch[2]),
        description: `${yearRangeMatch[1]}-${yearRangeMatch[2]}`
      };
    }

    // Extract specific year (e.g., "2024")
    const yearMatch = message.match(/\b(202\d)\b/);
    if (yearMatch) {
      const year = parseInt(yearMatch[1]);
      return {
        type: 'year',
        year: year,
        description: `${year} season`
      };
    }
  }

  // For Financial Markets
  else {
    // Extract month and year (e.g., "January 2024", "jan 2024")
    const monthYearMatch = message.match(/(\w+)\s+(\d{4})/i);
    if (monthYearMatch) {
      const month = monthYearMatch[1];
      const year = parseInt(monthYearMatch[2]);
      return {
        type: 'month_year',
        month: month,
        year: year,
        description: `${month} ${year}`
      };
    }

    // Extract year range (e.g., "2023-2024")
    const yearRangeMatch = message.match(/(\d{4})-(\d{4})/);
    if (yearRangeMatch) {
      return {
        type: 'year_range',
        start_year: parseInt(yearRangeMatch[1]),
        end_year: parseInt(yearRangeMatch[2]),
        description: `${yearRangeMatch[1]}-${yearRangeMatch[2]}`
      };
    }

    // Extract relative time (e.g., "last 6 months", "past year")
    const relativeMatch = message.match(/(?:last|past|previous)\s+(\d+|\w+)\s+(month|year|week|day)s?/i);
    if (relativeMatch) {
      const amount = relativeMatch[1];
      const unit = relativeMatch[2];
      return {
        type: 'relative',
        amount: amount === 'a' || amount === 'one' ? 1 : parseInt(amount),
        unit: unit,
        description: `Last ${amount} ${unit}${amount !== '1' ? 's' : ''}`
      };
    }

    // Extract specific year
    const yearMatch = message.match(/\b(202\d)\b/);
    if (yearMatch) {
      const year = parseInt(yearMatch[1]);
      return {
        type: 'year',
        year: year,
        description: `${year}`
      };
    }
  }

  return null;
}

type ParlayLegType = 'moneyline' | 'spread' | 'total';
type ParlayPick = 'home' | 'away' | 'over' | 'under';

function extractParlaySize(messageLower: string): number {
  const legMatch = messageLower.match(/(\d+)\s*[- ]?leg/);
  const byMatch = messageLower.match(/by\s+(\d+)'?s?/);
  const n = legMatch
    ? Number(legMatch[1])
    : byMatch
      ? Number(byMatch[1])
      : 2;
  return Math.max(2, Math.min(10, Number.isFinite(n) ? n : 2));
}

function extractStakePerParlay(messageLower: string): number {
  const stakeMatch = messageLower.match(/\$(\d+(\.\d+)?)/);
  const stake = stakeMatch ? Number(stakeMatch[1]) : 10;
  return Math.max(1, Number.isFinite(stake) ? stake : 10);
}

function extractSeasonYear(messageLower: string): number | null {
  // "2023-24" -> grab 2023; "2019 season" -> 2019
  const yearMatch = messageLower.match(/\b(20\d{2})\b/);
  if (!yearMatch) return null;
  const year = Number(yearMatch[1]);
  return Number.isFinite(year) ? year : null;
}

function extractDateRange(messageLower: string): { startDate?: string; endDate?: string } {
  // Supports "YYYY-MM-DD to YYYY-MM-DD" or "YYYY-MM-DD - YYYY-MM-DD"
  const m = messageLower.match(
    /(\d{4}-\d{2}-\d{2})\s*(?:to|through|until|-)\s*(\d{4}-\d{2}-\d{2})/i,
  );
  if (!m) return {};
  return { startDate: m[1], endDate: m[2] };
}

function buildParlayLegStrategy(messageLower: string): {
  type: ParlayLegType;
  pick: (game: any) => ParlayPick;
  condition: (game: any) => boolean;
  description: string;
} {
  const wantsOver = /\bover\b/.test(messageLower);
  const wantsUnder = /\bunder\b/.test(messageLower);
  const wantsTotal = /\b(total|over\/under|ou)\b/.test(messageLower) || wantsOver || wantsUnder;

  const wantsSpread = /\bspread\b/.test(messageLower) || /\bcover\b/.test(messageLower) || /\bline\b/.test(messageLower);

  // Totals
  if (wantsTotal) {
    const pick: ParlayPick = wantsUnder ? 'under' : 'over';
    return {
      type: 'total',
      pick: () => pick,
      condition: (game: any) => game?.odds?.totalLine != null && game?.result?.totalResult != null,
      description: `Totals: ${pick.toUpperCase()}`,
    };
  }

  // Spread
  if (wantsSpread) {
    const side: ParlayPick =
      /\baway\b/.test(messageLower) || /\broad\b/.test(messageLower) ? 'away' : 'home';
    return {
      type: 'spread',
      pick: () => side,
      condition: (game: any) =>
        (side === 'home' ? game?.odds?.spreadHome : game?.odds?.spreadAway) != null &&
        game?.result?.spreadCovered != null,
      description: `Spread: ${side.toUpperCase()} side`,
    };
  }

  // Moneyline (default)
  const wantsHome = /\bhome\b/.test(messageLower);
  const wantsAway = /\baway\b/.test(messageLower) || /\broad\b/.test(messageLower);
  const wantsUnderdog = /\bunderdog\b/.test(messageLower);
  const wantsFavorite = /\bfavorite\b/.test(messageLower) || /\bfavourite\b/.test(messageLower) || /\bfav\b/.test(messageLower);

  if (wantsUnderdog) {
    return {
      type: 'moneyline',
      pick: (game: any) => {
        const h = game?.odds?.moneylineHome;
        const a = game?.odds?.moneylineAway;
        if (h == null || a == null) return 'home';
        // Underdog tends to have the larger (more positive) American price
        return h > a ? 'home' : 'away';
      },
      condition: (game: any) => {
        const h = game?.odds?.moneylineHome;
        const a = game?.odds?.moneylineAway;
        return h != null && a != null && (h > 0 || a > 0);
      },
      description: 'Moneyline: UNDERDOG',
    };
  }

  if (wantsHome && wantsFavorite) {
    return {
      type: 'moneyline',
      pick: () => 'home',
      condition: (game: any) => (game?.odds?.moneylineHome ?? 0) < 0,
      description: 'Moneyline: HOME favorites',
    };
  }

  if (wantsAway && wantsFavorite) {
    return {
      type: 'moneyline',
      pick: () => 'away',
      condition: (game: any) => (game?.odds?.moneylineAway ?? 0) < 0,
      description: 'Moneyline: AWAY favorites',
    };
  }

  if (wantsFavorite) {
    return {
      type: 'moneyline',
      pick: (game: any) => {
        const h = game?.odds?.moneylineHome;
        const a = game?.odds?.moneylineAway;
        if (h == null || a == null) return 'home';

        // If one side is negative, that's the favorite.
        if (h < 0 && a > 0) return 'home';
        if (a < 0 && h > 0) return 'away';

        // If both negative, more negative is bigger favorite.
        if (h < 0 && a < 0) return h < a ? 'home' : 'away';

        // If both positive (rare), smaller positive is the favorite.
        return h < a ? 'home' : 'away';
      },
      condition: (game: any) => game?.odds?.moneylineHome != null && game?.odds?.moneylineAway != null,
      description: 'Moneyline: FAVORITES',
    };
  }

  // Default: home moneyline favorites (most common + stable)
  return {
    type: 'moneyline',
    pick: () => 'home',
    condition: (game: any) => (game?.odds?.moneylineHome ?? 0) < 0,
    description: 'Moneyline: HOME favorites (default)',
  };
}

function buildParlayBacktestConfig(message: string, sport: string): {
  season: number | null;
  startDate?: string;
  endDate?: string;
  parlaySize: number;
  stakePerParlay: number;
  limitDays?: number;
  strategyEquation: string;
  strategy: {
    type: ParlayLegType;
    condition: (game: any) => boolean;
    pick: (game: any) => ParlayPick;
  };
} {
  const msgLower = message.toLowerCase();
  const parlaySize = extractParlaySize(msgLower);
  const stakePerParlay = extractStakePerParlay(msgLower);
  const season = extractSeasonYear(msgLower);
  const { startDate, endDate } = extractDateRange(msgLower);
  const legStrategy = buildParlayLegStrategy(msgLower);
  const isRoundRobinRequest =
    msgLower.includes('round robin') ||
    msgLower.includes('roundrobin') ||
    (msgLower.includes('by ') && msgLower.includes("'s"));

  const periodLabel = startDate && endDate
    ? `${startDate} → ${endDate}`
    : season
      ? `${season} season`
      : 'unspecified time period';

  const equation = [
    `**${isRoundRobinRequest ? 'Round Robin' : 'Parlay'} Backtest Setup**`,
    ``,
    `• Sport: ${sport.toUpperCase()}`,
    `• Time period: ${periodLabel}`,
    `• ${isRoundRobinRequest ? 'Combo' : 'Parlay'} size: ${parlaySize} legs`,
    `• Stake per ${isRoundRobinRequest ? 'combo' : 'parlay'}: $${stakePerParlay}`,
    `• Leg rule: ${legStrategy.description}`,
    isRoundRobinRequest
      ? `• Construction: per day, select a small pool of qualifying legs and generate combinations “by ${parlaySize}'s” (capped for speed)`
      : `• Construction: 1 parlay per day, selecting the top ${parlaySize} qualifying legs (highest implied probability)`,
    ``,
    `**Note:** Odds coverage is currently historical (roughly 2011–2021 for NBA/NFL/NHL). If you request newer seasons, the system will report “no odds data yet.”`,
  ].join('\n');

  return {
    season,
    startDate,
    endDate,
    parlaySize,
    stakePerParlay,
    // Keep requests fast by default; user can ask for a date range to go deeper.
    limitDays: undefined,
    strategyEquation: equation,
    strategy: {
      type: legStrategy.type,
      condition: legStrategy.condition,
      pick: legStrategy.pick,
    },
  };
}

// Universal Backtesting Functions
async function runUniversalBacktest(message: string, aiResponse: string, market: string, dynamicCode?: string | null): Promise<any> {
  // Parse the message to determine strategy type and parameters
  const messageLower = message.toLowerCase();

  let strategyName = 'trend_following'; // Default
  let parameters: any = { stake: 1000 };
  let timeframe = '1hour'; // Default timeframe

  // Sports markets use SportsBacktestingEngine
  const sportsMarkets = ['nba', 'nfl', 'mlb', 'nhl', 'cfb', 'cbb'];

  if (sportsMarkets.includes(market)) {
    // Default to home_favorite for all sports
    strategyName = 'home_favorite';

    // Store strategy description for dynamic matching
    parameters.strategy_description = message;

    // Extract season from message
    const seasonMatch = messageLower.match(/20(\d{2})/);
    if (seasonMatch) {
      parameters.season = `20${seasonMatch[1]}`;
    } else {
      parameters.season = '2024'; // Default to current season
    }

    // Detect specific strategy patterns
    if (messageLower.includes('losing streak') || messageLower.includes('lost') && messageLower.includes('row')) {
      strategyName = 'home_team_losing_streak';
      const streakMatch = messageLower.match(/(\d+)\s*game/);
      if (streakMatch) parameters.losing_streak = parseInt(streakMatch[1]);
    } else if (messageLower.includes('underdog') || messageLower.includes('road')) {
      strategyName = 'road_underdog';
    } else if (messageLower.includes('back to back') || messageLower.includes('b2b')) {
      strategyName = 'back_to_back';
    } else if (messageLower.includes('momentum') || messageLower.includes('win streak') || messageLower.includes('hot')) {
      strategyName = 'momentum_reversal';
      const streakMatch = messageLower.match(/(\d+)\s*game/);
      if (streakMatch) parameters.win_streak = parseInt(streakMatch[1]);
    } else if (messageLower.includes('over') || messageLower.includes('under') || messageLower.includes('total')) {
      strategyName = 'over_under_trend';
    } else if (messageLower.includes('divisional') || messageLower.includes('conference')) {
      strategyName = market === 'nfl' ? 'divisional_game' : 'home_favorite';
    } else if (messageLower.includes('primetime') || messageLower.includes('prime time') || messageLower.includes('night')) {
      strategyName = 'primetime';
    } else if (messageLower.includes('spread') || messageLower.includes('cover')) {
      strategyName = market === 'cfb' ? 'spread_betting' : 'home_favorite';
    } else if (messageLower.includes('run line')) {
      strategyName = 'run_line';
    } else if (messageLower.includes('puck line')) {
      strategyName = 'puck_line';
    } else if (messageLower.includes('march madness') || messageLower.includes('tournament')) {
      strategyName = 'march_madness';
    }

    console.log(`🏀 Sports strategy detected: ${strategyName} for ${market.toUpperCase()} season ${parameters.season}`);
  } else {
    // Default to home_favorite strategy for unrecognized sports markets
    strategyName = 'home_favorite';
    console.log(`🏀 Using default sports strategy: ${strategyName} for ${market}`);
  }

  // This platform focuses on sports betting only
  // For any non-sports market request, default to NBA
  if (!sportsMarkets.includes(market)) {
    console.log(`⚠️ Non-sports market "${market}" requested - defaulting to NBA`);
    // Default market to NBA for sports-only platform
    // market = 'nba'; // Uncomment to force redirect
  }

  // Extract stake amount if mentioned
  const stakeMatch = messageLower.match(/\$(\d+)/);
  if (stakeMatch) parameters.stake = parseInt(stakeMatch[1]);

  // Extract timeframe if mentioned
  const timeframeMatch = messageLower.match(/(\d+)\s*(min|hour|day|week|month)/i);
  if (timeframeMatch) {
    const value = timeframeMatch[1];
    const unit = timeframeMatch[2].toLowerCase();
    if (unit === 'min') {
      timeframe = value === '1' ? '1min' : `${value}min`;
    } else if (unit === 'hour') {
      timeframe = value === '1' ? '1hour' : `${value}hour`;
    } else if (unit === 'day') {
      timeframe = value === '1' ? '1day' : `${value}day`;
    } else if (unit === 'week') {
      timeframe = value === '1' ? '1w' : `${value}w`;
    } else if (unit === 'month') {
      timeframe = value === '1' ? '1m' : `${value}m`;
    }
  }

  // Extract and validate time period for backtesting
  const timePeriod = extractTimePeriod(messageLower, market);
  if (timePeriod) {
    parameters.time_period = timePeriod;
    console.log(`📅 Extracted time period: ${JSON.stringify(timePeriod)}`);
  }

  // Detect walk forward analysis parameters
  let walkForwardConfig = null;
  const walkForwardMatch = messageLower.match(/walk.?forward|walkforward|forward.?test/i);
  if (walkForwardMatch) {
    // Extract walk forward parameters from message
    const inSampleMatch = messageLower.match(/(\d+)\s*month(?:s)?\s*in.?sample|in.?sample.*?(\d+)\s*month/i);
    const outSampleMatch = messageLower.match(/(\d+)\s*month(?:s)?\s*out.?of.?sample|out.*?sample.*?(\d+)\s*month/i);

    const inSampleMonths = inSampleMatch ? parseInt(inSampleMatch[1] || inSampleMatch[2]) : 12;
    const outSampleMonths = outSampleMatch ? parseInt(outSampleMatch[1] || outSampleMatch[2]) : 3;

    walkForwardConfig = {
      in_sample_months: inSampleMonths,
      out_of_sample_months: outSampleMonths,
      step_months: outSampleMonths, // Default to out-of-sample period
    };

    console.log(`🎯 Walk Forward Analysis Detected: ${inSampleMonths} months in-sample, ${outSampleMonths} months out-of-sample`);
  }

  // Detect parameter optimization requests
  let optimizationConfig = null;
  const optimizeMatch = messageLower.match(/optimize|optimise|parameter.*optimization|optimization.*parameter/i);
  if (optimizeMatch) {
    // Extract optimization parameters from message
    const methodMatch = messageLower.match(/method[:\s]+(grid|random|genetic)/i);
    const evaluationsMatch = messageLower.match(/(\d+)\s*evaluation|evaluations[:\s]+(\d+)/i);
    const metricMatch = messageLower.match(/metric[:\s]+(sharpe|profit|win.*rate|drawdown|expectancy)/i);

    const method = methodMatch ? methodMatch[1].toLowerCase() : 'grid_search';
    const maxEvaluations = evaluationsMatch ? parseInt(evaluationsMatch[1] || evaluationsMatch[2]) : 50;
    const targetMetric = metricMatch ? metricMatch[1].toLowerCase().replace(' ', '_') : 'sharpe_ratio';

    optimizationConfig = {
      method: method,
      max_evaluations: maxEvaluations,
      target_metric: targetMetric
    };

    console.log(`🎯 Parameter Optimization Detected: ${method} method, ${maxEvaluations} evaluations, target: ${targetMetric}`);
  }

  // Detect intrabar tick data requests
  const useIntrabarTicks = messageLower.match(/intrabar|tick.*data|within.*bar|bar.*data/i) !== null;
  if (useIntrabarTicks) {
    console.log(`📊 Intrabar tick data requested for detailed price analysis`);
  }

  console.log(`🎯 Running ${market} backtest: ${strategyName} on ${timeframe} timeframe with params:`, parameters);

  // If dynamic code is provided, use the dynamic strategy engine
  if (dynamicCode) {
    strategyName = 'dynamic_custom';
    console.log(`🚀 Using Dynamic Strategy Engine with generated Python code`);
  }

  // Call the Python microservice for backtesting
  try {
    const baseUrl = process.env.PYTHON_SERVICE_URL || 'http://localhost:8000';

    const backtestParams = {
      strategy_name: strategyName,
      market: market,
      params: {
        ...parameters,
        min_trades: 20,
        // Pass symbol for precise data file selection
        ...(parameters.symbol && { data_file: parameters.symbol }),
        timeframe: timeframe,
        use_intrabar_ticks: useIntrabarTicks,
      },
      time_period: parameters.time_period,
      strategy_code: dynamicCode,
      walk_forward_config: walkForwardConfig,
      optimization_config: optimizationConfig
    };

    console.log(`🚀 Calling Python microservice for ${market} backtesting: ${strategyName}`);

    const apiKey = process.env.FLASK_API_KEY;
    if (!apiKey) {
      throw new Error('FLASK_API_KEY environment variable is not set');
    }

    // NOTE: /backtest/run is a Python FastAPI endpoint, NOT a Next.js route
    // This is correct - the Python service at localhost:8000 uses /backtest/* paths
    const response = await fetch(`${baseUrl}/backtest/run`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey,
      },
      body: JSON.stringify(backtestParams),
      signal: AbortSignal.timeout(30000), // 30 second timeout for backtesting
    });

    if (!response.ok) {
      throw new Error(`Backtesting service returned ${response.status}: ${response.statusText}`);
    }

    const result = await response.json();

    if (!result.success) {
      throw new Error(result.error || 'Backtesting failed');
    }

    console.log(`✅ Backtesting completed successfully via microservice`);

    // Prefer returning the full payload (it includes trades + chart data).
    // Some legacy services wrap the full payload under `results` only; detect that case.
    let backtestResult: any = result;
    const looksWrapped =
      backtestResult &&
      typeof backtestResult === 'object' &&
      backtestResult.results &&
      !backtestResult.all_trades &&
      !backtestResult.trades &&
      !backtestResult.chart_visualization &&
      !backtestResult.total_trades &&
      !backtestResult.market &&
      !backtestResult.sport;

    if (looksWrapped) {
      backtestResult = backtestResult.results;
    }

    // Normalize sports trades for UI compatibility (action/return_pct/id)
    if (sportsMarkets.includes(market) && backtestResult) {
      const rawTrades = (backtestResult.all_trades || backtestResult.trades || []) as any[];
      if (Array.isArray(rawTrades) && rawTrades.length) {
        const normalizedTrades = rawTrades.map((t, i) => {
          const stake =
            typeof t?.stake === 'number'
              ? t.stake
              : typeof t?.stake === 'string'
                ? Number.parseFloat(t.stake)
                : undefined;

          const profit =
            typeof t?.profit === 'number'
              ? t.profit
              : typeof t?.profit === 'string'
                ? Number.parseFloat(t.profit)
                : undefined;

          const returnPct =
            typeof t?.return_pct === 'number'
              ? t.return_pct
              : stake && typeof profit === 'number'
                ? Number(((profit / stake) * 100).toFixed(2))
                : 0;

          return {
            ...t,
            id: t?.id || `${market}-${i}`,
            action: t?.action || 'BET',
            return_pct: returnPct,
          };
        });

        backtestResult.all_trades = normalizedTrades;
        backtestResult.trades = normalizedTrades.slice(-50);
      }

      // Add a `market` alias for consistency across UI components
      backtestResult.market = backtestResult.market || market;
    }

    // RAG FEEDBACK LOOP: Add successful strategies to vector store for global learning
    try {
      await addSuccessfulStrategyToRAG(backtestResult, market, message, strategyName, parameters);
    } catch (ragError) {
      console.warn('⚠️ RAG feedback loop failed:', ragError);
      // Don't fail the backtest if RAG update fails
    }

    return backtestResult;
  } catch (error) {
    console.error('Backtesting microservice call failed:', error);

    // Fallback to local execution if microservice is unavailable
    console.warn('🔄 Falling back to local backtesting execution');
    return runLocalBacktest(market, strategyName, parameters, timeframe, useIntrabarTicks, dynamicCode, walkForwardConfig, optimizationConfig);
  }
}

async function addSuccessfulStrategyToRAG(backtestResults: any, market: string, userMessage: string, strategyName: string, parameters: any): Promise<void> {
  // Evaluate backtest performance and add successful strategies to RAG vector store
  if (!backtestResults || !backtestResults.total_trades || backtestResults.total_trades < 20) {
    return; // Skip if insufficient trades
  }

  const winRate = backtestResults.win_rate || 0;
  const profitFactor = backtestResults.profit_factor || 0;
  const sharpeRatio = backtestResults.sharpe_ratio || 0;
  const totalReturn = backtestResults.total_profit || 0;

  // Define success criteria (adjustable thresholds)
  const isSuccessful =
    (winRate >= 55 && profitFactor >= 1.2) || // Good win rate with positive profit factor
    (sharpeRatio >= 1.5) || // Good risk-adjusted returns
    (totalReturn > 1000); // Significant absolute profit

  if (!isSuccessful) {
    console.log('📊 Backtest performance below threshold for RAG learning:', {
      winRate, profitFactor, sharpeRatio, totalReturn
    });
    return;
  }

  console.log('🎯 Adding successful strategy to RAG vector store for global learning');

  // Create a comprehensive strategy summary
  const strategySummary = `
STRATEGY SUCCESS CASE STUDY

Market: ${market.toUpperCase()}
Strategy: ${strategyName}
Performance Metrics:
- Win Rate: ${winRate}%
- Profit Factor: ${profitFactor}
- Sharpe Ratio: ${sharpeRatio}
- Total Return: $${totalReturn}
- Total Trades: ${backtestResults.total_trades}

User Request: "${userMessage}"

Strategy Parameters: ${JSON.stringify(parameters, null, 2)}

Key Insights:
- ${winRate >= 60 ? 'High win rate strategy' : 'Consistent profitability despite lower win rate'}
- ${profitFactor >= 1.5 ? 'Excellent risk-reward ratio' : 'Solid profit factor'}
- ${sharpeRatio >= 2.0 ? 'Exceptional risk-adjusted returns' : 'Good risk management'}
- Demonstrates effectiveness in ${market} market conditions

Lessons Learned:
- ${market === 'nba' ? 'NBA betting requires statistical edge identification and understanding of team dynamics' : ''}
- ${market === 'nfl' ? 'NFL betting benefits from spread analysis and divisional matchup awareness' : ''}
- ${market === 'mlb' ? 'MLB betting requires understanding pitching matchups and run line value' : ''}
- ${market === 'nhl' ? 'NHL betting benefits from goaltending analysis and puck line strategies' : ''}
- Sports betting requires proper bankroll management and edge identification

This strategy achieved ${winRate}% win rate with ${profitFactor} profit factor, proving effective in ${market.toUpperCase()} betting.
  `.trim();

  try {
    // Add to RAG vector store via the FastAPI microservice
    const baseUrl = process.env.PYTHON_SERVICE_URL || 'http://localhost:8000';
    const apiKey = process.env.FLASK_API_KEY;
    if (!apiKey) {
      throw new Error('FLASK_API_KEY environment variable is not set');
    }

    const ragResponse = await fetch(`${baseUrl}/rag/add-insight`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey,
      },
      body: JSON.stringify({
        content: strategySummary,
        metadata: {
          type: 'successful_strategy',
          market: market,
          strategy: strategyName,
          winRate: winRate,
          profitFactor: profitFactor,
          sharpeRatio: sharpeRatio,
          totalReturn: totalReturn,
          timestamp: new Date().toISOString()
        },
        source: 'backtest_feedback_loop'
      })
    });

    if (ragResponse.ok) {
      console.log('✅ Successfully added strategy to RAG vector store for global learning');
    } else {
      console.warn('⚠️ Failed to add strategy to RAG vector store:', await ragResponse.text());
    }
  } catch (error) {
    console.warn('⚠️ RAG API call failed:', error);
    // Don't throw - this shouldn't break the backtest
  }
}

function generateStrategyEquation(message: string, market: string): string {
  const messageLower = message.toLowerCase();

  // Extract parameters
  const stakeMatch = messageLower.match(/\$(\d+)/);
  const stake = stakeMatch ? parseInt(stakeMatch[1]) : 1000;

  let equation = '';

  // NBA Strategies
  if (market === 'nba') {
    if (messageLower.includes('player') && messageLower.includes('points') && messageLower.includes('over')) {
      const playerMatch = messageLower.match(/player (\w+)/i);
      const pointsMatch = messageLower.match(/(\d+) points/);
      const player = playerMatch ? playerMatch[1] : 'Player';
      const threshold = pointsMatch ? parseInt(pointsMatch[1]) : 25;

      equation = `**NBA Player Points Over Strategy**\n\n`;
      equation += `**Equation:** If Player_Points > ${threshold}, then WIN\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Player: ${player.charAt(0).toUpperCase() + player.slice(1)}\n`;
      equation += `• Points Threshold: ${threshold}\n`;
      equation += `• Stake per Trade: $${stake}\n`;
      equation += `• Win Payout: +$${stake}\n`;
      equation += `• Loss Payout: -$${stake}\n\n`;
      equation += `**Logic:** Bet on the player scoring more than ${threshold} points in each game.`;

    } else if (messageLower.includes('win streak') || messageLower.includes('hot team')) {
      const streakMatch = messageLower.match(/(\d+) game/);
      const minStreak = streakMatch ? parseInt(streakMatch[1]) : 3;

      equation = `**NBA Team Win Streak Strategy**\n\n`;
      equation += `**Equation:** If Team_Win_Streak ≥ ${minStreak}, then bet on WIN\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Win Streak: ${minStreak} games\n`;
      equation += `• Stake per Trade: $${stake}\n`;
      equation += `• Win Payout: +$${stake}\n`;
      equation += `• Loss Payout: -$${stake}\n\n`;
      equation += `**Logic:** Bet on teams that have won ${minStreak} or more games in a row.`;

    } else if (messageLower.includes('home') && messageLower.includes('advantage')) {
      equation = `**NBA Home Court Advantage Strategy**\n\n`;
      equation += `**Equation:** Bet on HOME_TEAM when Home_Win_Rate > 0.6\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Home Win Rate: 60%\n`;
      equation += `• Stake per Trade: $${stake}\n`;
      equation += `• Win Payout: +$${stake}\n`;
      equation += `• Loss Payout: -$${stake}\n\n`;
      equation += `**Logic:** Bet on home teams with >60% home court win rate.`;

    } else {
      // Default NBA strategy
      equation = `**NBA Moneyline Prediction Strategy**\n\n`;
      equation += `**Equation:** If Home_Win_Probability > 0.55, then bet on HOME_TEAM\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Win Probability Threshold: 55%\n`;
      equation += `• Stake per Trade: $${stake}\n`;
      equation += `• Win Payout: +$${stake}\n`;
      equation += `• Loss Payout: -$${stake}\n\n`;
      equation += `**Logic:** Bet on teams with >55% predicted win probability.`;
    }

    // NFL Strategies
  } else if (market === 'nfl') {
    if (messageLower.includes('underdog')) {
      equation = `**NFL Underdog Betting Strategy**\n\n`;
      equation += `**Equation:** If Spread > 7 AND Away_Team_Win_Rate > 0.4, then bet UNDERDOG\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Spread: 7 points\n`;
      equation += `• Minimum Away Win Rate: 40%\n`;
      equation += `• Stake per Trade: $${stake}\n`;
      equation += `• Win Payout: +$${stake}\n`;
      equation += `• Loss Payout: -$${stake}\n\n`;
      equation += `**Logic:** Bet on underdogs with large spreads who have decent away records.`;
    } else {
      equation = `**NFL Spread Betting Strategy**\n\n`;
      equation += `**Equation:** If Team_Win_Rate > 0.6 AND Spread < 3, then bet FAVORITE\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Win Rate: 60%\n`;
      equation += `• Maximum Spread: 3 points\n`;
      equation += `• Stake per Trade: $${stake}\n\n`;
      equation += `**Logic:** Bet on favorites with strong records and small spreads.`;
    }

    // MLB Strategies
  } else if (market === 'mlb') {
    if (messageLower.includes('pitcher') || messageLower.includes('era')) {
      equation = `**MLB Pitcher ERA Strategy**\n\n`;
      equation += `**Equation:** If Starting_Pitcher_ERA < 3.5 AND Opponent_BA < 0.250, then bet MONEYLINE\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Maximum Pitcher ERA: 3.50\n`;
      equation += `• Maximum Opponent Batting Average: .250\n`;
      equation += `• Stake per Trade: $${stake}\n\n`;
      equation += `**Logic:** Bet on teams with quality pitching facing weak offenses.`;
    } else {
      equation = `**MLB Moneyline Strategy**\n\n`;
      equation += `**Equation:** If Home_Win_Rate > 0.55, then bet HOME_TEAM\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Home Win Rate: 55%\n`;
      equation += `• Stake per Trade: $${stake}\n\n`;
      equation += `**Logic:** Bet on home teams with strong home records.`;
    }

    // NHL Strategies
  } else if (market === 'nhl') {
    if (messageLower.includes('goalie') || messageLower.includes('save')) {
      equation = `**NHL Goalie Performance Strategy**\n\n`;
      equation += `**Equation:** If Goalie_Save_Pct > 0.920 AND Goals_Against_Avg < 2.5, then bet MONEYLINE\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Save Percentage: .920\n`;
      equation += `• Maximum Goals Against Average: 2.50\n`;
      equation += `• Stake per Trade: $${stake}\n\n`;
      equation += `**Logic:** Bet on teams with elite goaltending.`;
    } else {
      equation = `**NHL Moneyline Strategy**\n\n`;
      equation += `**Equation:** If Home_Win_Rate > 0.55, then bet HOME_TEAM\n\n`;
      equation += `**Parameters:**\n`;
      equation += `• Minimum Home Win Rate: 55%\n`;
      equation += `• Stake per Trade: $${stake}\n\n`;
      equation += `**Logic:** Bet on home teams with strong home records.`;
    }

    // College Football Strategies
  } else if (market === 'cfb') {
    equation = `**College Football Spread Strategy**\n\n`;
    equation += `**Equation:** If Ranked_Team AND Spread > 14, then bet UNDERDOG\n\n`;
    equation += `**Parameters:**\n`;
    equation += `• Ranked Team Required: Yes\n`;
    equation += `• Minimum Spread: 14 points\n`;
    equation += `• Stake per Trade: $${stake}\n\n`;
    equation += `**Logic:** Fade big spreads - ranked teams often underperform large expectations.`;

    // College Basketball Strategies
  } else if (market === 'cbb') {
    equation = `**College Basketball Spread Strategy**\n\n`;
    equation += `**Equation:** If Conference_Game AND Spread > 10, then bet UNDERDOG\n\n`;
    equation += `**Parameters:**\n`;
    equation += `• Conference Game: Yes\n`;
    equation += `• Minimum Spread: 10 points\n`;
    equation += `• Stake per Trade: $${stake}\n\n`;
    equation += `**Logic:** Conference underdogs cover more often in rivalry games.`;

    // Default sports strategy
  } else {
    equation = `**Sports Betting Strategy**\n\n`;
    equation += `**Equation:** If Win_Probability > 0.55, then bet on FAVORITE\n\n`;
    equation += `**Parameters:**\n`;
    equation += `• Win Probability Threshold: 55%\n`;
    equation += `• Stake per Trade: $${stake}\n\n`;
    equation += `**Logic:** Bet on teams with >55% predicted win probability.`;
  }

  return equation || `**Strategy Analysis**\n\nAnalyzing your request: "${message}"\n\nMarket: ${market.toUpperCase()}\nStake: $${stake}`;
}

async function generateBacktestVisualization(backtestResults: any): Promise<any> {
  if (!backtestResults.success) {
    return null;
  }

  // Use trades from all_trades (full log) or trades (summary) or chart_visualization from Python
  const trades = backtestResults.all_trades || backtestResults.trades || [];
  const marketLabel = backtestResults.market || backtestResults.sport || 'general';

  // If Python already generated visualization, use it
  if (backtestResults.chart_visualization && !backtestResults.chart_visualization.error) {
    const cv = backtestResults.chart_visualization;

    // Plotly traces array
    if (Array.isArray(cv)) {
      return {
        chartType: 'profit_chart',
        market: marketLabel,
        format: 'plotly',
        chartData: cv,
        trades: trades.slice(-50),
      };
    }

    // Plotly figure object { data, layout }
    if (cv?.data && Array.isArray(cv.data)) {
      return {
        chartType: 'profit_chart',
        market: marketLabel,
        format: 'plotly',
        chartData: cv.data,
        trades: trades.slice(-50),
      };
    }

    // Sports engine returns Chart.js-style payload { type, labels, datasets }.
    // Prefer client-side rendering (no external CDN script dependency).
    if (Array.isArray(cv?.labels) && Array.isArray(cv?.datasets) && cv.datasets.length) {
      const mappedTrades = (trades as any[]).map((t, i) => ({
        id: t?.id || `${marketLabel}-${i}`,
        date: t?.date || t?.timestamp || new Date().toISOString(),
        outcome: t?.outcome === 'win' ? 'win' : 'loss',
        profit: typeof t?.profit === 'number' ? t.profit : Number.parseFloat(t?.profit || '0'),
        label: t?.game || t?.selection || t?.reason || `Trade ${i + 1}`,
        context: t,
      }));

      return {
        chartType: 'profit_over_time',
        market: marketLabel,
        format: 'client_render',
        trades: mappedTrades.slice(-200),
        summary: cv.summary,
      };
    }

    return {
      chartType: 'profit_chart',
      market: marketLabel,
      format: 'plotly',
      chartData: cv,
      trades: trades.slice(-50), // Include last 50 trades for reference
    };
  }

  if (trades.length === 0) {
    return null;
  }

  try {
    // Call Python visualization service directly
    const baseUrl = process.env.PYTHON_SERVICE_URL || 'http://localhost:8000';

    const response = await fetch(`${baseUrl}/visualization/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        chart_type: 'profit_chart',
        trades: trades,
        market: backtestResults.market || 'general',
        format: 'plotly'
      }),
      signal: AbortSignal.timeout(30000), // 30 second timeout
    });

    if (response.ok) {
      const chartResult = await response.json();
      return {
        chartType: 'profit_chart',
        market: backtestResults.market || 'general',
        format: 'plotly',
        chartData: chartResult,
        trades: trades.slice(-50), // Include last 50 trades for reference
      };
    }

    // Fallback: Return trades data for client-side charting
    console.warn('Python visualization service unavailable, returning trades for client-side rendering');
    return {
      chartType: 'profit_chart',
      market: backtestResults.market || 'general',
      format: 'client_render',
      trades: trades.slice(-100), // Last 100 trades for client-side chart
      summary: {
        total_trades: trades.length,
        total_profit: backtestResults.results?.total_profit || 0,
        win_rate: backtestResults.results?.win_rate || 0,
      }
    };
  } catch (error) {
    console.error('Error generating backtest visualization:', error);
    // Return trades data for fallback client-side rendering
    return {
      chartType: 'profit_chart',
      market: backtestResults.market || 'general',
      format: 'client_render',
      trades: trades.slice(-100),
      error: 'Visualization service timeout - showing raw trade data'
    };
  }
}

