/**
 * Caching Layer for Performance Optimization
 * In-memory cache with Redis-like interface for easy migration
 */

import { logger } from './error-handler';

export interface CacheOptions {
  ttl?: number; // Time to live in milliseconds
  tags?: string[]; // Cache tags for bulk invalidation
}

export interface CacheEntry<T = any> {
  data: T;
  timestamp: number;
  ttl: number;
  tags: string[];
  hits: number;
}

export class Cache {
  private static instance: Cache;
  private cache = new Map<string, CacheEntry>();
  private tagIndex = new Map<string, Set<string>>(); // tag -> keys

  private constructor() {
    // Clean up expired entries every 5 minutes
    setInterval(() => this.cleanup(), 5 * 60 * 1000);
  }

  static getInstance(): Cache {
    if (!Cache.instance) {
      Cache.instance = new Cache();
    }
    return Cache.instance;
  }

  // Core cache operations
  async get<T>(key: string): Promise<T | null> {
    const entry = this.cache.get(key);

    if (!entry) {
      return null;
    }

    // Check if expired
    if (Date.now() - entry.timestamp > entry.ttl) {
      this.delete(key);
      return null;
    }

    // Update hit count
    entry.hits++;

    logger.debug('Cache hit', { key, hits: entry.hits });
    return entry.data;
  }

  async set<T>(key: string, data: T, options: CacheOptions = {}): Promise<void> {
    const { ttl = 5 * 60 * 1000, tags = [] } = options; // Default 5 minutes TTL

    const entry: CacheEntry<T> = {
      data,
      timestamp: Date.now(),
      ttl,
      tags,
      hits: 0,
    };

    this.cache.set(key, entry);

    // Update tag index
    for (const tag of tags) {
      if (!this.tagIndex.has(tag)) {
        this.tagIndex.set(tag, new Set());
      }
      this.tagIndex.get(tag)!.add(key);
    }

    logger.debug('Cache set', { key, ttl, tags });
  }

  async delete(key: string): Promise<boolean> {
    const entry = this.cache.get(key);
    if (!entry) {
      return false;
    }

    // Remove from tag index
    for (const tag of entry.tags) {
      const tagKeys = this.tagIndex.get(tag);
      if (tagKeys) {
        tagKeys.delete(key);
        if (tagKeys.size === 0) {
          this.tagIndex.delete(tag);
        }
      }
    }

    this.cache.delete(key);
    logger.debug('Cache delete', { key });
    return true;
  }

  async clear(pattern?: string): Promise<number> {
    let deleted = 0;

    if (pattern) {
      // Delete keys matching pattern
      for (const key of this.cache.keys()) {
        if (key.includes(pattern)) {
          this.delete(key);
          deleted++;
        }
      }
    } else {
      // Clear all
      deleted = this.cache.size;
      this.cache.clear();
      this.tagIndex.clear();
    }

    logger.debug('Cache clear', { pattern, deleted });
    return deleted;
  }

  async invalidateTag(tag: string): Promise<number> {
    const tagKeys = this.tagIndex.get(tag);
    if (!tagKeys) {
      return 0;
    }

    let deleted = 0;
    for (const key of tagKeys) {
      this.delete(key);
      deleted++;
    }

    this.tagIndex.delete(tag);
    logger.debug('Tag invalidated', { tag, deleted });
    return deleted;
  }

  async invalidateTags(tags: string[]): Promise<number> {
    let totalDeleted = 0;
    for (const tag of tags) {
      totalDeleted += await this.invalidateTag(tag);
    }
    return totalDeleted;
  }

  // Utility methods
  async has(key: string): Promise<boolean> {
    const entry = this.cache.get(key);
    if (!entry) return false;

    // Check if expired
    if (Date.now() - entry.timestamp > entry.ttl) {
      this.delete(key);
      return false;
    }

    return true;
  }

  async getStats(): Promise<{
    entries: number;
    tags: number;
    hitRate: number;
    totalHits: number;
    oldestEntry: number;
    newestEntry: number;
  }> {
    const now = Date.now();
    let totalHits = 0;
    let oldestEntry = now;
    let newestEntry = 0;

    for (const entry of this.cache.values()) {
      totalHits += entry.hits;
      oldestEntry = Math.min(oldestEntry, entry.timestamp);
      newestEntry = Math.max(newestEntry, entry.timestamp);
    }

    return {
      entries: this.cache.size,
      tags: this.tagIndex.size,
      hitRate: totalHits > 0 ? totalHits / (totalHits + (this.cache.size - totalHits)) : 0,
      totalHits,
      oldestEntry: oldestEntry === now ? 0 : now - oldestEntry,
      newestEntry: newestEntry === 0 ? 0 : now - newestEntry,
    };
  }

  private cleanup(): void {
    const now = Date.now();
    let cleaned = 0;

    for (const [key, entry] of this.cache.entries()) {
      if (now - entry.timestamp > entry.ttl) {
        this.delete(key);
        cleaned++;
      }
    }

    if (cleaned > 0) {
      logger.debug('Cache cleanup completed', { cleaned });
    }
  }
}

// Cache keys and tags constants
export const CACHE_KEYS = {
  // User data
  userProfile: (userId: string) => `user:profile:${userId}`,
  userStats: (userId: string) => `user:stats:${userId}`,

  // Strategy data
  strategyDetails: (strategyId: string) => `strategy:details:${strategyId}`,
  userStrategies: (userId: string, page: number) => `user:strategies:${userId}:${page}`,
  publicStrategies: (domain?: string, page?: number) => `strategies:public:${domain || 'all'}:${page || 1}`,

  // Analytics
  userAnalytics: (userId: string, period: string) => `analytics:user:${userId}:${period}`,
  systemStats: () => 'stats:system',

  // API responses
  datasetOverview: () => 'api:datasets:overview',
  ragInsights: (query: string) => `rag:insights:${Buffer.from(query).toString('base64').substring(0, 20)}`,
};

export const CACHE_TAGS = {
  user: (userId: string) => `user:${userId}`,
  strategy: (strategyId: string) => `strategy:${strategyId}`,
  strategies: 'strategies',
  analytics: 'analytics',
  system: 'system',
};

// Helper functions for common caching patterns
export class CacheHelpers {
  static async cached<T>(
    key: string,
    fetcher: () => Promise<T>,
    options: CacheOptions = {}
  ): Promise<T> {
    const cache = Cache.getInstance();

    // Try cache first
    const cached = await cache.get<T>(key);
    if (cached !== null) {
      return cached;
    }

    // Fetch fresh data
    const data = await fetcher();

    // Cache the result
    await cache.set(key, data, options);

    return data;
  }

  static async invalidateUserData(userId: string): Promise<void> {
    const cache = Cache.getInstance();
    await cache.invalidateTag(CACHE_TAGS.user(userId));
  }

  static async invalidateStrategyData(strategyId: string): Promise<void> {
    const cache = Cache.getInstance();
    await cache.invalidateTags([
      CACHE_TAGS.strategy(strategyId),
      CACHE_TAGS.strategies,
    ]);
  }

  static async invalidateAnalytics(): Promise<void> {
    const cache = Cache.getInstance();
    await cache.invalidateTag(CACHE_TAGS.analytics);
  }
}

// Export singleton instance
export const cache = Cache.getInstance();
