/**
 * Authentication and Authorization Middleware
 * Handles JWT tokens, user verification, and subscription checks
 */

import { NextRequest, NextResponse } from 'next/server';
import { DatabaseService } from './database';
import jwt from 'jsonwebtoken';

// Re-export DatabaseService for backward compatibility
export { DatabaseService } from './database';

// CRITICAL SECURITY FIX: Remove hardcoded fallback - environment variable MUST be set
const JWT_SECRET = process.env.JWT_SECRET;

if (!JWT_SECRET) {
  throw new Error(
    'CRITICAL SECURITY ERROR: JWT_SECRET environment variable is not set. ' +
    'This is a critical security vulnerability. Set JWT_SECRET in your environment variables immediately.'
  );
}

// Types
export interface AuthUser {
  id: string;
  email: string;
  subscriptionTier: string;
  role?: string;
  firstName?: string;
  lastName?: string;
}

export interface AuthResult {
  user: AuthUser | null;
  error?: string;
}

// JWT Token Operations
export class AuthService {
  static generateToken(user: AuthUser): string {
    return jwt.sign(
      {
        userId: user.id,
        email: user.email,
        subscriptionTier: user.subscriptionTier,
        role: user.role || 'USER',
      },
      JWT_SECRET,
      { expiresIn: '7d' }
    );
  }

  static verifyToken(token: string): AuthUser | null {
    try {
      const decoded = jwt.verify(token, JWT_SECRET) as any;
      return {
        id: decoded.userId,
        email: decoded.email,
        subscriptionTier: decoded.subscriptionTier,
        role: decoded.role || 'USER',
      };
    } catch (error) {
      return null;
    }
  }

  static extractTokenFromRequest(request: NextRequest): string | null {
    // Check Authorization header
    const authHeader = request.headers.get('authorization');
    if (authHeader?.startsWith('Bearer ')) {
      return authHeader.substring(7);
    }

    // Check cookies
    const token = request.cookies.get('auth-token')?.value;
    if (token) {
      return token;
    }

    return null;
  }

  static async authenticateRequest(request: NextRequest): Promise<AuthResult> {
    const token = this.extractTokenFromRequest(request);

    if (!token) {
      return { user: null, error: 'No authentication token provided' };
    }

    const decodedUser = this.verifyToken(token);
    if (!decodedUser) {
      return { user: null, error: 'Invalid or expired token' };
    }

    // Fetch fresh user data from database
    const user = await DatabaseService.findUserById(decodedUser.id);
    if (!user) {
      return { user: null, error: 'User not found' };
    }

    // SECURITY FIX: Enforce email verification across all protected endpoints
    if (!user.emailVerified) {
      return { user: null, error: 'Email not verified. Please check your email and verify your account before accessing this feature.' };
    }

    const authUser: AuthUser = {
      id: user.id,
      email: user.email,
      subscriptionTier: user.subscriptionTier,
      role: user.role || decodedUser.role || 'USER',
      firstName: user.firstName || undefined,
      lastName: user.lastName || undefined,
    };

    return { user: authUser };
  }

  static checkSubscriptionTier(user: AuthUser, requiredTier: string): boolean {
    const tiers = ['FREE', 'TRIAL', 'PRO', 'ELITE'];
    const userTierIndex = tiers.indexOf(user.subscriptionTier);
    const requiredTierIndex = tiers.indexOf(requiredTier);

    // If user is on FREE tier and trial hasn't expired, give them PRO-level access
    if (user.subscriptionTier === 'FREE' && this.isTrialActive(user)) {
      return requiredTier === 'PRO' || requiredTier === 'FREE' || requiredTier === 'TRIAL';
    }

    return userTierIndex >= requiredTierIndex;
  }

  static isTrialActive(user: AuthUser): boolean {
    // This would need the full user object with trialEndsAt
    // For now, assume trial is active for FREE users (will be checked in API routes)
    return user.subscriptionTier === 'FREE';
  }
}

// Middleware Functions
export class AuthMiddleware {
  static async requireAuth(request: NextRequest): Promise<NextResponse | null> {
    const authResult = await AuthService.authenticateRequest(request);

    if (authResult.error || !authResult.user) {
      return NextResponse.json(
        { error: authResult.error || 'Authentication required' },
        { status: 401 }
      );
    }

    // Add user to request for downstream handlers
    (request as any).user = authResult.user;
    return null; // Continue to next handler
  }

  static async requireSubscription(
    request: NextRequest,
    requiredTier: string = 'FREE'
  ): Promise<NextResponse | null> {
    const authResult = await AuthService.authenticateRequest(request);

    if (authResult.error || !authResult.user) {
      return NextResponse.json(
        { error: authResult.error || 'Authentication required' },
        { status: 401 }
      );
    }

    if (!AuthService.checkSubscriptionTier(authResult.user, requiredTier)) {
      return NextResponse.json(
        {
          error: `This feature requires ${requiredTier} subscription or higher`,
          currentTier: authResult.user.subscriptionTier,
          requiredTier
        },
        { status: 403 }
      );
    }

    // Add user to request for downstream handlers
    (request as any).user = authResult.user;
    return null; // Continue to next handler
  }

  static async requireRole(
    request: NextRequest,
    requiredRole: string = 'USER'
  ): Promise<NextResponse | null> {
    const authResult = await AuthService.authenticateRequest(request);

    if (authResult.error || !authResult.user) {
      return NextResponse.json(
        { error: authResult.error || 'Authentication required' },
        { status: 401 }
      );
    }

    if (!authResult.user.role || authResult.user.role !== requiredRole) {
      return NextResponse.json(
        {
          error: `This feature requires ${requiredRole} role`,
          currentRole: authResult.user.role || 'USER',
          requiredRole
        },
        { status: 403 }
      );
    }

    // Add user to request for downstream handlers
    (request as any).user = authResult.user;
    return null; // Continue to next handler
  }

  static async optionalAuth(request: NextRequest): Promise<void> {
    const authResult = await AuthService.authenticateRequest(request);
    (request as any).user = authResult.user; // May be null
  }
}

// Rate Limiting - SECURITY FIX: Replace in-memory Map with file-based storage
// TODO: Replace with Redis (@upstash/ratelimit) for production scalability
export class RateLimiter {
  private static readonly RATE_LIMIT_FILE = './rate-limits.json';

  private static loadRateLimits(): Map<string, { count: number; resetTime: number }> {
    try {
      const fs = require('fs');
      if (fs.existsSync(this.RATE_LIMIT_FILE)) {
        const data = fs.readFileSync(this.RATE_LIMIT_FILE, 'utf8');
        const parsed = JSON.parse(data);
        // Convert back to Map
        return new Map(Object.entries(parsed));
      }
    } catch (error) {
      console.warn('Failed to load rate limits:', error);
    }
    return new Map();
  }

  private static saveRateLimits(requests: Map<string, { count: number; resetTime: number }>): void {
    try {
      const fs = require('fs');
      // Convert Map to plain object for JSON serialization
      const obj = Object.fromEntries(requests);
      fs.writeFileSync(this.RATE_LIMIT_FILE, JSON.stringify(obj, null, 2));
    } catch (error) {
      console.error('Failed to save rate limits:', error);
    }
  }

  static checkRateLimit(
    identifier: string,
    maxRequests: number = 100,
    windowMs: number = 15 * 60 * 1000 // 15 minutes
  ): boolean {
    const now = Date.now();
    const requests = this.loadRateLimits();
    const windowKey = identifier;

    const current = requests.get(windowKey);

    if (!current || now > current.resetTime) {
      // New window
      requests.set(windowKey, {
        count: 1,
        resetTime: now + windowMs,
      });
      this.saveRateLimits(requests);
      return true;
    }

    if (current.count >= maxRequests) {
      return false;
    }

    current.count++;
    this.saveRateLimits(requests);
    return true;
  }

  static getRemainingRequests(identifier: string): number {
    const requests = this.loadRateLimits();
    const current = requests.get(identifier);
    if (!current) return 100; // Default limit

    return Math.max(0, 100 - current.count);
  }

  static getResetTime(identifier: string): number {
    const requests = this.loadRateLimits();
    const current = requests.get(identifier);
    return current?.resetTime || Date.now();
  }
}

// API Security Middleware
export class SecurityMiddleware {
  static rateLimit(
    request: NextRequest,
    maxRequests: number = 100,
    windowMs: number = 15 * 60 * 1000
  ): NextResponse | null {
    // Use IP address or user ID for rate limiting
    const user = (request as any).user;
    const identifier = user?.id || (request as any).ip || 'anonymous';

    if (!RateLimiter.checkRateLimit(identifier, maxRequests, windowMs)) {
      const resetTime = RateLimiter.getResetTime(identifier);
      const remainingTime = Math.ceil((resetTime - Date.now()) / 1000);

      return NextResponse.json(
        {
          error: 'Rate limit exceeded',
          retryAfter: remainingTime,
          limit: maxRequests,
          remaining: 0
        },
        {
          status: 429,
          headers: {
            'Retry-After': remainingTime.toString(),
            'X-RateLimit-Limit': maxRequests.toString(),
            'X-RateLimit-Remaining': '0',
            'X-RateLimit-Reset': resetTime.toString(),
          },
        }
      );
    }

    return null; // Continue
  }

  static validateInput(data: any, schema: any): { valid: boolean; errors?: string[] } {
    // Basic validation - you might want to use Zod or Joi for more complex schemas
    const errors: string[] = [];

    if (schema.required) {
      for (const field of schema.required) {
        if (!data[field]) {
          errors.push(`Missing required field: ${field}`);
        }
      }
    }

    if (schema.stringFields) {
      for (const field of schema.stringFields) {
        if (data[field] && typeof data[field] !== 'string') {
          errors.push(`Field ${field} must be a string`);
        }
        if (data[field] && schema.maxLength && data[field].length > schema.maxLength[field]) {
          errors.push(`Field ${field} exceeds maximum length`);
        }
      }
    }

    if (schema.emailFields) {
      for (const field of schema.emailFields) {
        if (data[field] && !this.isValidEmail(data[field])) {
          errors.push(`Invalid email format: ${field}`);
        }
      }
    }

    return {
      valid: errors.length === 0,
      errors: errors.length > 0 ? errors : undefined,
    };
  }

  static isValidEmail(email: string): boolean {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  }

  static sanitizeInput(input: string): string {
    // Basic XSS prevention
    return input
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#x27;')
      .replace(/\//g, '&#x2F;');
  }

  static addSecurityHeaders(response: NextResponse): NextResponse {
    // Security headers
    response.headers.set('X-Content-Type-Options', 'nosniff');
    response.headers.set('X-Frame-Options', 'DENY');
    response.headers.set('X-XSS-Protection', '1; mode=block');
    response.headers.set('Referrer-Policy', 'strict-origin-when-cross-origin');
    response.headers.set('Permissions-Policy', 'geolocation=(), microphone=(), camera=()');

    // CORS headers for production domains
    response.headers.set('Access-Control-Allow-Origin', process.env.NODE_ENV === 'production'
      ? process.env.ALLOWED_ORIGINS || 'https://eventheodds.ai'
      : '*');
    response.headers.set('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
    response.headers.set('Access-Control-Allow-Headers', 'Content-Type, Authorization');
    response.headers.set('Access-Control-Max-Age', '86400');

    return response;
  }

  static handleOptionsRequest(): NextResponse {
    return new NextResponse(null, {
      status: 200,
      headers: {
        'Access-Control-Allow-Origin': '*',
        'Access-Control-Allow-Methods': 'GET, POST, PUT, DELETE, OPTIONS',
        'Access-Control-Allow-Headers': 'Content-Type, Authorization',
      },
    });
  }
}

// API Key Validation for External Services
export class APIKeyValidator {
  static validateGrokKey(): boolean {
    const key = process.env.GROK_API_KEY;
    return key && key.startsWith('xai-') && key.length > 20;
  }

  static validateStripeKeys(): boolean {
    const publishable = process.env.STRIPE_PUBLISHABLE_KEY;
    const secret = process.env.STRIPE_SECRET_KEY;

    return publishable?.startsWith('pk_') && secret?.startsWith('sk_');
  }

  static validateSMTPConfig(): boolean {
    const required = ['SMTP_HOST', 'SMTP_PORT', 'SMTP_USER', 'SMTP_PASS'];
    return required.every(key => process.env[key]);
  }
}

// Helper function to create secure API responses
export function createSecureResponse(
  data: any,
  status: number = 200,
  user?: AuthUser
): NextResponse {
  const response = NextResponse.json(data, { status });

  // Add user context if available
  if (user) {
    response.headers.set('X-User-ID', user.id);
    response.headers.set('X-Subscription-Tier', user.subscriptionTier);
  }

  // Add rate limit info
  const identifier = user?.id || 'anonymous';
  response.headers.set('X-RateLimit-Remaining', RateLimiter.getRemainingRequests(identifier).toString());
  response.headers.set('X-RateLimit-Reset', RateLimiter.getResetTime(identifier).toString());

  // Add security headers
  return SecurityMiddleware.addSecurityHeaders(response);
}

// Subscription-based feature access
export const SUBSCRIPTION_FEATURES = {
  FREE: {
    maxStrategies: 5,
    maxConversations: 10,
    ragAccess: false,
    advancedAnalytics: false,
    prioritySupport: false,
  },
  TRIAL: {
    maxStrategies: 20,
    maxConversations: 50,
    ragAccess: true,
    advancedAnalytics: false,
    prioritySupport: false,
  },
  PRO: {
    maxStrategies: 100,
    maxConversations: 500,
    ragAccess: true,
    advancedAnalytics: true,
    prioritySupport: true,
  },
  ELITE: {
    maxStrategies: -1, // Unlimited
    maxConversations: -1, // Unlimited
    ragAccess: true,
    advancedAnalytics: true,
    prioritySupport: true,
  },
};

export function checkFeatureAccess(userTier: string, feature: keyof typeof SUBSCRIPTION_FEATURES.FREE): boolean {
  return SUBSCRIPTION_FEATURES[userTier as keyof typeof SUBSCRIPTION_FEATURES][feature];
}

import { SubscriptionTier } from './types';

export async function authenticateAndAuthorize(
  request: NextRequest,
  requiredTiers?: SubscriptionTier[]
): Promise<{ status: number; response?: NextResponse; data?: { user: AuthUser } }> {
  // 1. Authenticate
  const authResult = await AuthService.authenticateRequest(request);
  if (authResult.error || !authResult.user) {
    return {
      status: 401,
      response: NextResponse.json({ error: authResult.error || 'Unauthorized' }, { status: 401 })
    };
  }

  // 2. Authorize (Tier check)
  if (requiredTiers && requiredTiers.length > 0) {
    // Check if user has access to ANY of the required tiers (or higher)
    // Logic: checkSubscriptionTier returns true if user is >= requiredTier.
    // So if ANY of the requiredTiers is satisfied?
    // Wait, usually list means "Must have one of these". But checkSubscriptionTier checks hierarchy.
    // If usage is [FREE, TRIAL, PRO, ELITE], it means basically anyone.
    // If usage is [PRO, ELITE], it means PRO or higher.
    // So we should check if user meets MINIMUM of the required tiers?
    // Or just satisfy ANY?
    // Given usage `[FREE, ... ELITE]`, it implies "Anyone in this list".
    // We can iterate.
    const hasAccess = requiredTiers.some(tier =>
      AuthService.checkSubscriptionTier(authResult.user!, tier)
    );
    if (!hasAccess) {
      return {
        status: 403,
        response: NextResponse.json({ error: 'Subscription tier insufficient' }, { status: 403 })
      };
    }
  }

  // 3. Success
  return {
    status: 200,
    data: { user: authResult.user }
  };
}
