/**
 * Strategy Engine Unit Tests
 * Tests backtesting logic, prompt parsing, and simulation accuracy
 */

import { promptToRule } from '../strategyEngine';
import { Domain } from '../types';

describe('Strategy Engine', () => {
  describe('promptToRule', () => {
    it('should parse momentum strategy prompts', () => {
      const prompt = 'Create a momentum strategy for stocks when price increases by 5%';
      const result = promptToRule(prompt, 'stocks');

      expect(result).toEqual({
        domain: 'stocks',
        lookback: 20, // Default
        trigger: 'momentumDip',
        threshold: 0.05,
        direction: 'long',
        description: expect.stringContaining('momentum'),
      });
    });

    it('should parse bounce strategy prompts', () => {
      const prompt = 'Buy stocks after 3 consecutive losses';
      const result = promptToRule(prompt, 'stocks');

      expect(result).toEqual({
        domain: 'stocks',
        lookback: 20,
        trigger: 'bounceAfterLosses',
        threshold: 3,
        direction: 'long',
        description: expect.stringContaining('bounce'),
      });
    });

    it('should parse parlay strategy prompts', () => {
      const prompt = 'Create a parlay strategy with $10 per leg for sports';
      const result = promptToRule(prompt, 'sports');

      expect(result).toEqual({
        domain: 'sports',
        lookback: 20,
        trigger: 'parlay',
        direction: 'long',
        description: expect.stringContaining('parlay'),
        stakePerLeg: 10,
      });
    });

    it('should handle different domains', () => {
      const cryptoPrompt = 'Crypto momentum strategy';
      const forexPrompt = 'Forex breakout strategy';

      const cryptoResult = promptToRule(cryptoPrompt, 'crypto');
      const forexResult = promptToRule(forexPrompt, 'forex');

      expect(cryptoResult.domain).toBe('crypto');
      expect(forexResult.domain).toBe('forex');
    });

    it('should use custom lookback when specified', () => {
      const prompt = 'Strategy with 50 day lookback for momentum';
      const result = promptToRule(prompt, 'stocks');

      expect(result.lookback).toBe(50);
    });
  });

  describe('edge cases', () => {
    it('should handle invalid prompts gracefully', () => {
      const invalidPrompt = '';
      const result = promptToRule(invalidPrompt, 'stocks');

      expect(result).toHaveProperty('domain');
      expect(result).toHaveProperty('trigger');
      expect(result).toHaveProperty('direction');
    });

    it('should handle extreme threshold values', () => {
      const extremePrompt = 'Strategy with 200% threshold';
      const result = promptToRule(extremePrompt, 'stocks');

      expect(result.threshold).toBe(2.0); // Should be normalized
    });

    it('should handle very short datasets', () => {
      const rule = {
        domain: 'stocks' as Domain,
        lookback: 50, // Much larger than dataset
        trigger: 'momentumDip' as const,
        threshold: 0.05,
        direction: 'long' as const,
        description: 'Test with small data',
      };

      const smallData = [mockStockData[0]]; // Only one record
      const result = simulateStrategyOnDataset(rule, smallData);

      // Should handle gracefully without crashing
      expect(Array.isArray(result)).toBe(true);
    });
  });
});