/**
 * Database Service Unit Tests
 * Tests user operations, strategy management, and analytics
 */

import { DatabaseService } from '../database';

jest.mock('@prisma/client', () => ({
  PrismaClient: jest.fn(() => ({
    user: {
      create: jest.fn(),
      findUnique: jest.fn(),
      findMany: jest.fn(),
      update: jest.fn(),
      delete: jest.fn(),
    },
    strategy: {
      create: jest.fn(),
      findUnique: jest.fn(),
      findMany: jest.fn(),
      update: jest.fn(),
      delete: jest.fn(),
      aggregate: jest.fn(),
    },
    conversation: {
      create: jest.fn(),
      findUnique: jest.fn(),
      findMany: jest.fn(),
      update: jest.fn(),
    },
    message: {
      create: jest.fn(),
    },
    userAnalytics: {
      create: jest.fn(),
    },
    systemAnalytics: {
      create: jest.fn(),
    },
    $connect: jest.fn(),
    $disconnect: jest.fn(),
    $queryRaw: jest.fn(),
  })),
}));

describe('DatabaseService', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('User Management', () => {
    const mockUser = {
      id: 'user_123',
      email: 'test@example.com',
      passwordHash: 'hashed_password',
      firstName: 'John',
      lastName: 'Doe',
      subscriptionTier: 'PRO',
      emailVerified: true,
      createdAt: new Date(),
      updatedAt: new Date(),
    };

    it('should create user successfully', async () => {
      const createData = {
        email: 'test@example.com',
        password: 'password123',
        firstName: 'John',
        lastName: 'Doe',
        subscriptionTier: 'FREE' as const,
      };

      const mockCreatedUser = { ...mockUser, ...createData };
      const prismaUserCreate = jest.fn().mockResolvedValue(mockCreatedUser);

      // Mock the prisma instance
      const mockPrisma = {
        user: { create: prismaUserCreate },
      };

      // Temporarily replace the prisma instance
      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.createUser(createData);

      expect(result).toEqual(mockCreatedUser);
      expect(prismaUserCreate).toHaveBeenCalledWith({
        data: {
          email: createData.email,
          passwordHash: expect.any(String), // Should be hashed
          firstName: createData.firstName,
          lastName: createData.lastName,
          subscriptionTier: createData.subscriptionTier,
        },
      });

      // Restore original prisma
      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should find user by email', async () => {
      const mockPrisma = {
        user: {
          findUnique: jest.fn().mockResolvedValue(mockUser),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.findUserByEmail('test@example.com');

      expect(result).toEqual(mockUser);
      expect(mockPrisma.user.findUnique).toHaveBeenCalledWith({
        where: { email: 'test@example.com' },
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should update user profile', async () => {
      const updateData = {
        firstName: 'Jane',
        riskTolerance: 'HIGH',
      };

      const updatedUser = { ...mockUser, ...updateData };
      const mockPrisma = {
        user: {
          update: jest.fn().mockResolvedValue(updatedUser),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.updateUser('user_123', updateData);

      expect(result).toEqual(updatedUser);
      expect(mockPrisma.user.update).toHaveBeenCalledWith({
        where: { id: 'user_123' },
        data: {
          firstName: 'Jane',
          riskTolerance: 'HIGH',
          updatedAt: expect.any(Date),
        },
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should generate verification tokens', () => {
      const token1 = DatabaseService.generateVerificationToken();
      const token2 = DatabaseService.generateVerificationToken();

      expect(typeof token1).toBe('string');
      expect(token1.length).toBeGreaterThan(0);
      expect(token1).not.toBe(token2); // Should be unique
    });
  });

  describe('Strategy Management', () => {
    const mockStrategy = {
      id: 'strategy_123',
      name: 'Test Strategy',
      description: 'A test strategy',
      domain: 'STOCKS',
      rules: { trigger: 'momentum', threshold: 0.05 },
      isPublic: false,
      userId: 'user_123',
      createdAt: new Date(),
      updatedAt: new Date(),
    };

    it('should create strategy', async () => {
      const createData = {
        name: 'Test Strategy',
        description: 'A test strategy',
        domain: 'STOCKS' as const,
        rules: { trigger: 'momentum', threshold: 0.05 },
        isPublic: false,
        userId: 'user_123',
      };

      const mockPrisma = {
        strategy: {
          create: jest.fn().mockResolvedValue(mockStrategy),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.createStrategy(createData);

      expect(result).toEqual(mockStrategy);
      expect(mockPrisma.strategy.create).toHaveBeenCalledWith({
        data: {
          name: createData.name,
          description: createData.description,
          domain: createData.domain,
          rules: createData.rules,
          isPublic: createData.isPublic,
          isTemplate: false,
          tags: [],
          userId: createData.userId,
        },
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should find strategies by user', async () => {
      const mockStrategies = [mockStrategy];
      const mockPrisma = {
        strategy: {
          findMany: jest.fn().mockResolvedValue(mockStrategies),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.findStrategiesByUser('user_123', 10);

      expect(result).toEqual(mockStrategies);
      expect(mockPrisma.strategy.findMany).toHaveBeenCalledWith({
        where: { userId: 'user_123' },
        orderBy: { updatedAt: 'desc' },
        take: 10,
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should find public strategies', async () => {
      const mockStrategies = [mockStrategy];
      const mockPrisma = {
        strategy: {
          findMany: jest.fn().mockResolvedValue(mockStrategies),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const result = await DatabaseService.findPublicStrategies(20);

      expect(result).toEqual(mockStrategies);
      expect(mockPrisma.strategy.findMany).toHaveBeenCalledWith({
        where: {
          isPublic: true,
        },
        include: expect.any(Object),
        orderBy: { likes: 'desc' },
        take: 20,
      });

      (DatabaseService as any).prisma = originalPrisma;
    });
  });

  describe('Analytics', () => {
    it('should record user events', async () => {
      const mockPrisma = {
        userAnalytics: {
          create: jest.fn().mockResolvedValue({ id: 'analytics_123' }),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      await DatabaseService.recordUserEvent('user_123', 'LOGIN_SUCCESS', {
        userAgent: 'test-agent',
        ip: '127.0.0.1',
      });

      expect(mockPrisma.userAnalytics.create).toHaveBeenCalledWith({
        data: {
          userId: 'user_123',
          eventType: 'LOGIN_SUCCESS',
          eventData: {
            userAgent: 'test-agent',
            ip: '127.0.0.1',
          },
          metadata: {},
        },
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should record strategy events', async () => {
      const mockPrisma = {
        strategyAnalytics: {
          create: jest.fn().mockResolvedValue({ id: 'analytics_123' }),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      await DatabaseService.recordStrategyEvent('strategy_123', 'STRATEGY_CREATED', {
        userId: 'user_123',
        domain: 'STOCKS',
      });

      expect(mockPrisma.strategyAnalytics.create).toHaveBeenCalledWith({
        data: {
          strategyId: 'strategy_123',
          eventType: 'STRATEGY_CREATED',
          eventData: {
            userId: 'user_123',
            domain: 'STOCKS',
          },
        },
      });

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should get user statistics', async () => {
      const mockPrisma = {
        strategy: {
          count: jest.fn().mockResolvedValue(5),
          aggregate: jest.fn().mockResolvedValue({ _sum: { likes: 25 } }),
        },
        conversation: {
          count: jest.fn().mockResolvedValue(10),
        },
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const stats = await DatabaseService.getUserStats('user_123');

      expect(stats).toEqual({
        totalStrategies: 5,
        totalConversations: 10,
        totalLikes: 25,
      });

      (DatabaseService as any).prisma = originalPrisma;
    });
  });

  describe('Password Operations', () => {
    it('should hash passwords', async () => {
      const password = 'testpassword123';
      const hash = await DatabaseService.hashPassword(password);

      expect(typeof hash).toBe('string');
      expect(hash).not.toBe(password);
      expect(hash.length).toBeGreaterThan(password.length);
    });

    it('should verify correct passwords', async () => {
      const password = 'testpassword123';
      const hash = await DatabaseService.hashPassword(password);

      const isValid = await DatabaseService.verifyPassword(password, hash);
      expect(isValid).toBe(true);
    });

    it('should reject incorrect passwords', async () => {
      const password = 'testpassword123';
      const hash = await DatabaseService.hashPassword(password);

      const isValid = await DatabaseService.verifyPassword('wrongpassword', hash);
      expect(isValid).toBe(false);
    });
  });

  describe('Health Checks', () => {
    it('should return healthy status when database is accessible', async () => {
      const mockPrisma = {
        $queryRaw: jest.fn().mockResolvedValue([]),
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const isHealthy = await DatabaseService.healthCheck();

      expect(isHealthy).toBe(true);
      expect(mockPrisma.$queryRaw).toHaveBeenCalledWith('SELECT 1');

      (DatabaseService as any).prisma = originalPrisma;
    });

    it('should return unhealthy status when database is inaccessible', async () => {
      const mockPrisma = {
        $queryRaw: jest.fn().mockRejectedValue(new Error('Connection failed')),
      };

      const originalPrisma = (DatabaseService as any).prisma;
      (DatabaseService as any).prisma = mockPrisma;

      const isHealthy = await DatabaseService.healthCheck();

      expect(isHealthy).toBe(false);

      (DatabaseService as any).prisma = originalPrisma;
    });
  });
});
