/**
 * Authentication Service Unit Tests
 * Tests JWT token generation, verification, and user authentication logic
 */

import { AuthService, AuthMiddleware } from '../auth';
import { DatabaseService } from '../database';

jest.mock('../database');

describe('AuthService', () => {
  const mockUser = {
    id: 'user_123',
    email: 'test@example.com',
    subscriptionTier: 'PRO' as const,
  };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('generateToken', () => {
    it('should generate a valid JWT token', () => {
      const token = AuthService.generateToken(mockUser);

      expect(typeof token).toBe('string');
      expect(token.split('.')).toHaveLength(3); // JWT has 3 parts
    });

    it('should include correct payload in token', () => {
      const token = AuthService.generateToken(mockUser);
      const decoded = AuthService.verifyToken(token);

      expect(decoded).toEqual({
        id: mockUser.id,
        email: mockUser.email,
        subscriptionTier: mockUser.subscriptionTier,
      });
    });
  });

  describe('verifyToken', () => {
    it('should verify a valid token', () => {
      const token = AuthService.generateToken(mockUser);
      const decoded = AuthService.verifyToken(token);

      expect(decoded).toEqual({
        id: mockUser.id,
        email: mockUser.email,
        subscriptionTier: mockUser.subscriptionTier,
      });
    });

    it('should return null for invalid token', () => {
      const decoded = AuthService.verifyToken('invalid.token.here');

      expect(decoded).toBeNull();
    });

    it('should return null for expired token', () => {
      // Create a token that expires immediately
      const pastToken = AuthService.generateToken(mockUser);
      // Mock JWT to be expired by advancing time (this would need a custom mock)

      const decoded = AuthService.verifyToken(pastToken);
      // Note: This test would need time manipulation or a custom JWT library mock
    });
  });

  describe('checkSubscriptionTier', () => {
    it('should allow access to same or lower tier', () => {
      expect(AuthService.checkSubscriptionTier(mockUser, 'FREE')).toBe(true);
      expect(AuthService.checkSubscriptionTier(mockUser, 'PRO')).toBe(true);
      expect(AuthService.checkSubscriptionTier({ ...mockUser, subscriptionTier: 'FREE' }, 'FREE')).toBe(true);
    });

    it('should deny access to higher tier', () => {
      expect(AuthService.checkSubscriptionTier(mockUser, 'ELITE')).toBe(false);
    });

    it('should grant PRO access to FREE users during trial', () => {
      const trialUser = { ...mockUser, subscriptionTier: 'FREE' as const };
      expect(AuthService.checkSubscriptionTier(trialUser, 'PRO')).toBe(true);
    });
  });
});

describe('AuthMiddleware', () => {
  let mockRequest: any;

  beforeEach(() => {
    mockRequest = {
      headers: new Map(),
      cookies: new Map(),
    };
  });

  describe('extractTokenFromRequest', () => {
    it('should extract token from Authorization header', () => {
      mockRequest.headers.set('authorization', 'Bearer test-token-123');
      const token = AuthService.extractTokenFromRequest(mockRequest);

      expect(token).toBe('test-token-123');
    });

    it('should extract token from cookies', () => {
      mockRequest.cookies.set('auth-token', { value: 'cookie-token-456' });
      const token = AuthService.extractTokenFromRequest(mockRequest);

      expect(token).toBe('cookie-token-456');
    });

    it('should prefer Authorization header over cookies', () => {
      mockRequest.headers.set('authorization', 'Bearer header-token');
      mockRequest.cookies.set('auth-token', { value: 'cookie-token' });

      const token = AuthService.extractTokenFromRequest(mockRequest);
      expect(token).toBe('header-token');
    });

    it('should return null when no token found', () => {
      const token = AuthService.extractTokenFromRequest(mockRequest);
      expect(token).toBeNull();
    });
  });

  describe('authenticateRequest', () => {
    beforeEach(() => {
      (DatabaseService.findUserById as jest.Mock).mockResolvedValue(mockUser);
    });

    it('should authenticate valid token', async () => {
      const token = AuthService.generateToken(mockUser);
      mockRequest.headers.set('authorization', `Bearer ${token}`);

      const result = await AuthService.authenticateRequest(mockRequest);

      expect(result.user).toEqual({
        id: mockUser.id,
        email: mockUser.email,
        subscriptionTier: mockUser.subscriptionTier,
        firstName: undefined,
        lastName: undefined,
      });
    });

    it('should return error for invalid token', async () => {
      mockRequest.headers.set('authorization', 'Bearer invalid-token');

      const result = await AuthService.authenticateRequest(mockRequest);

      expect(result.user).toBeNull();
      expect(result.error).toBe('Invalid or expired token');
    });

    it('should return error for missing token', async () => {
      const result = await AuthService.authenticateRequest(mockRequest);

      expect(result.user).toBeNull();
      expect(result.error).toBe('No authentication token provided');
    });
  });
});
