'use client';

import Link from 'next/link';
import { useEffect, useMemo, useState } from 'react';

interface StrategyRun {
  id: string;
  timestamp: string;
  market: string;
  name: string;
  trades: number;
  winRate: number; // 0..1
  profit: number;
  status: 'completed' | 'running' | 'failed';
}

function normalizeStatus(status: any): StrategyRun['status'] {
  const s = String(status || '').toUpperCase();
  if (s.includes('RUN')) return 'running';
  if (s.includes('FAIL') || s.includes('ERROR')) return 'failed';
  return 'completed';
}

function normalizeWinRate(raw: any): number {
  const n = Number(raw);
  if (!Number.isFinite(n)) return 0;
  // Stored backtests often use %; UI wants 0..1.
  if (n > 1) return n / 100;
  return Math.max(0, Math.min(1, n));
}

export function AdminStrategiesPage() {
  const [runs, setRuns] = useState<StrategyRun[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  const fetchRuns = async () => {
    try {
      setLoading(true);
      setError(null);

      const token = localStorage.getItem('authToken');
      const headers: Record<string, string> = { 'Content-Type': 'application/json' };
      if (token) headers.Authorization = `Bearer ${token}`;

      const res = await fetch('/api/admin/backtests?limit=50', {
        headers,
        credentials: 'include',
      });
      if (!res.ok) {
        const data = await res.json().catch(() => ({}));
        throw new Error(data?.error || `Failed to load backtests (${res.status})`);
      }

      const data = await res.json();
      const items: StrategyRun[] = (data?.backtests || []).map((bt: any) => {
        const trades = Number(bt?.summary?.totalTrades ?? 0);
        const winRate = normalizeWinRate(bt?.summary?.winRate);
        const profit = Number(bt?.summary?.totalProfit ?? 0);

        return {
          id: bt.id,
          timestamp: bt.createdAt,
          market: String(bt.market || 'sports'),
          name: String(bt.strategyName || 'Backtest'),
          trades: Number.isFinite(trades) ? trades : 0,
          winRate,
          profit: Number.isFinite(profit) ? profit : 0,
          status: normalizeStatus(bt.status),
        };
      });

      setRuns(items);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to load backtests');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    void fetchRuns();
  }, []);

  const marketBreakdown = useMemo(() => {
    const map = new Map<string, number>();
    runs.forEach((r) => {
      const key = r.market.toUpperCase();
      map.set(key, (map.get(key) || 0) + 1);
    });
    return Array.from(map.entries()).sort((a, b) => b[1] - a[1]);
  }, [runs]);

  const avgWinRate = useMemo(() => {
    const withTrades = runs.filter((r) => r.trades > 0);
    if (withTrades.length === 0) return 0;
    return withTrades.reduce((sum, r) => sum + r.winRate, 0) / withTrades.length;
  }, [runs]);

  return (
    <div className="space-y-8">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-white">Backtest Runs</h1>
          <p className="text-slate-400">Latest backtests across all users.</p>
        </div>
        <div className="flex gap-3">
          <button
            onClick={() => void fetchRuns()}
            className="rounded-lg border border-white/10 bg-slate-800/50 px-4 py-2 text-sm font-medium text-slate-200 hover:bg-slate-700/50"
            disabled={loading}
          >
            Refresh
          </button>
          <Link
            href="/dashboard"
            className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-500"
          >
            Run a backtest →
          </Link>
        </div>
      </div>

      {error && (
        <div className="rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
          {error}
        </div>
      )}

      <div className="rounded-2xl border border-white/10 bg-slate-900/50">
        <div className="border-b border-white/10 px-6 py-4">
          <h3 className="font-semibold text-white">Recent Backtests</h3>
        </div>
        <div className="divide-y divide-white/5">
          {loading ? (
            <div className="px-6 py-6 text-sm text-slate-400">Loading…</div>
          ) : runs.length === 0 ? (
            <div className="px-6 py-6 text-sm text-slate-400">No backtests found yet.</div>
          ) : runs.map((run) => (
            <div key={run.id} className="flex items-center justify-between px-6 py-4 hover:bg-white/5">
              <div className="flex items-center gap-4">
                <div className={`flex h-8 w-8 items-center justify-center rounded-full ${
                  run.status === 'completed' ? 'bg-emerald-500/20 text-emerald-400' :
                  run.status === 'running' ? 'bg-blue-500/20 text-blue-400' :
                  'bg-rose-500/20 text-rose-400'
                }`}>
                  {run.status === 'completed' ? (
                    <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                    </svg>
                  ) : run.status === 'running' ? (
                    <svg className="h-4 w-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                    </svg>
                  ) : (
                    <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                  )}
                </div>
                <div>
                  <p className="font-medium text-slate-200">{run.name}</p>
                  <div className="flex gap-4 text-xs text-slate-500">
                    <span className="uppercase">{run.market}</span>
                    <span>•</span>
                    <span>{new Date(run.timestamp).toLocaleString()}</span>
                  </div>
                </div>
              </div>
              <div className="flex items-center gap-6 text-right">
                <div>
                  <p className="text-sm font-medium text-white">{run.trades} trades</p>
                  <p className="text-xs text-slate-500">
                    {run.winRate > 0 ? `${(run.winRate * 100).toFixed(0)}% win rate` : 'No trades'}
                  </p>
                </div>
                <div>
                  <p className="text-sm font-medium text-white">
                    {run.profit > 0 ? `+${run.profit.toFixed(2)}` : run.profit.toFixed(2)}
                  </p>
                  <p className="text-xs text-slate-500">profit</p>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-3">
        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Performance Summary</h3>
          <div className="mt-4 space-y-4">
            <div className="flex items-center justify-between border-b border-white/5 pb-2">
              <span className="text-sm text-slate-300">Total Runs</span>
              <span className="text-sm font-bold text-white">{runs.length}</span>
            </div>
            <div className="flex items-center justify-between border-b border-white/5 pb-2">
              <span className="text-sm text-slate-300">Successful</span>
              <span className="text-sm font-bold text-emerald-400">
                {runs.filter(r => r.status === 'completed').length}
              </span>
            </div>
            <div className="flex items-center justify-between border-b border-white/5 pb-2">
              <span className="text-sm text-slate-300">Avg Win Rate</span>
              <span className="text-sm font-bold text-white">
                {(avgWinRate * 100).toFixed(0)}%
              </span>
            </div>
          </div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Market Breakdown</h3>
          <div className="mt-4 space-y-4">
            {marketBreakdown.length === 0 ? (
              <div className="text-sm text-slate-400">No data yet.</div>
            ) : (
              marketBreakdown.map(([market, count]) => (
                <div key={market} className="flex items-center justify-between">
                  <span className="text-sm text-slate-300">{market}</span>
                  <span className="text-sm font-bold text-white">{count}</span>
                </div>
              ))
            )}
          </div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Notes</h3>
          <p className="mt-2 text-sm text-slate-400">
            For detailed charts and exports, use the user-facing backtest history at <span className="text-slate-200">/backtest</span>.
          </p>
        </div>
      </div>
    </div>
  );
}
