'use client';

import { useEffect, useState } from 'react';

interface AdminSettingsResponse {
  keys: {
    grokConfigured: boolean;
    ballDontLieConfigured: boolean;
    stripeConfigured: boolean;
    smtpConfigured: boolean;
    ragApiKeyConfigured: boolean;
  };
  services: {
    rag: { url: string; healthy: boolean };
    python: { url: string; healthy: boolean };
  };
  env: {
    nodeEnv: string;
    baseUrl: string | null;
  };
}

export function AdminSettingsPage() {
  const [data, setData] = useState<AdminSettingsResponse | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const load = async () => {
      try {
        setLoading(true);
        setError(null);

        const token = localStorage.getItem('authToken');
        const headers: Record<string, string> = { 'Content-Type': 'application/json' };
        if (token) headers.Authorization = `Bearer ${token}`;

        const res = await fetch('/api/admin/settings', {
          headers,
          credentials: 'include',
        });
        if (!res.ok) {
          const body = await res.json().catch(() => ({}));
          throw new Error(body?.error || `Failed to load settings (${res.status})`);
        }

        const json = (await res.json()) as AdminSettingsResponse;
        setData(json);
      } catch (e) {
        setError(e instanceof Error ? e.message : 'Failed to load settings');
      } finally {
        setLoading(false);
      }
    };

    void load();
  }, []);

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-2xl font-bold text-white">System Settings</h1>
        <p className="text-slate-400">
          Configuration status for API keys and services. (Values are managed via server environment variables.)
        </p>
      </div>

      {error && (
        <div className="rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
          {error}
        </div>
      )}

      <div className="grid gap-6 md:grid-cols-2">
        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">API Keys</h3>
          <div className="mt-4 space-y-3 text-sm">
            <Row label="Grok API Key" value={loading ? '—' : data?.keys.grokConfigured ? 'Configured' : 'Missing'} ok={Boolean(data?.keys.grokConfigured)} />
            <Row label="BallDontLie API Key" value={loading ? '—' : data?.keys.ballDontLieConfigured ? 'Configured' : 'Missing'} ok={Boolean(data?.keys.ballDontLieConfigured)} />
            <Row label="Stripe Keys" value={loading ? '—' : data?.keys.stripeConfigured ? 'Configured' : 'Missing'} ok={Boolean(data?.keys.stripeConfigured)} />
            <Row label="SMTP" value={loading ? '—' : data?.keys.smtpConfigured ? 'Configured' : 'Missing'} ok={Boolean(data?.keys.smtpConfigured)} />
            <Row label="RAG API Key" value={loading ? '—' : data?.keys.ragApiKeyConfigured ? 'Configured' : 'Missing'} ok={Boolean(data?.keys.ragApiKeyConfigured)} />
          </div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Services</h3>
          <div className="mt-4 space-y-3 text-sm">
            <Row
              label="Python backtest service"
              value={loading ? '—' : data?.services.python.healthy ? 'Healthy' : 'Unreachable'}
              ok={Boolean(data?.services.python.healthy)}
              hint={data?.services.python.url}
            />
            <Row
              label="RAG service"
              value={loading ? '—' : data?.services.rag.healthy ? 'Healthy' : 'Unreachable'}
              ok={Boolean(data?.services.rag.healthy)}
              hint={data?.services.rag.url}
            />
          </div>
        </div>
      </div>

      <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
        <h3 className="font-semibold text-white">Environment</h3>
        <div className="mt-4 grid gap-3 md:grid-cols-3 text-sm">
          <Row label="NODE_ENV" value={loading ? '—' : data?.env.nodeEnv || '—'} ok />
          <Row label="Base URL" value={loading ? '—' : data?.env.baseUrl || '—'} ok />
        </div>
        <p className="mt-4 text-xs text-slate-500">
          To change these settings, update your server environment (e.g. <span className="text-slate-300">.env</span>) and restart the services.
        </p>
      </div>
    </div>
  );
}

function Row({
  label,
  value,
  ok,
  hint,
}: {
  label: string;
  value: string;
  ok: boolean;
  hint?: string;
}) {
  return (
    <div className="flex items-start justify-between gap-4 rounded-lg bg-slate-950/40 px-3 py-2 ring-1 ring-white/10">
      <div className="min-w-0">
        <div className="text-slate-200">{label}</div>
        {hint ? <div className="mt-0.5 truncate text-xs text-slate-500">{hint}</div> : null}
      </div>
      <div className={`shrink-0 text-sm font-semibold ${ok ? 'text-emerald-400' : 'text-rose-400'}`}>
        {value}
      </div>
    </div>
  );
}
