'use client';

import { useEffect, useMemo, useState } from 'react';

interface AnalyticsData {
  totalUsers: number;
  activeUsers: number;
  totalStrategies: number;
  totalRuns: number;
  avgResponseTime: number;
  errorRate: number; // 0..1
  marketUsage: Record<string, number>; // % distribution
  dailyRuns: { date: string; count: number }[];
}

export function AdminAnalyticsPage() {
  const [analytics, setAnalytics] = useState<AnalyticsData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const load = async () => {
      try {
        setLoading(true);
        setError(null);

        const token = localStorage.getItem('authToken');
        const headers: Record<string, string> = {
          'Content-Type': 'application/json',
        };
        if (token) headers.Authorization = `Bearer ${token}`;

        const res = await fetch('/api/admin/analytics', {
          headers,
          credentials: 'include',
        });

        if (!res.ok) {
          const data = await res.json().catch(() => ({}));
          throw new Error(data?.error || `Failed to load analytics (${res.status})`);
        }

        const data = (await res.json()) as AnalyticsData;
        setAnalytics(data);
      } catch (e) {
        setError(e instanceof Error ? e.message : 'Failed to load analytics');
      } finally {
        setLoading(false);
      }
    };

    void load();
  }, []);

  const maxDaily = useMemo(() => {
    const values = analytics?.dailyRuns?.map((d) => d.count) || [];
    return Math.max(1, ...values);
  }, [analytics]);

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-2xl font-bold text-white">Analytics Dashboard</h1>
        <p className="text-slate-400">Live platform analytics (users, backtests, and performance).</p>
      </div>

      {error && (
        <div className="rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
          {error}
        </div>
      )}

      <div className="grid gap-6 md:grid-cols-4">
        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <div className="flex items-start justify-between">
            <div>
              <p className="text-sm font-medium text-slate-400">Total Users</p>
              <h3 className="mt-2 text-3xl font-bold text-white">
                {loading ? '—' : (analytics?.totalUsers ?? 0)}
              </h3>
            </div>
            <div className="rounded-lg bg-white/5 p-2 text-slate-300">
              <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z" />
              </svg>
            </div>
          </div>
          <div className="mt-4 text-xs text-slate-500">All-time</div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <div className="flex items-start justify-between">
            <div>
              <p className="text-sm font-medium text-slate-400">Active Users</p>
              <h3 className="mt-2 text-3xl font-bold text-white">
                {loading ? '—' : (analytics?.activeUsers ?? 0)}
              </h3>
            </div>
            <div className="rounded-lg bg-white/5 p-2 text-slate-300">
              <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
              </svg>
            </div>
          </div>
          <div className="mt-4 text-xs text-slate-500">Last 7 days</div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <div className="flex items-start justify-between">
            <div>
              <p className="text-sm font-medium text-slate-400">Backtests Run</p>
              <h3 className="mt-2 text-3xl font-bold text-white">
                {loading ? '—' : (analytics?.totalRuns ?? 0).toLocaleString()}
              </h3>
            </div>
            <div className="rounded-lg bg-white/5 p-2 text-slate-300">
              <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
              </svg>
            </div>
          </div>
          <div className="mt-4 text-xs text-slate-500">All-time</div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <div className="flex items-start justify-between">
            <div>
              <p className="text-sm font-medium text-slate-400">Avg API Response</p>
              <h3 className="mt-2 text-3xl font-bold text-white">
                {loading ? '—' : `${analytics?.avgResponseTime ?? 0}ms`}
              </h3>
            </div>
            <div className="rounded-lg bg-white/5 p-2 text-slate-300">
              <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <circle cx={12} cy={12} r={10} />
                <path d="M12 6v6l4 2" />
              </svg>
            </div>
          </div>
          <div className="mt-4 text-xs text-slate-500">
            Error rate: {loading ? '—' : `${((analytics?.errorRate ?? 0) * 100).toFixed(2)}%`} (last 24h)
          </div>
        </div>
      </div>

      <div className="grid gap-6 md:grid-cols-2">
        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Market Usage</h3>
          <div className="mt-6 space-y-4">
            {loading ? (
              <div className="text-sm text-slate-400">Loading…</div>
            ) : (
              Object.entries(analytics?.marketUsage || {}).map(([market, percentage]) => (
              <div key={market} className="space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-slate-300 capitalize">{market}</span>
                  <span className="text-white font-medium">{percentage}%</span>
                </div>
                <div className="h-2 bg-slate-700 rounded-full overflow-hidden">
                  <div
                    className="h-full bg-gradient-to-r from-blue-500 to-blue-400 rounded-full transition-all duration-300"
                    style={{ width: `${percentage}%` }}
                  />
                </div>
              </div>
              ))
            )}
          </div>
        </div>

        <div className="rounded-2xl border border-white/10 bg-slate-900/50 p-6">
          <h3 className="font-semibold text-white">Daily Backtests</h3>
          <div className="mt-6">
            <div className="flex items-end justify-between h-32 gap-1">
              {(analytics?.dailyRuns || []).map((day) => (
                <div key={day.date} className="flex flex-col items-center flex-1">
                  <div
                    className="w-full bg-gradient-to-t from-blue-600 to-blue-400 rounded-t transition-all duration-300 hover:from-blue-500 hover:to-blue-300"
                    style={{ height: `${(day.count / maxDaily) * 100}%` }}
                  />
                  <span className="text-xs text-slate-500 mt-2">
                    {new Date(day.date).toLocaleDateString('en-US', { weekday: 'short' })}
                  </span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
