/**
 * Trial Status Component
 * Shows remaining trial time for FREE users
 */

'use client';

import { useState, useEffect } from 'react';
import { Clock, AlertTriangle, CheckCircle } from 'lucide-react';

interface TrialStatusProps {
  trialEndsAt?: Date;
  subscriptionTier: string;
  className?: string;
}

export default function TrialStatus({ trialEndsAt, subscriptionTier, className }: TrialStatusProps) {
  const [timeLeft, setTimeLeft] = useState<string>('');

  useEffect(() => {
    if (!trialEndsAt || subscriptionTier !== 'FREE') return;

    const updateTimeLeft = () => {
      const now = new Date();
      const end = new Date(trialEndsAt);
      const diffMs = end.getTime() - now.getTime();

      if (diffMs <= 0) {
        setTimeLeft('expired');
        return;
      }

      const days = Math.floor(diffMs / (1000 * 60 * 60 * 24));
      const hours = Math.floor((diffMs % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
      const minutes = Math.floor((diffMs % (1000 * 60 * 60)) / (1000 * 60));

      if (days > 0) {
        setTimeLeft(`${days} day${days > 1 ? 's' : ''} left`);
      } else if (hours > 0) {
        setTimeLeft(`${hours} hour${hours > 1 ? 's' : ''} left`);
      } else {
        setTimeLeft(`${minutes} minute${minutes > 1 ? 's' : ''} left`);
      }
    };

    updateTimeLeft();
    const interval = setInterval(updateTimeLeft, 60000); // Update every minute

    return () => clearInterval(interval);
  }, [trialEndsAt, subscriptionTier]);

  // Don't show if not on FREE tier or no trial
  if (subscriptionTier !== 'FREE' || !trialEndsAt) {
    return null;
  }

  const now = new Date();
  const end = new Date(trialEndsAt);
  const diffMs = end.getTime() - now.getTime();
  const isExpired = diffMs <= 0;
  const isUrgent = diffMs <= 24 * 60 * 60 * 1000; // Less than 24 hours

  if (isExpired) {
    return (
      <div className={`bg-red-50 border border-red-200 rounded-lg p-4 ${className}`}>
        <div className="flex items-center">
          <AlertTriangle className="h-5 w-5 text-red-500 mr-2" />
          <div>
            <h3 className="text-sm font-medium text-red-800">Trial Expired</h3>
            <p className="text-sm text-red-700">
              Your free trial has ended. Upgrade to Pro to continue using premium features.
            </p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className={`bg-blue-50 border border-blue-200 rounded-lg p-4 ${className}`}>
      <div className="flex items-center">
        <Clock className="h-5 w-5 text-blue-500 mr-2" />
        <div>
          <h3 className="text-sm font-medium text-blue-800">
            Trial Active {isUrgent ? '- Billing Starts Soon!' : ''}
          </h3>
          <p className="text-sm text-blue-700">
            {timeLeft} until your recurring $20/month subscription begins.
            {isUrgent && ' Cancel anytime before automatic billing starts.'}
          </p>
        </div>
        {isUrgent && (
          <AlertTriangle className="h-4 w-4 text-orange-500 ml-2" />
        )}
      </div>
    </div>
  );
}
