import React, { useMemo } from 'react';
import {
  LineChart,
  Line,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  ReferenceLine,
  ScatterChart,
  Scatter,
  Cell
} from 'recharts';
import { TrendingUp, TrendingDown, Target, AlertTriangle } from 'lucide-react';

interface TradeVisualization {
  id: string;
  index: number;
  action: 'BUY' | 'SELL';
  outcome: 'win' | 'loss' | 'unknown';
  profit: number;
  stake: number;
  entry: {
    bar_index: number;
    price: number;
    timestamp: string;
  };
  exit: {
    bar_index: number;
    price: number;
    timestamp: string;
    reason: string;
  };
  levels: {
    stop_loss: number;
    take_profit: number;
  };
  context: any;
  duration_bars: number;
}

interface ChartVisualization {
  price_data: Array<{
    index: number;
    timestamp: string;
    open: number;
    high: number;
    close: number;
    low: number;
    volume: number;
  }>;
  trades: TradeVisualization[];
  indicators: any;
  summary: {
    total_trades: number;
    price_bars: number;
    market: string;
    timeframe: string;
    last_price: number;
  };
}

interface TradeChartProps {
  visualizationData: ChartVisualization;
  height?: number;
}

export const TradeChart: React.FC<TradeChartProps> = ({
  visualizationData,
  height = 400
}) => {
  const chartData = useMemo(() => {
    if (!visualizationData?.price_data) return [];

    return visualizationData.price_data.map((bar, index) => {
      // Find trades for this bar
      const entryTrades = visualizationData.trades.filter(t => t.entry.bar_index === index);
      const exitTrades = visualizationData.trades.filter(t => t.exit.bar_index === index);

      return {
        ...bar,
        index,
        timestamp: new Date(bar.timestamp).toLocaleDateString(),
        entries: entryTrades,
        exits: exitTrades,
        // Add visual markers
        hasEntry: entryTrades.length > 0,
        hasExit: exitTrades.length > 0,
        entryType: entryTrades.length > 0 ? entryTrades[0].action : null,
        exitType: exitTrades.length > 0 ? exitTrades[0].outcome : null
      };
    });
  }, [visualizationData]);

  const CustomTooltip = ({ active, payload, label }: any) => {
    if (active && payload && payload.length) {
      const data = payload[0].payload;
      return (
        <div className="bg-slate-800 border border-slate-600 rounded-lg p-3 shadow-lg">
          <p className="text-slate-200 font-semibold">{`Bar ${data.index} - ${data.timestamp}`}</p>
          <p className="text-slate-300">{`Open: $${data.open.toFixed(4)}`}</p>
          <p className="text-slate-300">{`High: $${data.high.toFixed(4)}`}</p>
          <p className="text-slate-300">{`Low: $${data.low.toFixed(4)}`}</p>
          <p className="text-slate-300 font-semibold">{`Close: $${data.close.toFixed(4)}`}</p>

          {data.entries.length > 0 && (
            <div className="mt-2 pt-2 border-t border-slate-600">
              <p className="text-green-400 font-semibold">📈 Trade Entries:</p>
              {data.entries.map((trade: TradeVisualization) => (
                <div key={trade.id} className="text-sm">
                  <span className={trade.action === 'BUY' ? 'text-green-400' : 'text-red-400'}>
                    {trade.action}
                  </span>
                  {` @ $${trade.entry.price.toFixed(4)} ($${trade.profit > 0 ? '+' : ''}${trade.profit.toFixed(2)})`}
                </div>
              ))}
            </div>
          )}

          {data.exits.length > 0 && (
            <div className="mt-2 pt-2 border-t border-slate-600">
              <p className="text-blue-400 font-semibold">📉 Trade Exits:</p>
              {data.exits.map((trade: TradeVisualization) => (
                <div key={trade.id} className="text-sm">
                  <span className={trade.outcome === 'win' ? 'text-green-400' : 'text-red-400'}>
                    {trade.outcome.toUpperCase()}
                  </span>
                  {` @ $${trade.exit.price.toFixed(4)} (${trade.exit.reason})`}
                </div>
              ))}
            </div>
          )}
        </div>
      );
    }
    return null;
  };

  const renderTradeMarkers = () => {
    return visualizationData?.trades?.map((trade) => {
      const entryBar = chartData.find(d => d.index === trade.entry.bar_index);
      const exitBar = chartData.find(d => d.index === trade.exit.bar_index);

      if (!entryBar || !exitBar) return null;

      // Entry marker
      const entryY = trade.action === 'BUY' ?
        entryBar.low - (entryBar.high - entryBar.low) * 0.1 :
        entryBar.high + (entryBar.high - entryBar.low) * 0.1;

      // Exit marker
      const exitY = trade.outcome === 'win' ?
        exitBar.high + (exitBar.high - exitBar.low) * 0.1 :
        exitBar.low - (exitBar.high - exitBar.low) * 0.1;

      return (
        <g key={trade.id}>
          {/* Entry point */}
          <circle
            cx={entryBar.index * (800 / chartData.length)}
            cy={entryY}
            r="4"
            fill={trade.action === 'BUY' ? '#10B981' : '#EF4444'}
            stroke="#FFFFFF"
            strokeWidth="2"
          />
          <text
            x={entryBar.index * (800 / chartData.length)}
            y={entryY - 8}
            textAnchor="middle"
            className="text-xs font-semibold"
            fill={trade.action === 'BUY' ? '#10B981' : '#EF4444'}
          >
            {trade.action === 'BUY' ? '↗' : '↘'}
          </text>

          {/* Exit point */}
          <circle
            cx={exitBar.index * (800 / chartData.length)}
            cy={exitY}
            r="4"
            fill={trade.outcome === 'win' ? '#10B981' : '#EF4444'}
            stroke="#FFFFFF"
            strokeWidth="2"
          />

          {/* Stop Loss line */}
          <line
            x1={entryBar.index * (800 / chartData.length)}
            y1={trade.levels.stop_loss}
            x2={exitBar.index * (800 / chartData.length)}
            y2={trade.levels.stop_loss}
            stroke="#EF4444"
            strokeWidth="1"
            strokeDasharray="3,3"
            opacity="0.6"
          />

          {/* Take Profit line */}
          <line
            x1={entryBar.index * (800 / chartData.length)}
            y1={trade.levels.take_profit}
            x2={exitBar.index * (800 / chartData.length)}
            y2={trade.levels.take_profit}
            stroke="#10B981"
            strokeWidth="1"
            strokeDasharray="3,3"
            opacity="0.6"
          />
        </g>
      );
    });
  };

  if (!visualizationData?.price_data?.length) {
    return (
      <div className="bg-slate-800 rounded-lg p-6 text-center">
        <p className="text-slate-400">No chart data available for visualization</p>
      </div>
    );
  }

  return (
    <div className="bg-slate-800 rounded-lg p-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-xl font-semibold text-white flex items-center gap-2">
          <TrendingUp className="h-5 w-5 text-blue-400" />
          Trade Chart Visualization
        </h3>
        <div className="flex items-center gap-4 text-sm text-slate-400">
          <span>{visualizationData.summary?.total_trades || 0} trades</span>
          <span>{visualizationData.summary?.price_bars || 0} bars</span>
          <span>{visualizationData.summary?.market?.toUpperCase()}</span>
        </div>
      </div>

      {/* Legend */}
      <div className="flex flex-wrap gap-4 mb-4 text-sm">
        <div className="flex items-center gap-2">
          <div className="w-3 h-3 bg-green-500 rounded-full"></div>
          <span className="text-slate-300">Buy Entry</span>
        </div>
        <div className="flex items-center gap-2">
          <div className="w-3 h-3 bg-red-500 rounded-full"></div>
          <span className="text-slate-300">Sell Entry</span>
        </div>
        <div className="flex items-center gap-2">
          <div className="w-3 h-0.5 bg-green-500 opacity-60"></div>
          <span className="text-slate-300">Take Profit</span>
        </div>
        <div className="flex items-center gap-2">
          <div className="w-3 h-0.5 bg-red-500 opacity-60"></div>
          <span className="text-slate-300">Stop Loss</span>
        </div>
      </div>

      {/* Chart */}
      <ResponsiveContainer width="100%" height={height}>
        <LineChart data={chartData} margin={{ top: 5, right: 30, left: 20, bottom: 5 }}>
          <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
          <XAxis
            dataKey="index"
            stroke="#9CA3AF"
            fontSize={12}
            tickFormatter={(value) => `${value}`}
          />
          <YAxis
            stroke="#9CA3AF"
            fontSize={12}
            domain={['dataMin - 0.001', 'dataMax + 0.001']}
            tickFormatter={(value) => `$${value.toFixed(4)}`}
          />
          <Tooltip content={<CustomTooltip />} />

          {/* Price line */}
          <Line
            type="monotone"
            dataKey="close"
            stroke="#3B82F6"
            strokeWidth={2}
            dot={false}
            activeDot={{ r: 4, fill: '#3B82F6' }}
          />

          {/* Entry markers */}
          <Scatter
            data={chartData.filter(d => d.hasEntry)}
            fill="#10B981"
          />

          {/* Exit markers */}
          <Scatter
            data={chartData.filter(d => d.hasExit)}
            fill="#EF4444"
          />
        </LineChart>
      </ResponsiveContainer>

      {/* Trade Summary */}
      <div className="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-slate-700 rounded-lg p-3">
          <div className="text-sm text-slate-400">Winning Trades</div>
          <div className="text-lg font-semibold text-green-400">
            {visualizationData.trades?.filter(t => t.outcome === 'win').length || 0}
          </div>
        </div>
        <div className="bg-slate-700 rounded-lg p-3">
          <div className="text-sm text-slate-400">Losing Trades</div>
          <div className="text-lg font-semibold text-red-400">
            {visualizationData.trades?.filter(t => t.outcome === 'loss').length || 0}
          </div>
        </div>
        <div className="bg-slate-700 rounded-lg p-3">
          <div className="text-sm text-slate-400">Average Duration</div>
          <div className="text-lg font-semibold text-blue-400">
            {visualizationData.trades?.length ?
              Math.round(visualizationData.trades.reduce((sum, t) => sum + t.duration_bars, 0) / visualizationData.trades.length)
              : 0} bars
          </div>
        </div>
      </div>
    </div>
  );
};
