'use client';

import Link from 'next/link';
import { useMemo, useState } from 'react';
import { AlertTriangle, HelpCircle, TrendingUp, History, BarChart3 } from 'lucide-react';

import { ResultsChart } from '@/components/ResultsChart';
import type { StrategyResponsePayload } from '@/lib/types';

const promptShortcuts = [
  {
    label: 'Home Favorites',
    prompt:
      'Bet on home favorites with a spread of 5 or more points in the 2024 NBA season',
    domain: 'sports' as const,
  },
  {
    label: 'Underdog Bounce',
    prompt:
      'Bet on teams that lost their last 2 games if their point differential was under -5.',
    domain: 'sports' as const,
  },
  {
    label: 'Road Warriors',
    prompt:
      'Bet on away teams with a winning record (above .600) in 2024 NFL season',
    domain: 'sports' as const,
  },
];

const domainCopy = {
  sports: 'Real sportsbook odds + game results',
} as const;

export function StrategyPlayground() {
  const [prompt, setPrompt] = useState(promptShortcuts[0].prompt);
  const domain = 'sports' as const;
  const [result, setResult] = useState<StrategyResponsePayload | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [saveStrategy, setSaveStrategy] = useState(false);
  const [saveBacktest, setSaveBacktest] = useState(true);
  const [strategyName, setStrategyName] = useState('');
  const [isPublic, setIsPublic] = useState(false);
  const [saveStatus, setSaveStatus] = useState<string | null>(null);
  const [backtestSaveStatus, setBacktestSaveStatus] = useState<{
    message: string;
    backtestId?: string;
  } | null>(null);

  const headline = useMemo(() => {
    if (!result) {
      return 'Plug in a natural-language rule and hit run.';
    }

    const { summary } = result;
    if (!summary.totalTrades) {
      return 'No trades qualified. Try loosening the rule.';
    }

    return `Win rate ${Math.round(summary.winRate * 100)}% across ${summary.totalTrades} trades`;
  }, [result]);

  async function runSimulation() {
    setLoading(true);
    setError(null);
    setSaveStatus(null);
    setBacktestSaveStatus(null);

    try {
      const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
      const response = await fetch('/api/strategies', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
        body: JSON.stringify({
          prompt,
          domain,
          saveStrategy,
          saveBacktest,
          strategyName: strategyName || undefined,
          isPublic,
        }),
      });

      if (!response.ok) {
        const payload = await response.json();
        throw new Error(payload.error ?? 'Simulation failed');
      }

      const payload = (await response.json()) as StrategyResponsePayload;
      setResult(payload);

      if ((payload as any).savedStrategy?.id) {
        setSaveStatus(`Saved strategy: ${(payload as any).savedStrategy.name || 'Untitled'}`);
      } else if (saveStrategy) {
        setSaveStatus('Strategy not saved (sign in required).');
      }

      const savedBacktestId = (payload as any).savedBacktest?.id as string | undefined;
      if (savedBacktestId) {
        setBacktestSaveStatus({
          message: 'Backtest saved — your bet log is now available in your profile.',
          backtestId: savedBacktestId,
        });
      } else if (saveBacktest && !token) {
        setBacktestSaveStatus({
          message: 'Sign in to save this bet log to your profile.',
        });
      }
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Unknown error');
    } finally {
      setLoading(false);
    }
  }

  return (
    <section className="rounded-3xl border border-white/10 bg-slate-900/70 p-6 shadow-xl shadow-blue-500/10 backdrop-blur">
      {/* Header with clear purpose */}
      <div className="flex flex-wrap justify-between gap-4 mb-4">
        <div>
          <div className="flex items-center gap-2">
            <BarChart3 className="h-5 w-5 text-blue-400" />
            <p className="text-sm uppercase tracking-widest text-blue-300">
              Strategy Playground
            </p>
            <span className="bg-purple-500/20 text-purple-300 text-xs px-2 py-0.5 rounded-full font-medium">
              Backtesting Only
            </span>
          </div>
          <p className="text-lg text-slate-100 mt-1">{headline}</p>
        </div>
        <div className="text-right text-sm text-slate-400">
          <p>Dataset: {domainCopy[domain]}</p>
          <p>
            {result ? `Rows scanned: ${result.datasetSize}` : 'Rows scanned: –'}
          </p>
        </div>
      </div>

      {/* Beginner-friendly explanation */}
      <div className="bg-gradient-to-r from-blue-500/10 to-purple-500/10 border border-blue-500/20 rounded-xl p-4 mb-6">
        <div className="flex items-start gap-3">
          <HelpCircle className="h-5 w-5 text-blue-400 flex-shrink-0 mt-0.5" />
          <div>
            <h4 className="font-semibold text-white text-sm mb-1">How This Works</h4>
            <p className="text-slate-300 text-sm">
              Describe a betting strategy in plain English, and we'll test it against <strong>real historical data</strong> to show you how it would have performed. This is for <strong>research and education only</strong> – past results don't guarantee future outcomes.
            </p>
          </div>
        </div>
      </div>

      <div className="mt-4 grid gap-4 md:grid-cols-[2fr,1fr]">
        <div className="space-y-4">
          <label className="block text-sm font-semibold text-slate-200">
            <div className="flex items-center gap-2 mb-2">
              <History className="h-4 w-4 text-slate-400" />
              Describe your strategy to backtest
            </div>
            <textarea
              value={prompt}
              onChange={(event) => setPrompt(event.target.value)}
              className="mt-1 w-full rounded-2xl border border-white/10 bg-slate-950/40 p-4 text-sm text-slate-100 outline-none transition focus:border-blue-400"
              rows={4}
              placeholder="Example: Bet on NBA home favorites with a spread of 5+ points in 2024 season"
            />
          </label>

          {/* Quick Start Presets */}
          <div>
            <p className="text-xs text-slate-400 mb-2">🚀 Quick start – click to try:</p>
            <div className="flex flex-wrap gap-2">
              {promptShortcuts.map((preset) => (
                <button
                  key={preset.label}
                  type="button"
                  className={`rounded-full border px-4 py-1.5 text-xs font-semibold transition ${prompt === preset.prompt && domain === preset.domain
                    ? 'border-blue-400 bg-blue-950 text-blue-100'
                    : 'border-white/10 text-slate-300 hover:border-blue-400 hover:bg-white/5'
                    }`}
                  onClick={() => {
                    setPrompt(preset.prompt);
                  }}
                >
                  {preset.label}
                </button>
              ))}
            </div>
          </div>

          <div className="rounded-2xl border border-white/10 bg-slate-950/40 p-4">
            <div className="flex items-center justify-between gap-4">
              <label className="flex items-center gap-2 text-sm text-slate-200">
                <input
                  type="checkbox"
                  checked={saveStrategy}
                  onChange={(e) => setSaveStrategy(e.target.checked)}
                  className="rounded border-white/20 bg-slate-900"
                />
                Save to Strategy Library
              </label>
              <label className="flex items-center gap-2 text-sm text-slate-200">
                <input
                  type="checkbox"
                  checked={isPublic}
                  onChange={(e) => setIsPublic(e.target.checked)}
                  className="rounded border-white/20 bg-slate-900"
                  disabled={!saveStrategy}
                />
                Make public
              </label>
            </div>

            <div className="mt-3 flex items-center justify-between gap-4">
              <label className="flex items-center gap-2 text-sm text-slate-200">
                <input
                  type="checkbox"
                  checked={saveBacktest}
                  onChange={(e) => setSaveBacktest(e.target.checked)}
                  className="rounded border-white/20 bg-slate-900"
                />
                Save bet log to Profile / History
              </label>
              <span className="text-xs text-slate-500">Recommended</span>
            </div>

            {saveStrategy && (
              <input
                value={strategyName}
                onChange={(e) => setStrategyName(e.target.value)}
                placeholder="Strategy name (optional)"
                className="mt-3 w-full rounded-xl border border-white/10 bg-slate-900/60 p-3 text-sm text-slate-100 placeholder-slate-500"
              />
            )}

            {saveStatus && (
              <p className="mt-3 text-xs text-emerald-300">{saveStatus}</p>
            )}

            {backtestSaveStatus && (
              <div className="mt-2 text-xs text-blue-200">
                <div>{backtestSaveStatus.message}</div>
                {backtestSaveStatus.backtestId && (
                  <Link
                    href={`/backtest?id=${encodeURIComponent(backtestSaveStatus.backtestId)}`}
                    className="inline-flex mt-1 text-blue-400 hover:text-blue-300 underline"
                  >
                    View saved bet log →
                  </Link>
                )}
              </div>
            )}
          </div>
        </div>

        <div className="space-y-4 rounded-2xl border border-white/10 bg-slate-950/40 p-4">
          <button
            type="button"
            onClick={runSimulation}
            disabled={loading}
            className="w-full rounded-2xl bg-gradient-to-r from-sky-500 to-blue-600 py-3 text-sm font-semibold text-white disabled:cursor-not-allowed disabled:opacity-60 flex items-center justify-center gap-2"
          >
            <TrendingUp className="h-4 w-4" />
            {loading ? 'Running Backtest...' : 'Run Backtest'}
          </button>

          {error ? (
            <p className="text-sm text-rose-300">{error}</p>
          ) : (
            <div className="space-y-2">
              <p className="text-xs text-slate-400">
                Test your strategy against historical sports betting data with real sportsbook odds.
              </p>
              <div className="flex items-start gap-2 text-xs text-amber-400/80 bg-amber-500/10 rounded-lg p-2">
                <AlertTriangle className="h-3 w-3 flex-shrink-0 mt-0.5" />
                <span>Results show historical performance only. Not financial advice.</span>
              </div>
            </div>
          )}
        </div>
      </div>

      {result && (
        <div className="mt-6 grid gap-4 md:grid-cols-3">
          <StatCard
            heading="Win rate"
            value={`${Math.round(result.summary.winRate * 100)}%`}
            footnote={`${result.summary.wins}/${result.summary.totalTrades} winning trades`}
            variant={result.summary.winRate >= 0.5 ? 'positive' : 'negative'}
          />
          <StatCard
            heading="Net return"
            value={`${result.summary.netProfit >= 0 ? '+' : '-'}$${Math.abs(result.summary.netProfit * 100).toFixed(0)}`}
            footnote={`${result.summary.netProfit.toFixed(2)}R • Avg $${Math.abs(result.summary.avgReturn * 100).toFixed(0)}/trade`}
            variant={result.summary.netProfit >= 0 ? 'positive' : 'negative'}
          />
          <StatCard
            heading="RAG insight"
            value={result.ragInsight?.title ?? 'No match'}
            footnote={
              result.ragInsight?.summary ??
              'Drop your corpus into /rag to enable deep recall.'
            }
            variant="neutral"
          />

          <div className="md:col-span-3">
            <ResultsChart trades={result.summary.trades} />
          </div>

          {result.summary.trades.slice(0, 3).map((trade) => (
            <div
              key={trade.id}
              className="rounded-2xl border border-white/5 bg-slate-950/30 p-4"
            >
              <p className="text-sm font-semibold text-slate-100">
                {trade.label}
              </p>
              <p className="text-xs text-slate-400">{trade.date}</p>
              <p
                className={`mt-2 text-lg font-bold ${trade.outcome === 'win' ? 'text-emerald-300' : 'text-rose-300'
                  }`}
              >
                {trade.profit >= 0 ? '+' : '-'}${Math.abs(trade.profit * 100).toFixed(0)} <span className="text-sm font-normal opacity-60">({trade.profit.toFixed(2)}R)</span>
              </p>
              <div className="mt-2 text-xs text-slate-400">
                {Object.entries(trade.context).map(([key, value]) => (
                  <span key={key} className="mr-3">
                    {key}: {value}
                  </span>
                ))}
              </div>
            </div>
          ))}

          {result.externalSnapshot && (
            <div className="rounded-2xl border border-slate-800 bg-slate-950/50 p-4 md:col-span-3">
              <p className="text-sm font-semibold text-slate-200">
                Live Market Snapshot
              </p>
              <div className="mt-3 flex flex-wrap gap-4 text-sm text-slate-300">
                {result.externalSnapshot.map((asset) => (
                  <span
                    key={asset.asset}
                    className="rounded-full border border-white/10 px-3 py-1"
                  >
                    {asset.asset.toUpperCase()}: ${asset.priceUsd.toLocaleString()}
                  </span>
                ))}
              </div>
            </div>
          )}
        </div>
      )}
    </section>
  );
}

function StatCard({
  heading,
  value,
  footnote,
  variant = 'neutral',
}: {
  heading: string;
  value: string;
  footnote: string;
  variant?: 'positive' | 'negative' | 'neutral';
}) {
  const variantStyles = {
    positive: {
      border: 'border-emerald-500/30',
      bg: 'bg-emerald-950/20',
      heading: 'text-emerald-300',
      value: 'text-emerald-400',
      icon: '↑',
    },
    negative: {
      border: 'border-rose-500/30',
      bg: 'bg-rose-950/20',
      heading: 'text-rose-300',
      value: 'text-rose-400',
      icon: '↓',
    },
    neutral: {
      border: 'border-white/5',
      bg: 'bg-slate-950/30',
      heading: 'text-blue-200',
      value: 'text-slate-50',
      icon: '',
    },
  };

  const styles = variantStyles[variant];

  return (
    <div className={`rounded-2xl border ${styles.border} ${styles.bg} p-4`}>
      <p className={`text-xs uppercase tracking-widest ${styles.heading}`}>
        {heading}
      </p>
      <p className={`mt-1 text-2xl font-bold ${styles.value} flex items-center gap-2`}>
        {styles.icon && <span className="text-lg">{styles.icon}</span>}
        {value}
      </p>
      <p className="text-xs text-slate-400">{footnote}</p>
    </div>
  );
}

