'use client';

import { useState, useEffect, useCallback } from 'react';
import Link from 'next/link';
import { Search, Filter, Heart, GitFork, MessageSquare, Star, Code, TrendingUp } from 'lucide-react';

import { cn } from '@/lib/utils';
import type { Strategy, StrategyStats } from '@/lib/strategyLibrary/types';

interface StrategyLibraryProps {
  className?: string;
}

export function StrategyLibrary({ className }: StrategyLibraryProps) {
  const [strategies, setStrategies] = useState<Strategy[]>([]);
  const [stats, setStats] = useState<StrategyStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState('');
  const [sortBy, setSortBy] = useState<string>('newest');
  const [showFilters, setShowFilters] = useState(false);

  const loadStrategies = useCallback(async () => {
    try {
      const response = await fetch(
        '/api/strategies/library?' +
          new URLSearchParams({
            sortBy,
            ...(searchQuery && { searchQuery }),
            domain: 'sports',
          })
      );

      if (response.ok) {
        const data = await response.json();
        setStrategies(data.strategies);
      }
    } catch (error) {
      console.error('Failed to load strategies:', error);
    } finally {
      setLoading(false);
    }
  }, [sortBy, searchQuery]);

  useEffect(() => {
    loadStrategies();
  }, [loadStrategies]);

  useEffect(() => {
    loadStats();
  }, []); // Load stats once on mount

  const loadStats = async () => {
    try {
      const response = await fetch('/api/strategies/library/stats');
      if (response.ok) {
        const data = await response.json();
        setStats(data);
      }
    } catch (error) {
      console.error('Failed to load stats:', error);
    }
  };

  const handleLike = async (strategyId: string) => {
    try {
      const response = await fetch(`/api/strategies/library/${strategyId}/like`, {
        method: 'POST',
      });

      if (response.ok) {
        // Update local state
        setStrategies(prev => prev.map(s =>
          s.id === strategyId ? { ...s, likes: s.likes + 1 } : s
        ));
      }
    } catch (error) {
      console.error('Failed to like strategy:', error);
    }
  };

  const handleFork = async (strategyId: string) => {
    try {
      const response = await fetch(`/api/strategies/library/${strategyId}/fork`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          name: `Fork of ${strategies.find(s => s.id === strategyId)?.name}`,
        }),
      });

      if (response.ok) {
        const forkedStrategy = await response.json();
        setStrategies(prev => [forkedStrategy, ...prev]);
      }
    } catch (error) {
      console.error('Failed to fork strategy:', error);
    }
  };

  const getDomainColor = (domain: string) => {
    const colors = {
      sports: 'bg-green-500',
    };
    return colors[domain as keyof typeof colors] || 'bg-gray-500';
  };

  const getRiskColor = (risk: string) => {
    const colors = {
      low: 'text-green-400',
      medium: 'text-yellow-400',
      high: 'text-red-400',
    };
    return colors[risk as keyof typeof colors] || 'text-gray-400';
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center py-12">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className={cn("space-y-6", className)}>
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-white">Strategy Library</h2>
          <p className="text-slate-400">Discover, share, and improve sports betting strategies</p>
        </div>
        <Link
          href="/strategies"
          className="flex items-center gap-2 bg-blue-600 hover:bg-blue-500 px-4 py-2 rounded-lg text-white font-medium transition"
        >
          Create strategy →
        </Link>
      </div>

      {/* Stats Cards */}
      {stats && (
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <StatCard
            title="Total Strategies"
            value={stats.totalStrategies.toString()}
            icon={Code}
            color="text-blue-400"
          />
          <StatCard
            title="Public Strategies"
            value={stats.publicStrategies.toString()}
            icon={TrendingUp}
            color="text-green-400"
          />
          <StatCard
            title="Active Users"
            value={stats.totalUsers.toString()}
            icon={Heart}
            color="text-red-400"
          />
          <StatCard
            title="Popular Tags"
            value={stats.popularTags.length.toString()}
            icon={Filter}
            color="text-purple-400"
          />
        </div>
      )}

      {/* Search and Filters */}
      <div className="flex flex-col sm:flex-row gap-4">
        <div className="flex-1 relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
          <input
            type="text"
            placeholder="Search strategies..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="w-full pl-10 pr-4 py-2 bg-slate-800 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:outline-none focus:ring-2 focus:ring-blue-500"
          />
        </div>

        <div className="flex gap-2">
          <select
            value={sortBy}
            onChange={(e) => setSortBy(e.target.value)}
            className="px-3 py-2 bg-slate-800 border border-slate-700 rounded-lg text-white focus:outline-none focus:ring-2 focus:ring-blue-500"
          >
            <option value="newest">Newest</option>
            <option value="popular">Most Liked</option>
            <option value="rating">Highest Rated</option>
            <option value="forks">Most Forked</option>
          </select>

          <button
            onClick={() => setShowFilters(!showFilters)}
            className="px-3 py-2 bg-slate-800 border border-slate-700 rounded-lg text-white hover:bg-slate-700 transition"
          >
            <Filter className="h-4 w-4" />
          </button>
        </div>
      </div>

      {/* Strategy Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {strategies.map((strategy) => (
          <StrategyCard
            key={strategy.id}
            strategy={strategy}
            onLike={() => handleLike(strategy.id)}
            onFork={() => handleFork(strategy.id)}
            getDomainColor={getDomainColor}
            getRiskColor={getRiskColor}
          />
        ))}
      </div>

      {strategies.length === 0 && (
        <div className="text-center py-12">
          <Code className="h-12 w-12 text-slate-500 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-slate-300 mb-2">No strategies found</h3>
          <p className="text-slate-500">Try adjusting your search or filters</p>
        </div>
      )}
    </div>
  );
}

function StatCard({ title, value, icon: Icon, color }: {
  title: string;
  value: string;
  icon: any;
  color: string;
}) {
  return (
    <div className="bg-slate-800/50 border border-slate-700 rounded-lg p-4">
      <div className="flex items-center justify-between">
        <div>
          <p className="text-sm text-slate-400">{title}</p>
          <p className="text-2xl font-bold text-white">{value}</p>
        </div>
        <Icon className={`h-8 w-8 ${color}`} />
      </div>
    </div>
  );
}

function StrategyCard({
  strategy,
  onLike,
  onFork,
  getDomainColor,
  getRiskColor
}: {
  strategy: Strategy;
  onLike: () => void;
  onFork: () => void;
  getDomainColor: (domain: string) => string;
  getRiskColor: (risk: string) => string;
}) {
  const avgRating = strategy.ratings.length > 0
    ? strategy.ratings.reduce((sum, r) => sum + r.rating, 0) / strategy.ratings.length
    : 0;

  return (
    <div className="bg-slate-800/50 border border-slate-700 rounded-lg p-6 hover:border-slate-600 transition">
      <div className="flex items-start justify-between mb-3">
        <div className="flex items-center gap-2">
          <div className={`w-3 h-3 rounded-full ${getDomainColor(strategy.domain)}`} />
          <span className="text-xs uppercase font-medium text-slate-400">
            {strategy.domain}
          </span>
        </div>
        <div className="flex items-center gap-1">
          {avgRating > 0 && (
            <>
              <Star className="h-4 w-4 text-yellow-400 fill-current" />
              <span className="text-sm text-slate-300">{avgRating.toFixed(1)}</span>
            </>
          )}
        </div>
      </div>

      <h3 className="text-lg font-semibold text-white mb-2">{strategy.name}</h3>
      <p className="text-sm text-slate-400 mb-4 line-clamp-2">{strategy.description}</p>

      <div className="flex flex-wrap gap-1 mb-4">
        {strategy.tags.slice(0, 3).map((tag) => (
          <span
            key={tag}
            className="px-2 py-1 bg-slate-700 text-xs text-slate-300 rounded"
          >
            {tag}
          </span>
        ))}
      </div>

      <div className="flex items-center justify-between text-sm text-slate-500 mb-4">
        <span>by {strategy.authorName}</span>
        <span className={getRiskColor(strategy.riskLevel)}>
          {strategy.riskLevel} risk
        </span>
      </div>

      <div className="flex items-center justify-between">
        <div className="flex items-center gap-4 text-sm text-slate-400">
          <button
            onClick={onLike}
            className="flex items-center gap-1 hover:text-red-400 transition"
          >
            <Heart className="h-4 w-4" />
            {strategy.likes}
          </button>

          <div className="flex items-center gap-1">
            <GitFork className="h-4 w-4" />
            {strategy.forks}
          </div>

          <div className="flex items-center gap-1">
            <MessageSquare className="h-4 w-4" />
            {strategy.comments.length}
          </div>
        </div>

        <button
          onClick={onFork}
          className="bg-blue-600 hover:bg-blue-500 px-3 py-1 rounded text-sm text-white transition"
        >
          Fork
        </button>
      </div>
    </div>
  );
}
