'use client';

import { useState, useEffect, Suspense } from 'react';
import Link from 'next/link';
import { useSearchParams, useRouter } from 'next/navigation';
import {
  Eye,
  EyeOff,
  Mail,
  Lock,
  User,
  CreditCard,
  Check,
  ArrowRight,
  Star,
  Zap,
  Crown
} from 'lucide-react';

const pricingTiers = [
  {
    id: 'FREE',
    name: 'Free Trial',
    price: 1,
    period: 'for 7 days, then $20/month',
    description: 'Start with $1 - Full access for 7 days',
    features: ['Unlimited AI strategies', 'Advanced backtesting', 'All markets access', 'Priority support', '30-day money-back guarantee'],
    popular: true,
    icon: Star
  },
  {
    id: 'PRO',
    name: 'Pro Trader',
    price: 20,
    period: 'per month',
    description: 'For serious traders and bettors',
    features: ['Unlimited strategies', 'Advanced analytics', 'Parlay & Round Robin analysis', 'Priority email support', 'Custom strategy templates'],
    popular: false,
    icon: Zap
  },
  {
    id: 'ELITE',
    name: 'Elite Investor',
    price: 200,
    period: 'per month',
    description: 'For professional funds and institutions',
    features: ['Everything in Pro', 'Unlimited parameter optimization', 'Genetic algorithms', 'Intrabar tick data', 'Dedicated account manager', 'White-label solutions'],
    popular: false,
    icon: Crown
  }
];

function SignupPageContent() {
  const searchParams = useSearchParams();
  const router = useRouter();
  const trialParam = searchParams.get('trial');
  const tierParam = searchParams.get('tier');

  // Map URL parameters to plan IDs
  const getInitialPlan = () => {
    if (tierParam) {
      const tierMap: Record<string, string> = {
        'free': 'FREE',
        'pro': 'PRO',
        'elite': 'ELITE'
      };
      return tierMap[tierParam.toLowerCase()] || 'FREE';
    }
    if (trialParam === 'true') {
      return 'FREE';
    }
    return 'FREE';
  };

  const [step, setStep] = useState(1);
  const [selectedPlan, setSelectedPlan] = useState(getInitialPlan());

  // All plans require payment collection
  const requiresPayment = true;

  // Account details
  const [firstName, setFirstName] = useState('');
  const [lastName, setLastName] = useState('');
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [confirmPassword, setConfirmPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);

  // Payment details
  const [cardNumber, setCardNumber] = useState('');
  const [expiryDate, setExpiryDate] = useState('');
  const [cvv, setCvv] = useState('');
  const [cardName, setCardName] = useState('');

  const [isLoading, setIsLoading] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});

  // Track if user has interacted with form to avoid showing errors on initial load
  const [hasInteracted, setHasInteracted] = useState(false);

  const validateStep1 = () => {
    const newErrors: Record<string, string> = {};

    if (!firstName.trim()) newErrors.firstName = 'First name is required';
    if (!lastName.trim()) newErrors.lastName = 'Last name is required';
    if (!email.trim()) newErrors.email = 'Email is required';
    else if (!/\S+@\S+\.\S+/.test(email)) newErrors.email = 'Email is invalid';
    if (!password) newErrors.password = 'Password is required';
    else if (password.length < 8) newErrors.password = 'Password must be at least 8 characters';
    if (password !== confirmPassword) newErrors.confirmPassword = 'Passwords do not match';

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const validateStep2 = () => {
    const newErrors: Record<string, string> = {};

    if (!cardNumber.replace(/\s/g, '')) newErrors.cardNumber = 'Card number is required';
    if (!expiryDate) newErrors.expiryDate = 'Expiry date is required';
    if (!cvv) newErrors.cvv = 'CVV is required';
    if (!cardName.trim()) newErrors.cardName = 'Name on card is required';

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleStep1Submit = (e: React.FormEvent) => {
    e.preventDefault();
    setHasInteracted(true);
    if (validateStep1()) {
      console.log('DEBUG: Selected Plan:', selectedPlan);
      if (selectedPlan === 'FREE') {
        handleFinalSubmit();
      } else {
        setStep(2);
      }
    }
  };

  const handleStep2Submit = (e: React.FormEvent) => {
    e.preventDefault();
    setHasInteracted(true);
    if (validateStep2()) {
      handleFinalSubmit();
    }
  };

  const handleFinalSubmit = async () => {
    setIsLoading(true);

    try {
      // Create account via API
      const signupResponse = await fetch('/api/auth/signup', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          email,
          password,
          firstName,
          lastName,
          subscriptionTier: selectedPlan.toUpperCase(),
        }),
      });

      if (!signupResponse.ok) {
        const errorData = await signupResponse.json();
        throw new Error(errorData.error || 'Signup failed');
      }

      const signupData = await signupResponse.json();

      // If user selected a paid plan, create subscription
      if (selectedPlan !== 'free' && selectedPlan !== 'FREE') {
        try {
          const subscriptionResponse = await fetch('/api/billing/create-subscription', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Authorization': `Bearer ${signupData.token}`,
            },
            body: JSON.stringify({
              tier: selectedPlan.toUpperCase(),
            }),
          });

          if (!subscriptionResponse.ok) {
            // Subscription creation failed, but account was created
            console.warn('Subscription creation failed, but account was created');
          }
        } catch (subscriptionError) {
          console.warn('Subscription creation failed:', subscriptionError);
          // Continue to welcome page even if subscription fails
        }
      }

      // Redirect to welcome page
      window.location.href = '/welcome';

    } catch (error) {
      console.error('Signup error:', error);
      setErrors({ general: error instanceof Error ? error.message : 'Signup failed. Please try again.' });
    } finally {
      setIsLoading(false);
    }
  };

  const formatCardNumber = (value: string) => {
    const v = value.replace(/\s+/g, '').replace(/[^0-9]/gi, '');
    const matches = v.match(/\d{4,16}/g);
    const match = matches && matches[0] || '';
    const parts = [];
    for (let i = 0, len = match.length; i < len; i += 4) {
      parts.push(match.substring(i, i + 4));
    }
    if (parts.length) {
      return parts.join(' ');
    } else {
      return v;
    }
  };

  const formatExpiryDate = (value: string) => {
    const v = value.replace(/\D/g, '');
    if (v.length >= 2) {
      return v.substring(0, 2) + '/' + v.substring(2, 4);
    }
    return v;
  };

  return (
    <div className="min-h-screen bg-slate-950 flex items-center justify-center px-4 py-12">
      <div className="max-w-2xl w-full">
        <div className="text-center mb-8">
          <Link href="/" className="text-3xl font-bold text-white mb-4 block">
            LG<span className="text-blue-500">Backtest</span>
          </Link>
          <h1 className="text-3xl font-bold text-white mb-2">
            Start Your {pricingTiers.find(t => t.id === selectedPlan)?.name}
          </h1>
          <p className="text-slate-400">
            Join thousands of successful traders and bettors
          </p>
        </div>

        {/* Plan Selection */}
        {step === 1 && (
          <div className="mb-8">
            <h2 className="text-xl font-semibold text-white mb-4">Choose Your Plan</h2>
            <div className="grid md:grid-cols-3 gap-4">
              {pricingTiers.map((tier) => (
                <div
                  key={tier.id}
                  onClick={() => setSelectedPlan(tier.id)}
                  className={`relative p-4 border rounded-lg cursor-pointer transition-all ${selectedPlan === tier.id
                    ? 'border-blue-500 bg-blue-500/10'
                    : 'border-slate-700 bg-slate-900/50 hover:border-slate-600'
                    }`}
                >
                  {tier.popular && (
                    <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                      <span className="bg-blue-500 text-white px-3 py-1 rounded-full text-xs font-semibold">
                        Most Popular
                      </span>
                    </div>
                  )}

                  <div className="text-center">
                    <h3 className="font-semibold text-white mb-1">{tier.name}</h3>
                    <div className="text-2xl font-bold text-white mb-1">
                      {tier.price === 0 ? 'Free' : `$${tier.price}`}
                      {tier.price > 0 && <span className="text-sm text-slate-400">/{tier.period}</span>}
                    </div>
                    <p className="text-sm text-slate-400 mb-3">{tier.description}</p>
                    <ul className="text-xs text-slate-300 space-y-1">
                      {tier.features.map((feature, i) => (
                        <li key={i} className="flex items-center gap-2">
                          <Check className="h-3 w-3 text-green-400" />
                          {feature}
                        </li>
                      ))}
                    </ul>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Step Indicator */}
        <div className="flex items-center justify-center mb-8">
          <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${step >= 1 ? 'bg-blue-500 text-white' : 'bg-slate-700 text-slate-400'
            }`}>
            1
          </div>
          <div className={`w-8 h-1 mx-2 ${step >= 2 ? 'bg-blue-500' : 'bg-slate-700'
            }`}></div>
          <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${step >= 2 ? 'bg-blue-500 text-white' : 'bg-slate-700 text-slate-400'
            }`}>
            2
          </div>
        </div>

        {/* Step 1: Account Details */}
        {step === 1 && (
          <form onSubmit={handleStep1Submit} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-slate-300 mb-2">
                  First Name
                </label>
                <div className="relative">
                  <User className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                  <input
                    type="text"
                    required
                    value={firstName}
                    onChange={(e) => {
                      setFirstName(e.target.value);
                      setHasInteracted(true);
                    }}
                    className="w-full pl-10 pr-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    placeholder="John"
                  />
                </div>
                {hasInteracted && errors.firstName && <p className="text-red-400 text-sm mt-1">{errors.firstName}</p>}
              </div>

              <div>
                <label className="block text-sm font-medium text-slate-300 mb-2">
                  Last Name
                </label>
                <input
                  type="text"
                  required
                  value={lastName}
                  onChange={(e) => {
                    setLastName(e.target.value);
                    setHasInteracted(true);
                  }}
                  className="w-full px-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="Doe"
                />
                {hasInteracted && errors.lastName && <p className="text-red-400 text-sm mt-1">{errors.lastName}</p>}
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Email Address
              </label>
              <div className="relative">
                <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                <input
                  type="email"
                  required
                  value={email}
                  onChange={(e) => {
                    setEmail(e.target.value);
                    setHasInteracted(true);
                  }}
                  className="w-full pl-10 pr-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="john@example.com"
                />
              </div>
              {hasInteracted && errors.email && <p className="text-red-400 text-sm mt-1">{errors.email}</p>}
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Password
              </label>
              <div className="relative">
                <Lock className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                <input
                  type={showPassword ? 'text' : 'password'}
                  required
                  value={password}
                  onChange={(e) => {
                    setPassword(e.target.value);
                    setHasInteracted(true);
                  }}
                  className="w-full pl-10 pr-12 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="At least 8 characters"
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-3 top-1/2 transform -translate-y-1/2 text-slate-400 hover:text-slate-300"
                >
                  {showPassword ? <EyeOff className="h-5 w-5" /> : <Eye className="h-5 w-5" />}
                </button>
              </div>
              {hasInteracted && errors.password && <p className="text-red-400 text-sm mt-1">{errors.password}</p>}
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Confirm Password
              </label>
              <input
                type="password"
                required
                value={confirmPassword}
                onChange={(e) => {
                  setConfirmPassword(e.target.value);
                  setHasInteracted(true);
                }}
                className="w-full px-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                placeholder="Confirm your password"
              />
              {hasInteracted && errors.confirmPassword && <p className="text-red-400 text-sm mt-1">{errors.confirmPassword}</p>}
            </div>

            <button
              type="submit"
              disabled={isLoading}
              className="w-full bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
            >
              {isLoading ? (
                <>
                  <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                  Creating Account...
                </>
              ) : (
                <>
                  {selectedPlan === 'FREE' ? 'Create Free Account' : 'Continue to Payment'}
                  <ArrowRight className="h-5 w-5" />
                </>
              )}
            </button>
          </form>
        )}

        {/* Step 2: Payment Details */}
        {step === 2 && selectedPlan !== 'FREE' && (
          <form onSubmit={handleStep2Submit} className="space-y-6">
            <div className="bg-slate-900/50 border border-slate-700 rounded-lg p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-white">Payment Information</h3>
                <div className="flex items-center gap-2">
                  <Lock className="h-4 w-4 text-slate-400" />
                  <span className="text-sm text-slate-400">SSL Secured</span>
                </div>
              </div>

              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-slate-300 mb-2">
                    Card Number
                  </label>
                  <div className="relative">
                    <CreditCard className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                    <input
                      type="text"
                      required
                      value={cardNumber}
                      onChange={(e) => {
                        setCardNumber(formatCardNumber(e.target.value));
                        setHasInteracted(true);
                      }}
                      className="w-full pl-10 pr-4 py-3 bg-slate-800 border border-slate-600 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                      placeholder="1234 5678 9012 3456"
                      maxLength={19}
                    />
                  </div>
                  {hasInteracted && errors.cardNumber && <p className="text-red-400 text-sm mt-1">{errors.cardNumber}</p>}
                </div>

                <div className="grid md:grid-cols-3 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      Expiry Date
                    </label>
                    <input
                      type="text"
                      required
                      value={expiryDate}
                      onChange={(e) => {
                        setExpiryDate(formatExpiryDate(e.target.value));
                        setHasInteracted(true);
                      }}
                      className="w-full px-4 py-3 bg-slate-800 border border-slate-600 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                      placeholder="MM/YY"
                      maxLength={5}
                    />
                    {hasInteracted && errors.expiryDate && <p className="text-red-400 text-sm mt-1">{errors.expiryDate}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      CVV
                    </label>
                    <input
                      type="text"
                      required
                      value={cvv}
                      onChange={(e) => {
                        setCvv(e.target.value.replace(/\D/g, '').slice(0, 4));
                        setHasInteracted(true);
                      }}
                      className="w-full px-4 py-3 bg-slate-800 border border-slate-600 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                      placeholder="123"
                      maxLength={4}
                    />
                    {hasInteracted && errors.cvv && <p className="text-red-400 text-sm mt-1">{errors.cvv}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      ZIP Code
                    </label>
                    <input
                      type="text"
                      required
                      className="w-full px-4 py-3 bg-slate-800 border border-slate-600 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                      placeholder="12345"
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-slate-300 mb-2">
                    Name on Card
                  </label>
                  <input
                    type="text"
                    required
                    value={cardName}
                    onChange={(e) => {
                      setCardName(e.target.value);
                      setHasInteracted(true);
                    }}
                    className="w-full px-4 py-3 bg-slate-800 border border-slate-600 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    placeholder="John Doe"
                  />
                  {hasInteracted && errors.cardName && <p className="text-red-400 text-sm mt-1">{errors.cardName}</p>}
                </div>
              </div>
            </div>

            <div className="bg-blue-500/10 border border-blue-500/20 rounded-lg p-4">
              <div className="flex items-start gap-3">
                <Star className="h-5 w-5 text-blue-400 mt-0.5" />
                <div>
                  <h4 className="font-semibold text-white mb-1">30-Day Money-Back Guarantee</h4>
                  <p className="text-slate-300 text-sm">
                    Not satisfied? Get a full refund within 30 days, no questions asked.
                  </p>
                </div>
              </div>
            </div>

            <div className="flex gap-4">
              <button
                type="button"
                onClick={() => setStep(1)}
                className="flex-1 bg-slate-800 hover:bg-slate-700 text-white py-3 px-4 rounded-lg font-semibold transition"
              >
                Back
              </button>
              <button
                type="submit"
                disabled={isLoading}
                className="flex-1 bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
              >
                {isLoading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                    Processing...
                  </>
                ) : (
                  <>
                    Start {pricingTiers.find(t => t.id === selectedPlan)?.name}
                    <Zap className="h-5 w-5" />
                  </>
                )}
              </button>
            </div>
          </form>
        )}

        <div className="text-center mt-6">
          <p className="text-slate-400">
            Already have an account?{' '}
            <Link href="/login" className="text-blue-400 hover:text-blue-300 font-semibold">
              Sign in
            </Link>
          </p>
        </div>

        <div className="border-t border-slate-800 pt-6 mt-8">
          <div className="text-center text-sm text-slate-500">
            By creating an account, you agree to our{' '}
            <Link href="/terms" className="text-blue-400 hover:text-blue-300">Terms of Service</Link>
            {' '}and{' '}
            <Link href="/privacy" className="text-blue-400 hover:text-blue-300">Privacy Policy</Link>
          </div>
        </div>
      </div>
    </div>
  );
}

export default function SignupForm() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-slate-950 flex items-center justify-center px-4 py-12">
        <div className="max-w-2xl w-full">
          <div className="text-center mb-8">
            <Link href="/" className="text-3xl font-bold text-white mb-4 block">
              LG<span className="text-blue-500">Backtest</span>
            </Link>
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto"></div>
            <p className="text-slate-400 mt-4">Loading...</p>
          </div>
        </div>
      </div>
    }>
      <SignupPageContent />
    </Suspense>
  );
}
