'use client';

import {
  Area,
  AreaChart,
  CartesianGrid,
  ResponsiveContainer,
  Tooltip,
  XAxis,
  YAxis,
} from 'recharts';

import { SimulationTrade } from '@/lib/types';

export function ResultsChart({ trades }: { trades: SimulationTrade[] }) {
  if (!trades.length) return null;

  // Transform trades into a cumulative P/L curve
  let runningPnL = 0;
  const data = trades.map((t, i) => {
    runningPnL += t.profit;
    return {
      index: i + 1,
      label: t.label,
      pnl: Number(runningPnL.toFixed(2)),
      date: t.date.split('T')[0], // simplify date
    };
  });

  // Add start point
  const chartData = [{ index: 0, label: 'Start', pnl: 0, date: '' }, ...data];

  // Determine if final profit is positive or negative
  const finalPnL = data.length > 0 ? data[data.length - 1].pnl : 0;
  const isProfit = finalPnL >= 0;
  
  // Colors based on profit/loss
  const chartColor = isProfit ? '#10b981' : '#ef4444'; // emerald-500 or red-500
  const gradientId = isProfit ? 'colorPnlProfit' : 'colorPnlLoss';
  const borderColor = isProfit ? 'border-emerald-500/20' : 'border-rose-500/20';
  const bgColor = isProfit ? 'bg-emerald-950/10' : 'bg-rose-950/10';

  return (
    <div className={`h-[300px] w-full rounded-2xl border ${borderColor} ${bgColor} p-4`}>
      <div className="mb-4 flex items-center justify-between">
        <h3 className="text-sm font-semibold text-slate-400">
          Equity Curve (R-Multiples)
        </h3>
        <div className={`flex items-center gap-2 px-3 py-1 rounded-full text-sm font-semibold ${
          isProfit 
            ? 'bg-emerald-500/20 text-emerald-400' 
            : 'bg-rose-500/20 text-rose-400'
        }`}>
          <span>{isProfit ? '↑' : '↓'}</span>
          <span>{isProfit ? '+' : ''}{finalPnL.toFixed(2)}R</span>
        </div>
      </div>
      <ResponsiveContainer width="100%" height="85%">
        <AreaChart data={chartData}>
          <defs>
            <linearGradient id={gradientId} x1="0" y1="0" x2="0" y2="1">
              <stop offset="5%" stopColor={chartColor} stopOpacity={0.3} />
              <stop offset="95%" stopColor={chartColor} stopOpacity={0} />
            </linearGradient>
          </defs>
          <CartesianGrid strokeDasharray="3 3" stroke="#1e293b" />
          <XAxis
            dataKey="index"
            stroke="#475569"
            fontSize={12}
            tickLine={false}
            axisLine={false}
          />
          <YAxis
            stroke="#475569"
            fontSize={12}
            tickLine={false}
            axisLine={false}
            tickFormatter={(value) => `${value}R`}
          />
          <Tooltip
            contentStyle={{
              backgroundColor: '#0f172a',
              border: `1px solid ${chartColor}40`,
              borderRadius: '8px',
              color: '#f8fafc',
            }}
            itemStyle={{ color: chartColor }}
            formatter={(value: number) => [`${value}R`, 'Net Profit']}
            labelFormatter={(label) => `Trade #${label}`}
          />
          <Area
            type="monotone"
            dataKey="pnl"
            stroke={chartColor}
            strokeWidth={2}
            fillOpacity={1}
            fill={`url(#${gradientId})`}
          />
        </AreaChart>
      </ResponsiveContainer>
    </div>
  );
}

