'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import Link from 'next/link';
import { Lock, ArrowRight } from 'lucide-react';

interface ProtectedRouteProps {
  children: React.ReactNode;
}

export default function ProtectedRoute({ children }: ProtectedRouteProps) {
  const [isAuthenticated, setIsAuthenticated] = useState<boolean | null>(null);
  const router = useRouter();

  useEffect(() => {
    // Check authentication status - SECURITY FIX: Removed development bypass
    const token = localStorage.getItem('authToken');
    const authenticated = !!token;

    setIsAuthenticated(authenticated);

    // If not authenticated, show login prompt instead of redirecting
    if (!authenticated) {
      // We'll handle this in the render below
    }
  }, []);

  // Show loading state while checking authentication
  if (isAuthenticated === null) {
    return (
      <div className="min-h-screen bg-slate-950 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
          <p className="text-slate-400">Loading...</p>
        </div>
      </div>
    );
  }

  // Show login prompt for unauthenticated users
  if (!isAuthenticated) {
    return (
      <div className="min-h-screen bg-slate-950 flex items-center justify-center px-4">
        <div className="max-w-md w-full text-center">
          <div className="glass-panel p-8 rounded-2xl">
            <div className="w-16 h-16 bg-blue-500/20 rounded-full flex items-center justify-center mx-auto mb-6">
              <Lock className="h-8 w-8 text-blue-400" />
            </div>

            <h1 className="text-2xl font-bold text-white mb-4">Access Required</h1>

            <p className="text-slate-400 mb-8">
              You need to sign in to access the dashboard and start backtesting strategies.
            </p>

            <div className="space-y-4">
              <Link
                href="/login"
                className="w-full bg-blue-600 hover:bg-blue-500 text-white font-semibold py-3 px-6 rounded-xl transition-all hover:scale-105 shadow-lg shadow-blue-500/25 flex items-center justify-center gap-2"
              >
                Sign In to Your Account
                <ArrowRight className="h-4 w-4" />
              </Link>

              <div className="text-center">
                <span className="text-slate-500 text-sm">Don't have an account? </span>
                <Link href="/pricing" className="text-blue-400 hover:text-blue-300 font-semibold text-sm">
                  Start your trial
                </Link>
              </div>
            </div>

            <div className="mt-8 pt-6 border-t border-white/10">
              <div className="text-xs text-slate-500 space-y-1">
                <p>• Full access to AI strategy generation</p>
                <p>• Professional backtesting tools</p>
                <p>• Advanced analytics & optimization</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // User is authenticated, show the protected content
  return <>{children}</>;
}
