/**
 * Pricing Card Component
 * Displays subscription tiers with features and pricing
 */

'use client';

import { useState } from 'react';
import { Check, Star, Zap, Crown } from 'lucide-react';
import { SUBSCRIPTION_PLANS, SubscriptionTier } from '@/lib/stripe';
import { cn } from '@/lib/utils';

interface PricingCardProps {
  tier: SubscriptionTier;
  isPopular?: boolean;
  isCurrentPlan?: boolean;
  onSelect?: (tier: SubscriptionTier) => void;
  className?: string;
}

const tierIcons = {
  FREE: null,
  TRIAL: Star,
  PRO: Zap,
  ELITE: Crown,
};

const tierColors = {
  FREE: 'border-slate-200 bg-white',
  TRIAL: 'border-blue-200 bg-blue-50',
  PRO: 'border-purple-200 bg-purple-50',
  ELITE: 'border-yellow-200 bg-yellow-50',
};

export default function PricingCard({
  tier,
  isPopular = false,
  isCurrentPlan = false,
  onSelect,
  className
}: PricingCardProps) {
  const [isLoading, setIsLoading] = useState(false);
  const plan = SUBSCRIPTION_PLANS[tier];
  const Icon = tierIcons[tier];

  const handleSelect = async () => {
    if (isCurrentPlan || !onSelect) return;

    setIsLoading(true);
    try {
      await onSelect(tier);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className={cn(
      'relative rounded-lg border p-6 shadow-sm transition-all duration-200',
      tierColors[tier],
      isPopular && 'ring-2 ring-blue-500 ring-offset-2',
      isCurrentPlan && 'ring-2 ring-green-500 ring-offset-2',
      className
    )}>
      {/* Popular Badge */}
      {isPopular && (
        <div className="absolute -top-3 left-1/2 -translate-x-1/2">
          <span className="inline-flex items-center rounded-full bg-blue-500 px-3 py-1 text-xs font-medium text-white">
            Most Popular
          </span>
        </div>
      )}

      {/* Current Plan Badge */}
      {isCurrentPlan && (
        <div className="absolute -top-3 left-1/2 -translate-x-1/2">
          <span className="inline-flex items-center rounded-full bg-green-500 px-3 py-1 text-xs font-medium text-white">
            Current Plan
          </span>
        </div>
      )}

      <div className="text-center">
        {/* Icon */}
        {Icon && (
          <div className="mx-auto mb-4 flex h-12 w-12 items-center justify-center rounded-full bg-white shadow-sm">
            <Icon className="h-6 w-6 text-slate-600" />
          </div>
        )}

        {/* Plan Name */}
        <h3 className="text-lg font-semibold text-slate-900">{plan.name}</h3>

        {/* Pricing */}
        <div className="mt-4">
          {plan.price === 0 ? (
            <div className="text-3xl font-bold text-slate-900">
              {tier === 'FREE' ? 'Free Trial' : 'Free'}
            </div>
          ) : (
            <div className="flex items-baseline justify-center">
              <span className="text-3xl font-bold text-slate-900">${plan.price}</span>
              <span className="ml-1 text-slate-600">/month</span>
            </div>
          )}
        </div>

        {/* Trial Info */}
        {tier === 'FREE' && plan.trialDays && (
          <p className="mt-2 text-sm text-green-600 font-medium">
            {plan.trialDays} days of full access included
          </p>
        )}
        {tier === 'TRIAL' && plan.trialDays && (
          <p className="mt-2 text-sm text-slate-600">
            {plan.trialDays} days free, then ${plan.price}/month
          </p>
        )}

        {/* Description */}
        <p className="mt-4 text-sm text-slate-600">
          {tier === 'FREE' && '$1 for 7 days, then $20/month - cancel anytime'}
          {tier === 'PRO' && 'Advanced features for serious traders'}
          {tier === 'ELITE' && 'Unlimited access with priority support'}
        </p>
      </div>

      {/* Features */}
      <div className="mt-6 space-y-3">
        <div className="flex items-center text-sm">
          <Check className="mr-2 h-4 w-4 text-green-500" />
          <span>
            {plan.features.maxStrategies === -1
              ? 'Unlimited strategies'
              : `Up to ${plan.features.maxStrategies} strategies`}
          </span>
        </div>

        <div className="flex items-center text-sm">
          <Check className="mr-2 h-4 w-4 text-green-500" />
          <span>
            {plan.features.maxConversations === -1
              ? 'Unlimited conversations'
              : `Up to ${plan.features.maxConversations} conversations`}
          </span>
        </div>

        {plan.features.ragAccess && (
          <div className="flex items-center text-sm">
            <Check className="mr-2 h-4 w-4 text-green-500" />
            <span>AI knowledge base access</span>
          </div>
        )}

        {plan.features.advancedAnalytics && (
          <div className="flex items-center text-sm">
            <Check className="mr-2 h-4 w-4 text-green-500" />
            <span>Advanced analytics</span>
          </div>
        )}

        {plan.features.prioritySupport && (
          <div className="flex items-center text-sm">
            <Check className="mr-2 h-4 w-4 text-green-500" />
            <span>Priority support</span>
          </div>
        )}
      </div>

      {/* Action Button */}
      <div className="mt-6">
        {isCurrentPlan ? (
          <button
            disabled
            className="w-full rounded-lg bg-green-100 px-4 py-2 text-sm font-medium text-green-800"
          >
            Current Plan
          </button>
        ) : (
          <button
            onClick={handleSelect}
            disabled={isLoading}
            className={cn(
              'w-full rounded-lg px-4 py-2 text-sm font-medium transition-colors',
              isPopular
                ? 'bg-blue-600 text-white hover:bg-blue-700 focus:ring-2 focus:ring-blue-500'
                : 'bg-slate-900 text-white hover:bg-slate-800 focus:ring-2 focus:ring-slate-500',
              isLoading && 'cursor-not-allowed opacity-50'
            )}
          >
            {isLoading ? 'Processing...' : `Choose ${plan.name}`}
          </button>
        )}
      </div>
    </div>
  );
}
