'use client';

import { useState } from 'react';
import {
  PaymentElement,
  useStripe,
  useElements,
} from '@stripe/react-stripe-js';
import { Loader2, Lock, Shield, Star } from 'lucide-react';

interface PaymentFormProps {
  onSuccess: () => void;
  onError: (error: string) => void;
  returnUrl: string;
  planName: string;
  price: number;
}

export function PaymentForm({ onSuccess, onError, returnUrl, planName, price }: PaymentFormProps) {
  const stripe = useStripe();
  const elements = useElements();
  const [isProcessing, setIsProcessing] = useState(false);
  const [errorMessage, setErrorMessage] = useState<string | null>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!stripe || !elements) {
      return;
    }

    setIsProcessing(true);
    setErrorMessage(null);

    try {
      const { error, paymentIntent } = await stripe.confirmPayment({
        elements,
        confirmParams: {
          return_url: returnUrl,
        },
        redirect: 'if_required',
      });

      if (error) {
        setErrorMessage(error.message || 'An error occurred');
        onError(error.message || 'Payment failed');
      } else if (paymentIntent && paymentIntent.status === 'succeeded') {
        onSuccess();
      }
    } catch (err) {
      const message = err instanceof Error ? err.message : 'Payment failed';
      setErrorMessage(message);
      onError(message);
    } finally {
      setIsProcessing(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div className="bg-slate-900/50 border border-slate-700 rounded-lg p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-white">Payment Information</h3>
          <div className="flex items-center gap-2">
            <Lock className="h-4 w-4 text-green-400" />
            <span className="text-sm text-green-400">Secured by Stripe</span>
          </div>
        </div>

        {/* Stripe Payment Element - handles all card input securely */}
        <PaymentElement 
          options={{
            layout: 'tabs',
          }}
        />

        {errorMessage && (
          <div className="mt-4 p-3 bg-red-500/10 border border-red-500/20 rounded-lg">
            <p className="text-red-400 text-sm">{errorMessage}</p>
          </div>
        )}
      </div>

      {/* Trust badges */}
      <div className="bg-blue-500/10 border border-blue-500/20 rounded-lg p-4">
        <div className="flex items-start gap-3">
          <Star className="h-5 w-5 text-blue-400 mt-0.5" />
          <div>
            <h4 className="font-semibold text-white mb-1">30-Day Money-Back Guarantee</h4>
            <p className="text-slate-300 text-sm">
              Not satisfied? Get a full refund within 30 days, no questions asked.
            </p>
          </div>
        </div>
      </div>

      <div className="flex items-center justify-center gap-4 text-slate-400 text-xs">
        <div className="flex items-center gap-1">
          <Shield className="h-4 w-4" />
          <span>PCI Compliant</span>
        </div>
        <span>•</span>
        <span>256-bit SSL Encryption</span>
        <span>•</span>
        <span>Secure Payment</span>
      </div>

      <button
        type="submit"
        disabled={!stripe || isProcessing}
        className="w-full bg-blue-600 hover:bg-blue-500 disabled:bg-slate-600 disabled:cursor-not-allowed text-white font-semibold py-4 px-6 rounded-xl transition-all hover:scale-[1.02] shadow-lg shadow-blue-500/25 disabled:hover:scale-100 flex items-center justify-center gap-2"
      >
        {isProcessing ? (
          <>
            <Loader2 className="h-5 w-5 animate-spin" />
            Processing...
          </>
        ) : (
          <>
            Start {planName} - ${price === 1 ? '1 for 7 days' : `${price}/mo`}
          </>
        )}
      </button>
    </form>
  );
}
