/**
 * NBA Data Viewer Component
 * Displays NBA players, teams, games, and statistics
 */

'use client';

import { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Loader2, Users, Trophy, Calendar, BarChart3, RefreshCw } from 'lucide-react';

interface NBAPlayer {
  id: string;
  full_name: string;
  first_name: string;
  last_name: string;
  is_active: boolean;
  last_updated: string;
}

interface NBATeam {
  id: string;
  full_name: string;
  abbreviation: string;
  nickname: string;
  city: string;
  state?: string;
  year_founded?: number;
  last_updated: string;
}

interface NBAGame {
  game_id: string;
  game_date: string;
  home_team_name: string;
  home_score?: number;
  away_team_id: string;
  season: string;
  season_type: string;
  last_updated: string;
}

interface NBALiveGame {
  game_id: string;
  home_team: string;
  home_score: number;
  away_team: string;
  away_score: number;
  status: string;
  period: number;
  game_clock: string;
  last_updated: string;
}

interface PlayerStats {
  player_id: string;
  stats: Array<{
    season: string;
    team: string;
    games_played: number;
    points: number;
    rebounds: number;
    assists: number;
    steals: number;
    blocks: number;
  }>;
  total_seasons: number;
}

export default function NBADataViewer() {
  const [activeTab, setActiveTab] = useState('players');
  const [loading, setLoading] = useState(false);
  const [players, setPlayers] = useState<NBAPlayer[]>([]);
  const [teams, setTeams] = useState<NBATeam[]>([]);
  const [games, setGames] = useState<NBAGame[]>([]);
  const [liveGames, setLiveGames] = useState<NBALiveGame[]>([]);
  const [playerStats, setPlayerStats] = useState<PlayerStats | null>(null);
  const [selectedPlayerId, setSelectedPlayerId] = useState<string>('');
  const [searchTerm, setSearchTerm] = useState('');
  const [season, setSeason] = useState('2023-24');
  const [cacheEnabled, setCacheEnabled] = useState(true);

  // Load initial data
  useEffect(() => {
    loadPlayers();
    loadTeams();
    loadGames();
    loadLiveGames();
  }, []);

  const getAuthHeaders = () => {
    const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
    return token ? { Authorization: `Bearer ${token}` } : {};
  };

  const loadPlayers = async () => {
    setLoading(true);
    try {
      const response = await fetch(`/api/nba/players?cache=${cacheEnabled}&limit=50`, {
        headers: {
          ...getAuthHeaders(),
        },
      });
      const data = await response.json();
      if (data.success) {
        setPlayers(data.data);
        console.log(`✅ Loaded ${data.data.length} NBA players`);
      }
    } catch (error) {
      console.error('Failed to load players:', error);
    } finally {
      setLoading(false);
    }
  };

  const loadTeams = async () => {
    try {
      const response = await fetch(`/api/nba/teams?cache=${cacheEnabled}`, {
        headers: {
          ...getAuthHeaders(),
        },
      });
      const data = await response.json();
      if (data.success) {
        setTeams(data.data);
        console.log(`✅ Loaded ${data.data.length} NBA teams`);
      }
    } catch (error) {
      console.error('Failed to load teams:', error);
    }
  };

  const loadGames = async () => {
    try {
      const response = await fetch(`/api/nba/games?cache=${cacheEnabled}&season=${season}&limit=20`, {
        headers: {
          ...getAuthHeaders(),
        },
      });
      const data = await response.json();
      if (data.success) {
        setGames(data.data);
        console.log(`✅ Loaded ${data.data.length} NBA games for ${season}`);
      }
    } catch (error) {
      console.error('Failed to load games:', error);
    }
  };

  const loadLiveGames = async () => {
    try {
      const response = await fetch(`/api/nba/games/live?cache=false`, { // Always fresh for live games
        headers: {
          ...getAuthHeaders(),
        },
      });
      const data = await response.json();
      if (data.success && data.data.games) {
        setLiveGames(data.data.games);
        console.log(`🔴 Loaded ${data.data.games.length} live NBA games`);
      }
    } catch (error) {
      console.error('Failed to load live games:', error);
    }
  };

  const loadPlayerStats = async (playerId: string) => {
    if (!playerId) return;

    setLoading(true);
    try {
      const response = await fetch(`/api/nba/players/${playerId}/stats?cache=${cacheEnabled}`, {
        headers: {
          ...getAuthHeaders(),
        },
      });
      const data = await response.json();
      if (data.success) {
        setPlayerStats(data.data);
        console.log(`📊 Loaded stats for player ${playerId}: ${data.data.total_seasons} seasons`);
      }
    } catch (error) {
      console.error('Failed to load player stats:', error);
    } finally {
      setLoading(false);
    }
  };

  const filteredPlayers = players.filter(player =>
    player.full_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    player.last_name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const filteredTeams = teams.filter(team =>
    team.full_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    team.abbreviation.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-white">NBA Data Explorer</h2>
          <p className="text-slate-400">Explore NBA statistics for sports betting strategies</p>
        </div>

        <div className="flex items-center gap-4">
          <div className="flex items-center gap-2">
            <input
              type="checkbox"
              id="cache"
              checked={cacheEnabled}
              onChange={(e) => setCacheEnabled(e.target.checked)}
              className="rounded"
            />
            <label htmlFor="cache" className="text-sm text-slate-300">
              Enable Cache
            </label>
          </div>

          <Button
            onClick={() => {
              loadPlayers();
              loadTeams();
              loadGames();
              loadLiveGames();
            }}
            variant="outline"
            size="sm"
            disabled={loading}
          >
            {loading ? <Loader2 className="h-4 w-4 animate-spin" /> : <RefreshCw className="h-4 w-4" />}
            Refresh Data
          </Button>
        </div>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="players" className="flex items-center gap-2">
            <Users className="h-4 w-4" />
            Players
          </TabsTrigger>
          <TabsTrigger value="teams" className="flex items-center gap-2">
            <Trophy className="h-4 w-4" />
            Teams
          </TabsTrigger>
          <TabsTrigger value="games" className="flex items-center gap-2">
            <Calendar className="h-4 w-4" />
            Games
          </TabsTrigger>
          <TabsTrigger value="live" className="flex items-center gap-2">
            <div className="relative">
              <div className="absolute -top-1 -right-1 w-2 h-2 bg-red-500 rounded-full animate-pulse"></div>
              Live
            </div>
          </TabsTrigger>
          <TabsTrigger value="stats" className="flex items-center gap-2">
            <BarChart3 className="h-4 w-4" />
            Player Stats
          </TabsTrigger>
        </TabsList>

        <TabsContent value="players" className="space-y-4">
          <div className="flex items-center gap-4">
            <Input
              placeholder="Search players..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="max-w-sm"
            />
            <Badge variant="secondary">
              {filteredPlayers.length} players
            </Badge>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {filteredPlayers.slice(0, 24).map((player) => (
              <Card key={player.id} className="bg-slate-800 border-slate-700">
                <CardHeader className="pb-2">
                  <CardTitle className="text-white text-lg">{player.full_name}</CardTitle>
                  <CardDescription className="flex items-center gap-2">
                    <Badge variant={player.is_active ? "default" : "secondary"}>
                      {player.is_active ? "Active" : "Inactive"}
                    </Badge>
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => {
                      setSelectedPlayerId(player.id);
                      setActiveTab('stats');
                      loadPlayerStats(player.id);
                    }}
                    className="w-full"
                  >
                    View Stats
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="teams" className="space-y-4">
          <div className="flex items-center gap-4">
            <Input
              placeholder="Search teams..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="max-w-sm"
            />
            <Badge variant="secondary">
              {filteredTeams.length} teams
            </Badge>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {filteredTeams.map((team) => (
              <Card key={team.id} className="bg-slate-800 border-slate-700">
                <CardHeader className="pb-2">
                  <CardTitle className="text-white text-lg">{team.full_name}</CardTitle>
                  <CardDescription className="flex items-center gap-2">
                    <Badge variant="outline">{team.abbreviation}</Badge>
                    {team.year_founded && (
                      <span className="text-slate-400">Founded {team.year_founded}</span>
                    )}
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="text-sm text-slate-300">
                    <div>{team.city}{team.state && `, ${team.state}`}</div>
                    <div className="text-slate-400">{team.nickname}</div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="games" className="space-y-4">
          <div className="flex items-center gap-4">
            <select
              value={season}
              onChange={(e) => setSeason(e.target.value)}
              className="w-32 h-9 rounded-md border border-input bg-background px-3 py-1 text-sm"
            >
              <option value="2023-24">2023-24</option>
              <option value="2022-23">2022-23</option>
              <option value="2021-22">2021-22</option>
            </select>
            <Button onClick={loadGames} size="sm" variant="outline">
              Load Games
            </Button>
            <Badge variant="secondary">
              {games.length} games
            </Badge>
          </div>

          <div className="space-y-2">
            {games.map((game) => (
              <Card key={game.game_id} className="bg-slate-800 border-slate-700">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                      <div className="text-white font-medium">{game.home_team_name}</div>
                      <div className="text-slate-400">vs</div>
                      <div className="text-white font-medium">{game.away_team_id}</div>
                    </div>
                    <div className="flex items-center gap-4">
                      <Badge variant="outline">{game.season_type}</Badge>
                      <span className="text-slate-400 text-sm">
                        {new Date(game.game_date).toLocaleDateString()}
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="live" className="space-y-4">
          <div className="flex items-center gap-4">
            <Button onClick={loadLiveGames} size="sm" variant="outline">
              <RefreshCw className="h-4 w-4 mr-2" />
              Refresh Live Games
            </Button>
            <Badge variant="destructive" className="animate-pulse">
              {liveGames.length} live games
            </Badge>
          </div>

          <div className="space-y-2">
            {liveGames.map((game) => (
              <Card key={game.game_id} className="bg-slate-800 border-slate-700 border-l-4 border-l-red-500">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                      <div className="text-white font-medium">
                        {game.home_team} {game.home_score}
                      </div>
                      <div className="text-red-400 font-bold">VS</div>
                      <div className="text-white font-medium">
                        {game.away_score} {game.away_team}
                      </div>
                    </div>
                    <div className="flex items-center gap-4">
                      <Badge variant="destructive">{game.status}</Badge>
                      {game.game_clock && (
                        <span className="text-red-400 font-mono">
                          Q{game.period} {game.game_clock}
                        </span>
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="stats" className="space-y-4">
          <div className="flex items-center gap-4">
            <Input
              placeholder="Enter player ID..."
              value={selectedPlayerId}
              onChange={(e) => setSelectedPlayerId(e.target.value)}
              className="max-w-sm"
            />
            <Button
              onClick={() => loadPlayerStats(selectedPlayerId)}
              disabled={loading || !selectedPlayerId}
              size="sm"
            >
              {loading ? <Loader2 className="h-4 w-4 animate-spin" /> : <BarChart3 className="h-4 w-4" />}
              Load Stats
            </Button>
          </div>

          {playerStats && (
            <div className="space-y-4">
              <div className="flex items-center gap-4">
                <h3 className="text-xl font-bold text-white">Player Statistics</h3>
                <Badge variant="secondary">
                  {playerStats.total_seasons} seasons
                </Badge>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {playerStats.stats.map((season, index) => (
                  <Card key={index} className="bg-slate-800 border-slate-700">
                    <CardHeader className="pb-2">
                      <CardTitle className="text-white">{season.season}</CardTitle>
                      <CardDescription>{season.team}</CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-2">
                      <div className="grid grid-cols-2 gap-4 text-sm">
                        <div>
                          <div className="text-slate-400">Games</div>
                          <div className="text-white font-medium">{season.games_played}</div>
                        </div>
                        <div>
                          <div className="text-slate-400">Points</div>
                          <div className="text-white font-medium">{season.points.toFixed(1)}</div>
                        </div>
                        <div>
                          <div className="text-slate-400">Rebounds</div>
                          <div className="text-white font-medium">{season.rebounds.toFixed(1)}</div>
                        </div>
                        <div>
                          <div className="text-slate-400">Assists</div>
                          <div className="text-white font-medium">{season.assists.toFixed(1)}</div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </div>
          )}
        </TabsContent>
      </Tabs>
    </div>
  );
}
