'use client';

import React, { useState, useRef, useEffect } from 'react';
import { MessageSquare, X, Send, User, Minimize2, Users, Mail } from 'lucide-react';

interface Message {
  id: string;
  content: string;
  role: 'user' | 'assistant' | 'system';
  timestamp: Date;
}

interface LiveChatWidgetProps {
  isOpen?: boolean;
  onClose?: () => void;
  embedded?: boolean;
}

export default function LiveChatWidget({ isOpen: externalOpen, onClose, embedded = false }: LiveChatWidgetProps) {
  const [isOpen, setIsOpen] = useState(externalOpen ?? false);
  const [sessionId, setSessionId] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState('');
  const [userName, setUserName] = useState('');
  const [showEmailForm, setShowEmailForm] = useState(true);
  const [messages, setMessages] = useState<Message[]>([
    {
      id: '1',
      content: "Hi! 👋 Welcome to EventheOdds support. Please enter your details below so our team can get back to you.",
      role: 'assistant',
      timestamp: new Date(),
    }
  ]);
  const [input, setInput] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [messageSent, setMessageSent] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  useEffect(() => {
    if (externalOpen !== undefined) {
      setIsOpen(externalOpen);
    }
  }, [externalOpen]);

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  }, [messages]);

  useEffect(() => {
    if (isOpen && !showEmailForm) {
      setTimeout(() => textareaRef.current?.focus(), 100);
    }
  }, [isOpen, showEmailForm]);

  const handleClose = () => {
    setIsOpen(false);
    onClose?.();
  };

  const handleOpen = () => {
    setIsOpen(true);
  };

  const handleStartChat = (e: React.FormEvent) => {
    e.preventDefault();
    if (!userEmail.trim()) return;

    setShowEmailForm(false);
    setMessages(prev => [
      ...prev,
      {
        id: Date.now().toString(),
        content: `Great! You're chatting as ${userName || 'Guest'} (${userEmail}). How can we help you today?`,
        role: 'assistant',
        timestamp: new Date(),
      }
    ]);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!input.trim() || isLoading) return;

    const userMessage: Message = {
      id: Date.now().toString(),
      content: input.trim(),
      role: 'user',
      timestamp: new Date(),
    };

    setMessages(prev => [...prev, userMessage]);
    setInput('');
    setIsLoading(true);

    try {
      const response = await fetch('/api/support/chat', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({
          message: userMessage.content,
          sessionId,
          userEmail,
          userName,
        }),
      });

      const data = await response.json();

      if (response.ok) {
        if (data.sessionId) {
          setSessionId(data.sessionId);
        }

        const assistantMessage: Message = {
          id: (Date.now() + 1).toString(),
          content: getHumanResponse(messageSent),
          role: 'assistant',
          timestamp: new Date(),
        };

        setMessages(prev => [...prev, assistantMessage]);
        setMessageSent(true);
      } else {
        throw new Error(data.error || 'Failed to send message');
      }
    } catch (error) {
      console.error('Chat error:', error);
      const errorMessage: Message = {
        id: (Date.now() + 1).toString(),
        content: "Sorry, there was an issue sending your message. Please try again or email us directly at support@eventheodds.ai",
        role: 'assistant',
        timestamp: new Date(),
      };
      setMessages(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
    }
  };

  const getHumanResponse = (alreadySent: boolean): string => {
    if (!alreadySent) {
      return "✅ Message received! Our support team has been notified and will respond to your email shortly. Average response time is under 2 hours during business hours.\n\nFeel free to send additional details or questions - we'll receive all your messages.";
    }
    return "📨 Got it! We've added this to your conversation. Our team will see all your messages and respond via email.";
  };

  const chatContent = (
    <div className={`flex flex-col ${embedded ? 'h-[500px]' : 'h-[500px]'} bg-slate-900 ${embedded ? 'rounded-xl border border-white/10' : 'rounded-2xl shadow-2xl'}`}>
      {/* Header */}
      <div className="flex items-center justify-between px-4 py-3 bg-gradient-to-r from-blue-600 to-purple-600 rounded-t-xl">
        <div className="flex items-center gap-3">
          <div className="w-8 h-8 bg-white/20 rounded-full flex items-center justify-center">
            <Users className="h-5 w-5 text-white" />
          </div>
          <div>
            <h3 className="text-white font-semibold text-sm">Live Support</h3>
            <div className="flex items-center gap-1">
              <span className="w-2 h-2 bg-green-400 rounded-full animate-pulse" />
              <p className="text-white/70 text-xs">Human support • We'll email you back</p>
            </div>
          </div>
        </div>
        {!embedded && (
          <button
            onClick={handleClose}
            className="text-white/70 hover:text-white transition p-1"
          >
            <X className="h-5 w-5" />
          </button>
        )}
      </div>

      {/* Messages */}
      <div className="flex-1 overflow-y-auto p-4 space-y-4">
        {messages.map((message) => (
          <div
            key={message.id}
            className={`flex gap-3 ${message.role === 'user' ? 'flex-row-reverse' : ''}`}
          >
            <div className={`w-8 h-8 rounded-full flex items-center justify-center flex-shrink-0 ${
              message.role === 'user' 
                ? 'bg-blue-500' 
                : 'bg-gradient-to-br from-purple-500 to-blue-500'
            }`}>
              {message.role === 'user' ? (
                <User className="h-4 w-4 text-white" />
              ) : (
                <Users className="h-4 w-4 text-white" />
              )}
            </div>
            <div className={`max-w-[75%] ${message.role === 'user' ? 'text-right' : ''}`}>
              <div className={`rounded-2xl px-4 py-2 ${
                message.role === 'user'
                  ? 'bg-blue-600 text-white rounded-tr-sm'
                  : 'bg-slate-800 text-slate-200 rounded-tl-sm'
              }`}>
                <p className="text-sm whitespace-pre-wrap">{message.content}</p>
              </div>
              <p className="text-xs text-slate-500 mt-1">
                {message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
              </p>
            </div>
          </div>
        ))}
        
        {isLoading && (
          <div className="flex gap-3">
            <div className="w-8 h-8 rounded-full bg-gradient-to-br from-purple-500 to-blue-500 flex items-center justify-center flex-shrink-0">
              <Users className="h-4 w-4 text-white" />
            </div>
            <div className="bg-slate-800 rounded-2xl rounded-tl-sm px-4 py-3">
              <div className="flex gap-1">
                <div className="w-2 h-2 bg-slate-500 rounded-full animate-bounce" style={{ animationDelay: '0ms' }} />
                <div className="w-2 h-2 bg-slate-500 rounded-full animate-bounce" style={{ animationDelay: '150ms' }} />
                <div className="w-2 h-2 bg-slate-500 rounded-full animate-bounce" style={{ animationDelay: '300ms' }} />
              </div>
            </div>
          </div>
        )}
        
        <div ref={messagesEndRef} />
      </div>

      {/* Email Form or Chat Input */}
      {showEmailForm ? (
        <form onSubmit={handleStartChat} className="p-4 border-t border-slate-700 space-y-3">
          <div>
            <input
              ref={inputRef}
              type="text"
              value={userName}
              onChange={(e) => setUserName(e.target.value)}
              placeholder="Your name (optional)"
              className="w-full bg-slate-800 border border-slate-600 rounded-lg px-4 py-2 text-white text-sm placeholder-slate-400 focus:border-blue-500 focus:outline-none"
            />
          </div>
          <div>
            <input
              type="email"
              required
              value={userEmail}
              onChange={(e) => setUserEmail(e.target.value)}
              placeholder="Your email (required) *"
              className="w-full bg-slate-800 border border-slate-600 rounded-lg px-4 py-2 text-white text-sm placeholder-slate-400 focus:border-blue-500 focus:outline-none"
            />
          </div>
          <button
            type="submit"
            disabled={!userEmail.trim()}
            className="w-full bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-white py-2.5 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
          >
            <MessageSquare className="h-4 w-4" />
            Start Chat
          </button>
          <p className="text-xs text-slate-500 text-center">
            We'll respond to your email address
          </p>
        </form>
      ) : (
        <form onSubmit={handleSubmit} className="p-4 border-t border-slate-700">
          <div className="flex gap-2">
            <textarea
              ref={textareaRef}
              value={input}
              onChange={(e) => setInput(e.target.value)}
              onKeyDown={(e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                  e.preventDefault();
                  handleSubmit(e);
                }
              }}
              placeholder="Type your message... (Enter to send)"
              disabled={isLoading}
              rows={2}
              className="flex-1 bg-slate-800 border border-slate-600 rounded-lg px-4 py-2 text-white text-sm placeholder-slate-400 focus:border-blue-500 focus:outline-none disabled:opacity-50 resize-none"
            />
            <button
              type="submit"
              disabled={!input.trim() || isLoading}
              className="w-10 h-10 bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed rounded-lg flex items-center justify-center transition self-end"
            >
              <Send className="h-4 w-4 text-white" />
            </button>
          </div>
          <div className="flex items-center justify-between mt-2">
            <p className="text-xs text-slate-500 flex items-center gap-1">
              <Mail className="h-3 w-3" />
              Replies sent to {userEmail}
            </p>
          </div>
        </form>
      )}
    </div>
  );

  if (embedded) {
    return chatContent;
  }

  return (
    <>
      {isOpen && (
        <div className="fixed bottom-24 right-6 w-[380px] z-50 animate-in slide-in-from-bottom-5 duration-300">
          {chatContent}
        </div>
      )}

      <button
        onClick={isOpen ? handleClose : handleOpen}
        className={`fixed bottom-6 right-6 w-14 h-14 rounded-full shadow-lg z-50 transition-all duration-300 flex items-center justify-center ${
          isOpen 
            ? 'bg-slate-700 hover:bg-slate-600' 
            : 'bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-500 hover:to-purple-500'
        }`}
      >
        {isOpen ? (
          <Minimize2 className="h-6 w-6 text-white" />
        ) : (
          <>
            <MessageSquare className="h-6 w-6 text-white" />
            <span className="absolute -top-1 -right-1 w-3 h-3 bg-green-500 rounded-full border-2 border-slate-900" />
          </>
        )}
      </button>
    </>
  );
}
