'use client';

import { useState, useRef, useEffect } from 'react';
import TextareaAutosize from 'react-textarea-autosize';
import { Send, Bot, User, Loader2, Sparkles, BookOpen, ExternalLink, Download, FileText, FileSpreadsheet } from 'lucide-react';

import { cn } from '@/lib/utils';
import type { ChatMessage } from '@/lib/chat';
import BacktestChart from './BacktestChart';
import { TradeChart } from './TradeChart';

interface ChatInterfaceProps {
  domain: 'sports';
  onStrategyGenerated?: (code: string, name: string) => void;
  className?: string;
}

export function ChatInterface({ domain, onStrategyGenerated, className }: ChatInterfaceProps) {
  // Generate a simple user ID for memory (in a real app, this would come from auth)
  const userId = `user-${typeof window !== 'undefined' ? localStorage.getItem('userId') || Math.random().toString(36).substr(2, 9) : 'anonymous'}`;

  // Store userId in localStorage for persistence
  if (typeof window !== 'undefined') {
    localStorage.setItem('userId', userId.replace('user-', ''));
  }

  const [messages, setMessages] = useState<ChatMessage[]>([
    {
      id: 'welcome',
      role: 'assistant',
      content: `🏆 **Welcome to AI Sport Guru!**

I'm your intelligent sports analytics assistant with access to real sportsbook data, historical statistics, and comprehensive player/team information.

**What I can help you with:**
• 📊 **Statistics & Analysis** - Team standings, player stats, historical trends
• 🔍 **Strategy Research** - Ask about betting strategies and their historical performance
• 🧪 **Backtest Strategies** - Test your ideas against real historical data
• 📈 **Odds Analysis** - View real sportsbook odds from DraftKings, FanDuel, BetMGM & more

**Example questions to ask:**
• "Show me NBA standings for 2025"
• "What are the current NFL betting odds?"
• "Backtest betting on NBA home favorites in 2024"
• "Show player props for tonight's NFL games"

**To run a backtest, try:**
• "Backtest NFL home favorites with spread coverage for 2025 season"
• "Test a 3-leg NBA parlay of moneyline favorites in 2024"

⚠️ **Important Disclaimer:** I provide **statistics and historical backtest results only**. Past performance does not guarantee future results. I cannot and will not predict winners or guarantee positive outcomes. All information is for research and educational purposes.

💡 **Tip:** Use the **Strategy Playground** tab for quick backtests with visual results!`,
      timestamp: new Date(),
    },
  ]);
  const [input, setInput] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [showChart, setShowChart] = useState<{ [key: number]: boolean }>({});
  const [showTradeHistory, setShowTradeHistory] = useState<{ [key: number]: boolean }>({});
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const isSendingRef = useRef(false);
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  // Function to download backtest reports
  const downloadReport = async (format: 'csv' | 'txt', messageIndex: number) => {
    const message = messages[messageIndex];
    if (!message.backtestResults) return;

    const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;

    try {
      const response = await fetch('/api/reports/generate', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(token ? { 'Authorization': `Bearer ${token}` } : {})
        },
        body: JSON.stringify({
          backtestResults: message.backtestResults,
          strategyName: 'AI Generated Strategy',
          market: domain.toUpperCase(),
          timestamp: message.timestamp.toISOString(),
          format: format
        })
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.style.display = 'none';
        a.href = url;

        const fileName = `eventheodds-backtest-${domain}-${new Date().toISOString().split('T')[0]}.${format === 'csv' ? 'csv' : 'txt'}`;
        a.download = fileName;

        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);

        // Track download in memory system
        try {
          await fetch('/api/chat', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
            },
            body: JSON.stringify({
              message: `I just downloaded a ${format.toUpperCase()} report for my ${domain} backtest with ${message.backtestResults.total_trades} trades and ${message.backtestResults.results?.win_rate}% win rate.`,
              domain: domain,
              memoryOnly: true // Flag to indicate this is just for memory tracking
            })
          });
        } catch (memoryError) {
          console.log('Memory tracking failed, but download succeeded');
        }
      } else {
        console.error('Failed to download report');
      }
    } catch (error) {
      console.error('Error downloading report:', error);
    }
  };

  const showTradeChart = (messageIndex: number) => {
    setShowChart(prev => ({
      ...prev,
      [messageIndex]: !prev[messageIndex]
    }));
  };

  const toggleTradeHistory = (messageIndex: number) => {
    setShowTradeHistory(prev => ({
      ...prev,
      [messageIndex]: !prev[messageIndex]
    }));
  };

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  // Poll for async backtest job status
  const pollBacktestJob = async (jobId: string, maxAttempts: number = 60): Promise<any> => {
    const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
    
    for (let attempt = 0; attempt < maxAttempts; attempt++) {
      try {
        const response = await fetch(`/api/backtest/status/${jobId}`, {
          method: 'GET',
          headers: { 
            'Content-Type': 'application/json',
            ...(token ? { 'Authorization': `Bearer ${token}` } : {})
          },
        });

        if (!response.ok) {
          throw new Error('Failed to check job status');
        }

        const data = await response.json();

        if (data.status === 'completed') {
          return data.results;
        } else if (data.status === 'failed') {
          throw new Error(data.error || 'Backtest job failed');
        }

        // Update loading message with progress
        setMessages(prev => {
          const lastMsg = prev[prev.length - 1];
          if (lastMsg.id.startsWith('loading-')) {
            return [...prev.slice(0, -1), {
              ...lastMsg,
              content: `⏳ Processing large dataset... (${attempt + 1}/${maxAttempts})\n\nThis may take a minute for tick-level data analysis.`,
            }];
          }
          return prev;
        });

        // Wait before next poll (2 seconds)
        await new Promise(resolve => setTimeout(resolve, 2000));
      } catch (error) {
        if (attempt === maxAttempts - 1) {
          throw error;
        }
        await new Promise(resolve => setTimeout(resolve, 2000));
      }
    }
    throw new Error('Job timed out after maximum attempts');
  };

  const toDate = (value: any): Date => {
    if (value instanceof Date) return value;
    if (typeof value === 'string' || typeof value === 'number') {
      const d = new Date(value);
      return isNaN(d.getTime()) ? new Date() : d;
    }
    return new Date();
  };

  const normalizeMessage = (msg: any): ChatMessage => ({
    id: msg?.id || `msg-${Date.now()}`,
    role: msg?.role === 'user' ? 'user' : 'assistant',
    content: msg?.content || '',
    timestamp: toDate(msg?.timestamp),
    ragInsight: msg?.ragInsight,
    visualization: msg?.visualization,
    backtestResults: msg?.backtestResults,
  });

  const sendMessage = async () => {
    if (isSendingRef.current) return;
    const currentText = input.trim() || textareaRef.current?.value?.trim() || '';
    if (!currentText || isLoading) return;

    isSendingRef.current = true;

    const userMessage: ChatMessage = {
      id: `user-${Date.now()}`,
      role: 'user',
      content: currentText,
      timestamp: new Date(),
    };

    setMessages(prev => [...prev, userMessage]);
    setInput('');
    if (textareaRef.current) textareaRef.current.value = '';
    setIsLoading(true);

    try {
      const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
      
      const response = await fetch('/api/chat', {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          ...(token ? { 'Authorization': `Bearer ${token}` } : {})
        },
        body: JSON.stringify({ message: userMessage.content, domain, userId }),
      });

      if (!response.ok) {
        throw new Error('Chat API failed');
      }

      const data = await response.json();

      // Check if this is an async job that needs polling
      if (data.job_id && data.status === 'accepted') {
        // Add loading message
        const loadingMessage: ChatMessage = {
          id: `loading-${Date.now()}`,
          role: 'assistant',
          content: `⏳ Starting backtest analysis...\n\nProcessing large dataset with tick-level data. This may take a minute.`,
          timestamp: new Date(),
        };
        setMessages(prev => [...prev, loadingMessage]);

        try {
          // Poll for results
          const results = await pollBacktestJob(data.job_id);

          // Remove loading message and add real results
          setMessages(prev => {
            const withoutLoading = prev.filter(m => !m.id.startsWith('loading-'));
            const assistantMessage: ChatMessage = {
              id: `assistant-${Date.now()}`,
              role: 'assistant',
              content: `✅ **Backtest Complete!**\n\n${results.performance_summary || 'Analysis complete.'}\n\n**Key Metrics:**\n• Total Trades: ${results.total_trades || 0}\n• Win Rate: ${results.win_rate || 0}%\n• Total Profit: $${results.total_profit || 0}\n• Sharpe Ratio: ${results.sharpe_ratio || 0}`,
              timestamp: new Date(),
              backtestResults: results,
              visualization: results.chart_visualization,
            };
            return [...withoutLoading, assistantMessage];
          });
        } catch (pollError) {
          // Remove loading message and show error
          setMessages(prev => {
            const withoutLoading = prev.filter(m => !m.id.startsWith('loading-'));
            const errorMessage: ChatMessage = {
              id: `error-${Date.now()}`,
              role: 'assistant',
              content: `❌ Backtest processing failed: ${pollError instanceof Error ? pollError.message : 'Unknown error'}\n\nPlease try again with a smaller dataset or different parameters.`,
              timestamp: new Date(),
            };
            return [...withoutLoading, errorMessage];
          });
        }
      } else {
        // Normal synchronous response
        const assistantMessage: ChatMessage = data.message
          ? normalizeMessage(data.message)
          : {
              id: `assistant-${Date.now()}`,
              role: 'assistant',
              content: data.assistantContent || '✅ Done.',
              timestamp: new Date(),
            };

        // Attach top-level fields so UI can render backtest widgets/downloads
        assistantMessage.backtestResults = data.backtestResults || assistantMessage.backtestResults;
        assistantMessage.visualization = data.visualization || assistantMessage.visualization;
        assistantMessage.ragInsight = data.ragInsight || assistantMessage.ragInsight;

        setMessages(prev => [...prev, assistantMessage]);

        if (data.strategyCode && data.strategyName && onStrategyGenerated) {
          onStrategyGenerated(data.strategyCode, data.strategyName);
        }
      }
    } catch (error) {
      const errorMessage: ChatMessage = {
        id: `error-${Date.now()}`,
        role: 'assistant',
        content: `Sorry, I encountered an error. Please try again. Error: ${error instanceof Error ? error.message : 'Unknown error'}`,
        timestamp: new Date(),
      };
      setMessages(prev => [...prev, errorMessage]);
    } finally {
      setIsLoading(false);
      isSendingRef.current = false;
    }
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    void sendMessage();
  };

  return (
    <div className={cn("rounded-3xl border border-white/10 bg-slate-900/70 shadow-xl shadow-blue-500/10 backdrop-blur", className)}>
      <div className="flex h-16 items-center justify-between border-b border-white/10 px-6">
        <div className="flex items-center gap-3">
          <div className="flex h-10 w-10 items-center justify-center rounded-xl bg-gradient-to-br from-blue-500/30 to-purple-500/30 border border-blue-400/30">
            <span className="text-lg">🏆</span>
          </div>
          <div>
            <p className="text-sm font-bold text-white">AI Sport Guru</p>
            <p className="text-xs text-slate-400">Stats • Analysis • Backtest Strategies</p>
          </div>
        </div>
        <div className="flex items-center gap-3">
          <div className="flex items-center gap-1 text-xs text-amber-400 bg-amber-500/10 px-2 py-1 rounded-full">
            <span>⚠️</span>
            <span>Stats Only • No Guarantees</span>
          </div>
          <div className="flex items-center gap-2 text-xs text-slate-500">
            <BookOpen className="h-4 w-4" />
            RAG + API + Cache
          </div>
        </div>
      </div>

      <div className="flex h-[500px] flex-col">
        <div className="flex-1 overflow-y-auto p-4 space-y-4">
          {messages.map((message, index) => (
            <div
              key={message.id}
              className={cn(
                "flex gap-3",
                message.role === 'user' ? "justify-end" : "justify-start"
              )}
            >
              {message.role === 'assistant' && (
                <div className="flex h-8 w-8 items-center justify-center rounded-full bg-blue-500/20">
                  <Bot className="h-4 w-4 text-blue-400" />
                </div>
              )}
              <div
                className={cn(
                  "max-w-[80%] rounded-2xl px-4 py-3 text-sm",
                  message.role === 'user'
                    ? "bg-blue-600 text-white"
                    : "bg-slate-950/50 text-slate-300 border border-white/5"
                )}
              >
                <p className="whitespace-pre-wrap">{message.content}</p>

                {/* Show RAG insights if available */}
                {message.ragInsight && message.ragInsight.relevanceScore > 0.6 && (
                  <div className="mt-3 rounded-lg bg-slate-800/50 p-3 border border-blue-500/20">
                    <div className="flex items-center gap-2 mb-2">
                      <BookOpen className="h-4 w-4 text-blue-400" />
                      <span className="text-xs font-medium text-blue-400">Betting Knowledge</span>
                      <span className="text-xs text-slate-500">
                        Relevance: {(message.ragInsight.relevanceScore * 100).toFixed(0)}%
                      </span>
                    </div>
                    <p className="text-xs text-slate-300 mb-2">{message.ragInsight.answer}</p>
                    {message.ragInsight.sources.length > 0 && (
                      <div className="flex flex-wrap gap-1">
                        {message.ragInsight.sources.slice(0, 2).map((source, idx) => (
                          <span
                            key={idx}
                            className="inline-flex items-center gap-1 text-xs bg-slate-700/50 px-2 py-1 rounded"
                          >
                            <ExternalLink className="h-3 w-3" />
                            {source.source.split('/').pop()?.split('.')[0]}
                          </span>
                        ))}
                      </div>
                    )}
                  </div>
                )}

                {/* Backtest Results with Auto-Displayed Chart */}
                {message.role === 'assistant' && message.backtestResults && (
                  <div className="mt-4 space-y-4">
                    {/* Results Header */}
                    <div className="p-4 bg-gradient-to-r from-blue-500/10 to-green-500/10 border border-blue-500/30 rounded-lg">
                      <div className="flex items-center gap-2 mb-3">
                        <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
                        <span className="text-sm font-bold text-blue-400">
                          📊 BACKTEST RESULTS
                        </span>
                        <div className="text-xs bg-green-500/20 text-green-400 px-2 py-1 rounded uppercase">
                          {message.backtestResults.sport || 'HISTORICAL DATA'}
                        </div>
                      </div>
                      
                      {/* Key Metrics Grid */}
                      <div className="grid grid-cols-2 md:grid-cols-4 gap-3 mt-3">
                        <div className="bg-slate-800/50 rounded-lg p-3 text-center">
                          <p className="text-2xl font-bold text-white">{message.backtestResults.total_trades || message.backtestResults.results?.total_trades || 0}</p>
                          <p className="text-xs text-slate-400">Total Trades</p>
                        </div>
                        <div className="bg-slate-800/50 rounded-lg p-3 text-center">
                          <p className="text-2xl font-bold text-green-400">{message.backtestResults.results?.win_rate?.toFixed(1) || message.backtestResults.win_rate || 0}%</p>
                          <p className="text-xs text-slate-400">Win Rate</p>
                        </div>
                        <div className="bg-slate-800/50 rounded-lg p-3 text-center">
                          <p className={`text-2xl font-bold ${(message.backtestResults.results?.total_profit || message.backtestResults.total_profit || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                            ${(message.backtestResults.results?.total_profit || message.backtestResults.total_profit || 0).toFixed(2)}
                          </p>
                          <p className="text-xs text-slate-400">Total Profit</p>
                        </div>
                        <div className="bg-slate-800/50 rounded-lg p-3 text-center">
                          <p className={`text-2xl font-bold ${(message.backtestResults.results?.roi || message.backtestResults.roi || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                            {(message.backtestResults.results?.roi || message.backtestResults.roi || 0).toFixed(1)}%
                          </p>
                          <p className="text-xs text-slate-400">ROI</p>
                        </div>
                      </div>
                    </div>

                    {/* AUTO-DISPLAYED CHART - Always shows when trades available */}
                    {(message.backtestResults.all_trades?.length > 0 || message.backtestResults.trades?.length > 0 || message.backtestResults.chart_visualization) && (
                      <div className="bg-slate-900/50 rounded-lg p-4 border border-blue-500/20">
                        <div className="flex items-center gap-2 mb-3">
                          <Sparkles className="h-4 w-4 text-purple-400" />
                          <span className="text-sm font-semibold text-white">Performance Chart</span>
                        </div>
                        {message.backtestResults.chart_visualization?.price_data ? (
                          <TradeChart
                            visualizationData={message.backtestResults.chart_visualization}
                            height={350}
                          />
                        ) : (
                          <BacktestChart
                            trades={(message.backtestResults.all_trades || message.backtestResults.trades || []).map((t: any, i: number) => ({
                              id: t?.id || `${message.backtestResults.market || message.backtestResults.sport || 'trade'}-${i}`,
                              date: t?.date || new Date().toISOString(),
                              outcome: t?.outcome === 'win' ? 'win' : 'loss',
                              profit: typeof t?.profit === 'number' ? t.profit : Number.parseFloat(t?.profit || '0'),
                              label: t?.game || t?.selection || t?.reason || `Trade ${i + 1}`,
                              context: t,
                            }))}
                            market={(message.backtestResults.market || message.backtestResults.sport || domain).toString().toUpperCase()}
                            title="Cumulative Profit Over Time"
                            chartType="profit"
                            height={350}
                          />
                        )}
                      </div>
                    )}

                    {/* Disclaimer */}
                    <div className="flex items-start gap-2 p-3 bg-amber-500/10 border border-amber-500/20 rounded-lg">
                      <span className="text-amber-400">⚠️</span>
                      <p className="text-xs text-amber-300">
                        <strong>Disclaimer:</strong> These are historical backtest results only. Past performance does not guarantee future results. 
                        This is for educational and research purposes – not financial or betting advice.
                      </p>
                    </div>
                  </div>
                )}

                {/* Visualization */}
                {message.visualization && (
                  <div className="mt-4 p-4 bg-slate-900/30 rounded-lg border border-blue-500/20">
                    <div className="flex items-center gap-2 mb-3">
                      <div className="w-2 h-2 bg-blue-400 rounded-full animate-pulse"></div>
                      <span className="text-sm font-medium text-blue-400">
                        {message.visualization.chartType.replace('_', ' ').toUpperCase()} Chart
                      </span>
                      <span className="text-xs text-slate-500">
                        {message.visualization.market}
                      </span>
                    </div>

                    {message.visualization.format === 'plotly' && message.visualization.chartData ? (
                      <div className="w-full">
                        <iframe
                          srcDoc={`
                            <html>
                              <head>
                                <script src="https://cdn.plot.ly/plotly-latest.min.js"></script>
                              </head>
                              <body style="margin:0;">
                                <div id="chart" style="width:100%;height:300px;"></div>
                                <script>
                                  const data = ${JSON.stringify(message.visualization.chartData)};
                                  Plotly.newPlot('chart', data);
                                </script>
                              </body>
                            </html>
                          `}
                          className="w-full h-80 border-0 rounded"
                          title="Backtest Chart"
                        />
                      </div>
                    ) : message.visualization.imageBase64 ? (
                      <div className="w-full">
                        <img
                          src={`data:image/png;base64,${message.visualization.imageBase64}`}
                          alt="Backtest Chart"
                          className="w-full rounded border border-slate-700"
                        />
                      </div>
                    ) : message.visualization.trades ? (
                      <div className="w-full">
                        <BacktestChart
                          trades={message.visualization.trades}
                          market={message.visualization.market}
                          chartType={message.visualization.chartType === 'profit_over_time' ? 'profit' : 'returns'}
                          height={300}
                        />
                      </div>
                    ) : (
                      <div className="text-center text-slate-400 py-8">
                        Chart visualization not available
                      </div>
                    )}
                  </div>
                )}

                {/* Download & Actions Panel */}
                {message.role === 'assistant' && message.backtestResults && (
                  <div className="mt-4 p-4 bg-gradient-to-r from-slate-800/80 to-slate-900/80 border border-slate-600 rounded-xl">
                    <div className="flex items-center justify-between mb-4">
                      <h4 className="text-white font-semibold flex items-center gap-2">
                        <Download className="h-5 w-5 text-green-400" />
                        Download & Export
                      </h4>
                      <span className="text-xs text-slate-400 bg-slate-700 px-2 py-1 rounded">
                        {message.backtestResults.total_trades || message.backtestResults.results?.total_trades || 0} trades
                      </span>
                    </div>
                    
                    {/* Download Buttons - More Prominent */}
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-2 mb-4">
                      <button
                        onClick={() => downloadReport('csv', index)}
                        className="flex flex-col items-center gap-1 p-3 bg-green-600 hover:bg-green-500 text-white rounded-lg transition-all hover:scale-105 shadow-lg"
                      >
                        <FileSpreadsheet className="h-6 w-6" />
                        <span className="text-xs font-medium">CSV Report</span>
                      </button>
                      <button
                        onClick={() => downloadReport('txt', index)}
                        className="flex flex-col items-center gap-1 p-3 bg-blue-600 hover:bg-blue-500 text-white rounded-lg transition-all hover:scale-105 shadow-lg"
                      >
                        <FileText className="h-6 w-6" />
                        <span className="text-xs font-medium">Text Report</span>
                      </button>
                      <button
                        onClick={() => toggleTradeHistory(index)}
                        className="flex flex-col items-center gap-1 p-3 bg-orange-600 hover:bg-orange-500 text-white rounded-lg transition-all hover:scale-105 shadow-lg"
                      >
                        <FileText className="h-6 w-6" />
                        <span className="text-xs font-medium">Trade Log</span>
                      </button>
                      <button
                        onClick={() => showTradeChart(index)}
                        className="flex flex-col items-center gap-1 p-3 bg-purple-600 hover:bg-purple-500 text-white rounded-lg transition-all hover:scale-105 shadow-lg"
                      >
                        <Sparkles className="h-6 w-6" />
                        <span className="text-xs font-medium">Full Chart</span>
                      </button>
                    </div>
                    
                    <p className="text-slate-400 text-xs text-center">
                      📁 Reports include: all trades, P&L breakdown, risk metrics, and strategy analysis
                    </p>
                  </div>
                )}

                {/* Trade Chart Visualization */}
                {message.role === 'assistant' && message.backtestResults?.chart_visualization && showChart[index] && (
                  <div className="mt-4">
                    {message.backtestResults.chart_visualization?.price_data ? (
                    <TradeChart
                      visualizationData={message.backtestResults.chart_visualization}
                      height={500}
                    />
                    ) : (
                      <BacktestChart
                        trades={(message.backtestResults.all_trades || message.backtestResults.trades || []).map((t: any, i: number) => ({
                          id: t?.id || `${message.backtestResults.market || message.backtestResults.sport || 'trade'}-${i}`,
                          date: t?.date || new Date().toISOString(),
                          outcome: t?.outcome === 'win' ? 'win' : 'loss',
                          profit: typeof t?.profit === 'number' ? t.profit : Number.parseFloat(t?.profit || '0'),
                          label: t?.game || t?.selection || t?.reason || `Trade ${i + 1}`,
                          context: t,
                        }))}
                        market={(message.backtestResults.market || message.backtestResults.sport || domain).toString().toUpperCase()}
                        title="Trade Profit (Cumulative)"
                        chartType="profit"
                        height={400}
                      />
                    )}
                  </div>
                )}

                {/* Trade History Table */}
                {message.role === 'assistant' && message.backtestResults?.all_trades && showTradeHistory[index] && (
                  <div className="mt-4">
                    <div className="bg-slate-800 rounded-lg p-4 border border-slate-600">
                      <div className="flex items-center justify-between mb-4">
                        <h4 className="text-lg font-semibold text-white flex items-center gap-2">
                          <FileText className="h-5 w-5 text-orange-400" />
                          Trade History ({message.backtestResults.all_trades.length} trades)
                        </h4>
                        <button
                          onClick={() => toggleTradeHistory(index)}
                          className="text-slate-400 hover:text-white text-sm"
                        >
                          ✕ Close
                        </button>
                      </div>

                      <div className="overflow-x-auto">
                        <table className="w-full text-sm">
                          <thead>
                            <tr className="border-b border-slate-600">
                              <th className="text-left py-2 px-3 text-slate-300 font-semibold">#</th>
                              <th className="text-left py-2 px-3 text-slate-300 font-semibold">Date</th>
                              <th className="text-left py-2 px-3 text-slate-300 font-semibold">Action</th>
                              <th className="text-left py-2 px-3 text-slate-300 font-semibold">Outcome</th>
                              <th className="text-right py-2 px-3 text-slate-300 font-semibold">Stake ($)</th>
                              <th className="text-right py-2 px-3 text-slate-300 font-semibold">P&L ($)</th>
                              <th className="text-right py-2 px-3 text-slate-300 font-semibold">Return (%)</th>
                            </tr>
                          </thead>
                          <tbody>
                            {message.backtestResults.all_trades.slice(0, 100).map((trade: any, tradeIndex: number) => {
                              const returnPct = trade.stake ? ((trade.profit / trade.stake) * 100).toFixed(2) : '0.00';
                              const isWin = trade.outcome === 'win';
                              const isLoss = trade.outcome === 'loss';

                              return (
                                <tr key={tradeIndex} className="border-b border-slate-700/50 hover:bg-slate-700/30">
                                  <td className="py-2 px-3 text-slate-400">{tradeIndex + 1}</td>
                                  <td className="py-2 px-3 text-slate-300" suppressHydrationWarning>
                                    {new Date(trade.date).toLocaleDateString()}
                                  </td>
                                  <td className="py-2 px-3">
                                    <span className={`font-semibold ${trade.action === 'BUY' ? 'text-green-400' : 'text-red-400'}`}>
                                      {trade.action}
                                    </span>
                                  </td>
                                  <td className="py-2 px-3">
                                    <span className={`font-semibold ${isWin ? 'text-green-400' : isLoss ? 'text-red-400' : 'text-yellow-400'}`}>
                                      {trade.outcome?.toUpperCase() || 'PENDING'}
                                    </span>
                                  </td>
                                  <td className="py-2 px-3 text-right text-slate-300">
                                    ${trade.stake?.toFixed(2) || '0.00'}
                                  </td>
                                  <td className={`py-2 px-3 text-right font-semibold ${trade.profit >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                    {trade.profit >= 0 ? '+' : ''}${trade.profit?.toFixed(2) || '0.00'}
                                  </td>
                                  <td className={`py-2 px-3 text-right font-semibold ${parseFloat(returnPct) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                    {parseFloat(returnPct) >= 0 ? '+' : ''}{returnPct}%
                                  </td>
                                </tr>
                              );
                            })}
                          </tbody>
                        </table>

                        {message.backtestResults.all_trades.length > 100 && (
                          <div className="mt-4 text-center text-slate-400 text-sm">
                            Showing first 100 trades. Download full report to see all {message.backtestResults.all_trades.length} trades.
                          </div>
                        )}
                      </div>

                      {/* Trade Summary Stats */}
                      <div className="mt-4 grid grid-cols-2 md:grid-cols-4 gap-4 pt-4 border-t border-slate-600">
                        <div className="text-center">
                          <div className="text-lg font-bold text-green-400">
                            {message.backtestResults.all_trades.filter((t: any) => t.outcome === 'win').length}
                          </div>
                          <div className="text-xs text-slate-400">Winning Trades</div>
                        </div>
                        <div className="text-center">
                          <div className="text-lg font-bold text-red-400">
                            {message.backtestResults.all_trades.filter((t: any) => t.outcome === 'loss').length}
                          </div>
                          <div className="text-xs text-slate-400">Losing Trades</div>
                        </div>
                        <div className="text-center">
                          <div className="text-lg font-bold text-blue-400">
                            ${(message.backtestResults.all_trades.reduce((sum: number, t: any) => sum + (t.profit || 0), 0)).toFixed(2)}
                          </div>
                          <div className="text-xs text-slate-400">Total P&L</div>
                        </div>
                        <div className="text-center">
                          <div className="text-lg font-bold text-purple-400">
                            {message.backtestResults.results?.win_rate?.toFixed(1) || '0.0'}%
                          </div>
                          <div className="text-xs text-slate-400">Win Rate</div>
                        </div>
                      </div>
                    </div>
                  </div>
                )}

                <p className="mt-2 text-xs opacity-60" suppressHydrationWarning>
                  {message.timestamp.toLocaleTimeString()}
                </p>
              </div>
              {message.role === 'user' && (
                <div className="flex h-8 w-8 items-center justify-center rounded-full bg-slate-700">
                  <User className="h-4 w-4" />
                </div>
              )}
            </div>
          ))}
          {isLoading && (
            <div className="flex gap-3">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-gradient-to-br from-blue-500/20 to-purple-500/20">
                <Loader2 className="h-4 w-4 animate-spin text-blue-400" />
              </div>
              <div className="rounded-2xl bg-slate-950/50 px-4 py-3 text-sm text-slate-300 border border-white/5">
                <p className="flex items-center gap-2">
                  <span>🏆</span>
                  AI Sport Guru is analyzing data and running calculations…
                </p>
              </div>
            </div>
          )}
          <div ref={messagesEndRef} />
        </div>

        <div className="border-t border-white/10 p-4">
          <form onSubmit={handleSubmit} className="flex gap-3">
            <TextareaAutosize
              ref={textareaRef}
              value={input}
              onChange={(e) => setInput(e.target.value)}
              onKeyDown={(e) => {
                // Send on Enter, allow newline with Shift+Enter
                if (e.key === 'Enter' && !e.shiftKey) {
                  e.preventDefault();
                  void sendMessage();
                }
              }}
              placeholder={`Describe your ${domain} strategy...`}
              className="flex-1 rounded-2xl border border-white/10 bg-slate-950/40 px-4 py-3 text-sm text-slate-100 placeholder-slate-500 outline-none transition focus:border-blue-400 resize-none"
              minRows={1}
              maxRows={4}
              disabled={isLoading}
            />
            <button
              type="submit"
              disabled={!input.trim() || isLoading}
              onClick={() => void sendMessage()}
              aria-label="Send message"
              className="flex h-12 w-12 items-center justify-center rounded-2xl bg-blue-600 text-white transition hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              <Send className="h-5 w-5" />
            </button>
          </form>
        </div>
      </div>
    </div>
  );
}
