/**
 * Interactive Backtest Chart Component using D3.js
 * Displays backtest results with interactive features
 */

'use client';

import { useEffect, useRef, useState } from 'react';
import * as d3 from 'd3';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Download, ZoomIn, ZoomOut, RotateCcw } from 'lucide-react';

interface Trade {
  id: string;
  date: string;
  outcome: 'win' | 'loss';
  profit: number;
  label: string;
  context?: any;
}

interface BacktestChartProps {
  trades: Trade[];
  market?: string;
  title?: string;
  chartType?: 'profit' | 'returns' | 'drawdown' | 'comparison';
  height?: number;
}

export default function BacktestChart({
  trades,
  market = 'General',
  title,
  chartType = 'profit',
  height = 400
}: BacktestChartProps) {
  const svgRef = useRef<SVGSVGElement>(null);
  const tooltipRef = useRef<HTMLDivElement | null>(null);
  const [selectedChartType, setSelectedChartType] = useState(chartType);
  const [zoomLevel, setZoomLevel] = useState(1);
  const [isLoading, setIsLoading] = useState(false);

  // Cleanup tooltip on unmount
  useEffect(() => {
    return () => {
      if (tooltipRef.current) {
        tooltipRef.current.remove();
        tooltipRef.current = null;
      }
    };
  }, []);

  useEffect(() => {
    if (trades.length > 0) {
      drawChart();
    }
  }, [trades, selectedChartType, zoomLevel]);

  const drawChart = () => {
    if (!svgRef.current || trades.length === 0) return;

    // Clear previous chart
    d3.select(svgRef.current).selectAll('*').remove();

    const margin = { top: 20, right: 80, bottom: 60, left: 80 };
    const width = (svgRef.current.parentElement?.clientWidth || 800) - margin.left - margin.right;
    const chartHeight = height - margin.top - margin.bottom;

    const svg = d3.select(svgRef.current)
      .attr('width', width + margin.left + margin.right)
      .attr('height', chartHeight + margin.top + margin.bottom);

    const g = svg.append('g')
      .attr('transform', `translate(${margin.left},${margin.top})`);

    // Prepare data based on chart type
    const processedData = processDataForChart(trades, selectedChartType);

    // Create scales
    const xScale = d3.scaleTime()
      .domain(d3.extent(processedData, d => new Date(d.date)) as [Date, Date])
      .range([0, width]);

    const yScale = d3.scaleLinear()
      .domain([
        d3.min(processedData, d => d.value) as number * 1.1,
        d3.max(processedData, d => d.value) as number * 1.1
      ])
      .range([chartHeight, 0]);

    // Add zoom behavior
    const zoom = d3.zoom<SVGSVGElement, unknown>()
      .scaleExtent([0.5, 5])
      .on('zoom', (event) => {
        const newXScale = event.transform.rescaleX(xScale);
        const newYScale = event.transform.rescaleY(yScale);

        // Update axes
        g.select('.x-axis').call(d3.axisBottom(newXScale) as any);
        g.select('.y-axis').call(d3.axisLeft(newYScale) as any);

        // Update chart elements based on type
        updateChartElements(g, processedData, newXScale, newYScale, selectedChartType);
      });

    svg.call(zoom);

    // Add axes
    g.append('g')
      .attr('class', 'x-axis')
      .attr('transform', `translate(0,${chartHeight})`)
      .call(d3.axisBottom(xScale));

    g.append('g')
      .attr('class', 'y-axis')
      .call(d3.axisLeft(yScale));

    // Add axis labels
    g.append('text')
      .attr('class', 'x-label')
      .attr('text-anchor', 'middle')
      .attr('x', width / 2)
      .attr('y', chartHeight + 40)
      .text('Date')
      .style('font-size', '12px')
      .style('fill', '#666');

    g.append('text')
      .attr('class', 'y-label')
      .attr('text-anchor', 'middle')
      .attr('transform', 'rotate(-90)')
      .attr('x', -chartHeight / 2)
      .attr('y', -60)
      .text(getYAxisLabel(selectedChartType))
      .style('font-size', '12px')
      .style('fill', '#666');

    // Draw chart elements
    updateChartElements(g, processedData, xScale, yScale, selectedChartType);

    // Add grid lines
    addGridLines(g, width, chartHeight, xScale, yScale);

    // Add tooltips
    addTooltips(g, processedData);
  };

  const processDataForChart = (trades: Trade[], type: string) => {
    const sortedTrades = [...trades].sort((a, b) =>
      new Date(a.date).getTime() - new Date(b.date).getTime()
    );

    switch (type) {
      case 'profit':
        let cumulativeProfit = 0;
        return sortedTrades.map(trade => {
          cumulativeProfit += trade.profit;
          return {
            ...trade,
            value: cumulativeProfit,
            date: trade.date
          };
        });

      case 'returns':
        return sortedTrades.map(trade => ({
          ...trade,
          value: trade.profit,
          date: trade.date
        }));

      case 'drawdown':
        let peak = 0;
        let drawdownCumulative = 0;
        return sortedTrades.map(trade => {
          drawdownCumulative += trade.profit;
          peak = Math.max(peak, drawdownCumulative);
          const drawdown = drawdownCumulative - peak;
          return {
            ...trade,
            value: drawdown,
            date: trade.date
          };
        });

      default:
        return sortedTrades.map(trade => ({
          ...trade,
          value: trade.profit,
          date: trade.date
        }));
    }
  };

  const updateChartElements = (g: any, data: any[], xScale: any, yScale: any, type: string) => {
    // Clear existing elements
    g.selectAll('.chart-element').remove();

    const elements = g.selectAll('.chart-element')
      .data(data)
      .enter()
      .append('g')
      .attr('class', 'chart-element');

    switch (type) {
      case 'profit':
        // Line chart for cumulative profit
        const line = d3.line()
          .x((d: any) => xScale(new Date(d.date)))
          .y((d: any) => yScale(d.value))
          .curve(d3.curveMonotoneX);

        elements.append('path')
          .datum(data)
          .attr('fill', 'none')
          .attr('stroke', '#667eea')
          .attr('stroke-width', 2)
          .attr('d', line);

        // Add area fill
        const area = d3.area()
          .x((d: any) => xScale(new Date(d.date)))
          .y0(yScale(0))
          .y1((d: any) => yScale(d.value))
          .curve(d3.curveMonotoneX);

        g.append('path')
          .datum(data)
          .attr('fill', 'rgba(102, 126, 234, 0.1)')
          .attr('d', area);

        break;

      case 'returns':
        // Bar chart for individual returns
        elements.append('rect')
          .attr('x', (d: any) => xScale(new Date(d.date)) - 3)
          .attr('y', (d: any) => d.value >= 0 ? yScale(d.value) : yScale(0))
          .attr('width', 6)
          .attr('height', (d: any) => Math.abs(yScale(d.value) - yScale(0)))
          .attr('fill', (d: any) => d.outcome === 'win' ? '#27ae60' : '#e74c3c')
          .attr('opacity', 0.8);
        break;

      case 'drawdown':
        // Area chart for drawdown
        const drawdownArea = d3.area()
          .x((d: any) => xScale(new Date(d.date)))
          .y0(yScale(0))
          .y1((d: any) => yScale(d.value))
          .curve(d3.curveMonotoneX);

        g.append('path')
          .datum(data)
          .attr('fill', 'rgba(231, 76, 60, 0.3)')
          .attr('d', drawdownArea);

        elements.append('circle')
          .attr('cx', (d: any) => xScale(new Date(d.date)))
          .attr('cy', (d: any) => yScale(d.value))
          .attr('r', 2)
          .attr('fill', '#e74c3c');
        break;
    }
  };

  const addGridLines = (g: any, width: number, height: number, xScale: any, yScale: any) => {
    // Vertical grid lines
    g.selectAll('.grid-vertical')
      .data(xScale.ticks(10))
      .enter()
      .append('line')
      .attr('class', 'grid-vertical')
      .attr('x1', (d: any) => xScale(d))
      .attr('x2', (d: any) => xScale(d))
      .attr('y1', 0)
      .attr('y2', height)
      .attr('stroke', '#e0e0e0')
      .attr('stroke-width', 0.5)
      .attr('opacity', 0.3);

    // Horizontal grid lines
    g.selectAll('.grid-horizontal')
      .data(yScale.ticks(5))
      .enter()
      .append('line')
      .attr('class', 'grid-horizontal')
      .attr('x1', 0)
      .attr('x2', width)
      .attr('y1', (d: any) => yScale(d))
      .attr('y2', (d: any) => yScale(d))
      .attr('stroke', '#e0e0e0')
      .attr('stroke-width', 0.5)
      .attr('opacity', 0.3);
  };

  const addTooltips = (g: any, data: any[]) => {
    // Remove existing tooltip if any to prevent DOM leaks
    if (tooltipRef.current) {
      tooltipRef.current.remove();
    }
    
    // Also clean up any orphaned tooltips from previous renders
    d3.selectAll('.chart-tooltip').remove();

    // Create new tooltip
    const tooltipDiv = document.createElement('div');
    tooltipDiv.className = 'chart-tooltip';
    tooltipDiv.style.cssText = `
      position: absolute;
      visibility: hidden;
      background: rgba(0, 0, 0, 0.8);
      color: white;
      padding: 8px;
      border-radius: 4px;
      font-size: 12px;
      pointer-events: none;
      z-index: 1000;
    `;
    document.body.appendChild(tooltipDiv);
    tooltipRef.current = tooltipDiv;

    const tooltip = d3.select(tooltipDiv);

    g.selectAll('.chart-element')
      .on('mouseover', function (event: any, d: any) {
        tooltip.style('visibility', 'visible')
          .html(`
            <div><strong>${d.label}</strong></div>
            <div>Date: ${new Date(d.date).toLocaleDateString()}</div>
            <div>Profit: $${d.profit?.toFixed(2) || 'N/A'}</div>
            <div>Outcome: ${d.outcome}</div>
            <div>Value: ${d.value?.toFixed(2) || 'N/A'}</div>
          `);
      })
      .on('mousemove', function (event: any) {
        tooltip.style('top', (event.pageY - 10) + 'px')
          .style('left', (event.pageX + 10) + 'px');
      })
      .on('mouseout', function () {
        tooltip.style('visibility', 'hidden');
      });
  };

  const getYAxisLabel = (type: string): string => {
    switch (type) {
      case 'profit': return 'Cumulative Profit ($)';
      case 'returns': return 'Daily Return ($)';
      case 'drawdown': return 'Drawdown ($)';
      default: return 'Value';
    }
  };

  const handleZoomIn = () => {
    setZoomLevel(prev => Math.min(prev * 1.2, 5));
  };

  const handleZoomOut = () => {
    setZoomLevel(prev => Math.max(prev / 1.2, 0.5));
  };

  const handleReset = () => {
    setZoomLevel(1);
  };

  const exportChart = () => {
    if (svgRef.current) {
      const svgData = new XMLSerializer().serializeToString(svgRef.current);
      const svgBlob = new Blob([svgData], { type: 'image/svg+xml;charset=utf-8' });
      const svgUrl = URL.createObjectURL(svgBlob);

      const downloadLink = document.createElement('a');
      downloadLink.href = svgUrl;
      downloadLink.download = `backtest-chart-${market}-${selectedChartType}.svg`;
      document.body.appendChild(downloadLink);
      downloadLink.click();
      document.body.removeChild(downloadLink);

      URL.revokeObjectURL(svgUrl);
    }
  };

  // Calculate summary statistics
  const totalTrades = trades.length;
  const winningTrades = trades.filter(t => t.outcome === 'win').length;
  const losingTrades = trades.filter(t => t.outcome === 'loss').length;
  const winRate = totalTrades > 0 ? (winningTrades / totalTrades * 100).toFixed(1) : '0';
  const totalProfit = trades.reduce((sum, t) => sum + t.profit, 0).toFixed(2);
  const avgProfit = totalTrades > 0 ? (parseFloat(totalProfit) / totalTrades).toFixed(2) : '0';

  return (
    <Card className="w-full">
      <CardHeader>
        <div className="flex items-center justify-between">
          <div>
            <CardTitle className="flex items-center gap-2">
              {title || `${market} Backtest Analysis`}
              <Badge variant="outline">{totalTrades} trades</Badge>
            </CardTitle>
            <CardDescription>
              Win Rate: {winRate}% | Total P&L: ${totalProfit} | Avg Trade: ${avgProfit}
            </CardDescription>
          </div>

          <div className="flex items-center gap-2">
            <select
              value={selectedChartType}
              onChange={(e) => setSelectedChartType(e.target.value as 'profit' | 'returns' | 'drawdown' | 'comparison')}
              className="w-32 h-9 rounded-md border border-input bg-background px-3 py-1 text-sm"
            >
              <option value="profit">Profit</option>
              <option value="returns">Returns</option>
              <option value="drawdown">Drawdown</option>
            </select>

            <Button variant="outline" size="sm" onClick={handleZoomIn}>
              <ZoomIn className="h-4 w-4" />
            </Button>
            <Button variant="outline" size="sm" onClick={handleZoomOut}>
              <ZoomOut className="h-4 w-4" />
            </Button>
            <Button variant="outline" size="sm" onClick={handleReset}>
              <RotateCcw className="h-4 w-4" />
            </Button>
            <Button variant="outline" size="sm" onClick={exportChart}>
              <Download className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent>
        <div className="w-full overflow-x-auto">
          <svg
            ref={svgRef}
            className="w-full border border-gray-200 rounded"
            style={{ minHeight: height }}
          />
        </div>

        <div className="mt-4 grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
          <div className="text-center">
            <div className="font-semibold text-green-600">{winningTrades}</div>
            <div className="text-gray-600">Winning Trades</div>
          </div>
          <div className="text-center">
            <div className="font-semibold text-red-600">{losingTrades}</div>
            <div className="text-gray-600">Losing Trades</div>
          </div>
          <div className="text-center">
            <div className="font-semibold text-blue-600">{winRate}%</div>
            <div className="text-gray-600">Win Rate</div>
          </div>
          <div className="text-center">
            <div className={`font-semibold ${parseFloat(totalProfit) >= 0 ? 'text-green-600' : 'text-red-600'}`}>
              ${totalProfit}
            </div>
            <div className="text-gray-600">Total P&L</div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}
