'use client';

import React, { useState } from 'react';
import Link from 'next/link';
import { ArrowLeft, MessageSquare, Mail, Phone, Clock, Send, CheckCircle } from 'lucide-react';
import ProtectedRoute from '@/components/ProtectedRoute';
import LiveChatWidget from '@/components/LiveChatWidget';

export default function SupportPage() {
  const [showChat, setShowChat] = useState(false);
  const [contactForm, setContactForm] = useState({
    name: '',
    email: '',
    subject: '',
    message: '',
    priority: 'normal'
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitted, setSubmitted] = useState(false);
  const [submitError, setSubmitError] = useState<string | null>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setSubmitError(null);

    const type =
      contactForm.priority === 'urgent' || contactForm.priority === 'high'
        ? 'bug'
        : contactForm.priority === 'low'
          ? 'question'
          : 'other';

    try {
      const response = await fetch('/api/feedback', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({
          type,
          title: contactForm.subject,
          description: `Priority: ${contactForm.priority}\n\n${contactForm.message}`,
        }),
      });

      const data = await response.json().catch(() => ({}));
      if (!response.ok) {
        throw new Error(data?.error || data?.message || 'Failed to submit support request');
      }

      setSubmitted(true);
    } catch (err) {
      setSubmitError(err instanceof Error ? err.message : 'Failed to submit support request');
    } finally {
    setIsSubmitting(false);
    }

    // Reset form after showing success
    setTimeout(() => {
      setSubmitted(false);
      setContactForm({
        name: '',
        email: '',
        subject: '',
        message: '',
        priority: 'normal'
      });
    }, 3000);
  };

  const handleInputChange = (field: string, value: string) => {
    setContactForm(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const supportOptions = [
    {
      icon: <MessageSquare className="h-6 w-6" />,
      title: "Live Chat",
      description: "Get instant help from our support team",
      availability: "Available 24/7",
      responseTime: "Instant",
      action: "Start Chat",
      color: "blue"
    },
    {
      icon: <Mail className="h-6 w-6" />,
      title: "Email Support",
      description: "Send us a detailed message about your issue",
      availability: "Mon-Fri, 9AM-6PM EST",
      responseTime: "Within 24 hours",
      action: "Send Email",
      color: "green"
    },
    {
      icon: <Phone className="h-6 w-6" />,
      title: "Phone Support",
      description: "Speak directly with a support specialist",
      availability: "Mon-Fri, 10AM-4PM EST",
      responseTime: "Immediate",
      action: "Call Now",
      color: "purple"
    }
  ];

  return (
    <ProtectedRoute>
    <div className="min-h-screen bg-slate-950">
      {/* Header */}
      <header className="bg-slate-900/50 border-b border-white/10">
        <div className="mx-auto max-w-7xl px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Link href="/dashboard" className="text-slate-400 hover:text-white transition">
                <ArrowLeft className="h-5 w-5" />
              </Link>
              <div>
                <h1 className="text-2xl font-bold text-white">Contact Support</h1>
                <p className="text-slate-400 text-sm">We're here to help you succeed</p>
              </div>
            </div>
          </div>
        </div>
      </header>

      <div className="mx-auto max-w-7xl px-6 py-8">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* Support Options */}
          <div className="lg:col-span-1 space-y-6">
            {supportOptions.map((option, index) => (
              <div key={index} className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                <div className="flex items-center gap-3 mb-4">
                  <div className={`w-10 h-10 bg-${option.color}-500/20 rounded-lg flex items-center justify-center text-${option.color}-400`}>
                    {option.icon}
                  </div>
                  <h3 className="text-lg font-semibold text-white">{option.title}</h3>
                </div>

                <p className="text-slate-400 text-sm mb-4">{option.description}</p>

                <div className="space-y-2 mb-4">
                  <div className="flex items-center gap-2 text-sm">
                    <Clock className="h-4 w-4 text-slate-400" />
                    <span className="text-slate-400">{option.availability}</span>
                  </div>
                  <div className="flex items-center gap-2 text-sm">
                    <CheckCircle className="h-4 w-4 text-green-400" />
                    <span className="text-green-400">{option.responseTime}</span>
                  </div>
                </div>

                <button
                  onClick={() => {
                    if (option.title === 'Live Chat') {
                      setShowChat(true);
                    } else if (option.title === 'Email Support') {
                      window.location.href = 'mailto:support@eventheodds.ai';
                    } else if (option.title === 'Phone Support') {
                      alert('Phone support coming soon! Please use Live Chat or Email for now.');
                    }
                  }}
                  className={`w-full bg-${option.color}-600 hover:bg-${option.color}-500 text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2`}
                >
                  {option.action}
                  <Send className="h-4 w-4" />
                </button>
              </div>
            ))}
          </div>

          {/* Contact Form */}
          <div className="lg:col-span-2">
            <div className="bg-slate-900/50 border border-white/10 rounded-xl p-8">
              <div className="mb-6">
                <h2 className="text-2xl font-bold text-white mb-2">Send us a Message</h2>
                <p className="text-slate-400">
                  Describe your issue in detail and we'll get back to you as soon as possible.
                </p>
              </div>

              {submitError && (
                <div className="mb-6 rounded-lg border border-red-500/20 bg-red-500/10 p-4 text-red-300">
                  {submitError}
                </div>
              )}

              {submitted ? (
                <div className="text-center py-12">
                  <div className="w-16 h-16 bg-green-500/20 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle className="h-8 w-8 text-green-400" />
                  </div>
                  <h3 className="text-xl font-semibold text-white mb-2">Message Sent!</h3>
                  <p className="text-slate-400">
                    Thank you for contacting us. We'll respond to your message within 24 hours.
                  </p>
                </div>
              ) : (
                <form onSubmit={handleSubmit} className="space-y-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-slate-300 mb-2">
                        Full Name
                      </label>
                      <input
                        type="text"
                        required
                        value={contactForm.name}
                        onChange={(e) => handleInputChange('name', e.target.value)}
                        className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none"
                        placeholder="Your full name"
                      />
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-slate-300 mb-2">
                        Email Address
                      </label>
                      <input
                        type="email"
                        required
                        value={contactForm.email}
                        onChange={(e) => handleInputChange('email', e.target.value)}
                        className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none"
                        placeholder="your@email.com"
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      Priority Level
                    </label>
                    <select
                      value={contactForm.priority}
                      onChange={(e) => handleInputChange('priority', e.target.value)}
                      className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white focus:border-blue-500 focus:outline-none"
                    >
                      <option value="low">Low - General question</option>
                      <option value="normal">Normal - Feature request or issue</option>
                      <option value="high">High - Urgent technical problem</option>
                      <option value="urgent">Urgent - Service disruption</option>
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      Subject
                    </label>
                    <input
                      type="text"
                      required
                      value={contactForm.subject}
                      onChange={(e) => handleInputChange('subject', e.target.value)}
                      className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none"
                      placeholder="Brief description of your issue"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-slate-300 mb-2">
                      Message
                    </label>
                    <textarea
                      required
                      value={contactForm.message}
                      onChange={(e) => handleInputChange('message', e.target.value)}
                      rows={8}
                      className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none resize-none"
                      placeholder="Please provide as much detail as possible about your issue, including steps to reproduce if it's a bug, or what you're trying to accomplish."
                    />
                  </div>

                  <button
                    type="submit"
                    disabled={isSubmitting}
                    className="w-full bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
                  >
                    {isSubmitting ? (
                      <>
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                        Sending Message...
                      </>
                    ) : (
                      <>
                        <Send className="h-4 w-4" />
                        Send Message
                      </>
                    )}
                  </button>
                </form>
              )}
            </div>

            {/* FAQ Section */}
            <div className="mt-8 bg-slate-900/50 border border-white/10 rounded-xl p-6">
              <h3 className="text-lg font-semibold text-white mb-4">Quick Answers</h3>
              <div className="space-y-3">
                <Link href="/help" className="block text-blue-400 hover:text-blue-300 text-sm">
                  📚 Browse our Help Center
                </Link>
                <Link href="/dashboard?tab=feedback" className="block text-blue-400 hover:text-blue-300 text-sm">
                  💡 Submit feedback or feature requests
                </Link>
                <Link href="/dashboard?tab=updates" className="block text-blue-400 hover:text-blue-300 text-sm">
                  📢 Check latest platform updates
                </Link>
                <Link href="/community" className="block text-blue-400 hover:text-blue-300 text-sm">
                  👥 Join the community forum
                </Link>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
    
    {/* Live Chat Widget */}
    <LiveChatWidget 
      isOpen={showChat} 
      onClose={() => setShowChat(false)} 
    />
    </ProtectedRoute>
  );
}
