'use client';

import { useEffect, useMemo, useState } from 'react';
import Link from 'next/link';
import { Copy, Loader2, RefreshCw, Search, Shield } from 'lucide-react';
import ProtectedRoute from '@/components/ProtectedRoute';
import { StrategyPlayground } from '@/components/StrategyPlayground';

export default function StrategiesPage() {
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [strategies, setStrategies] = useState<any[]>([]);
  const [selectedId, setSelectedId] = useState<string | null>(null);
  const [search, setSearch] = useState('');

  const selected = useMemo(
    () => strategies.find((s) => s.id === selectedId) || null,
    [strategies, selectedId]
  );

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    if (!q) return strategies;
    return strategies.filter((s) =>
      (s.name || '').toLowerCase().includes(q) ||
      (s.description || '').toLowerCase().includes(q) ||
      (s.tags || []).some((t: string) => t.toLowerCase().includes(q))
    );
  }, [strategies, search]);

  async function fetchStrategies() {
    setLoading(true);
    setError(null);
    try {
      const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
      const res = await fetch('/api/user/strategies?limit=50', {
        headers: {
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
      });
      const data = await res.json();
      if (!res.ok) throw new Error(data?.error || 'Failed to load strategies');
      setStrategies(data.strategies || []);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to load strategies');
    } finally {
      setLoading(false);
    }
  }

  useEffect(() => {
    void fetchStrategies();
  }, []);

  async function copyToClipboard(text: string) {
    try {
      await navigator.clipboard.writeText(text);
    } catch (e) {
      setError('Failed to copy to clipboard.');
    }
  }

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-slate-950">
        <div className="max-w-7xl mx-auto px-6 py-12 space-y-10">
          <div className="flex flex-col md:flex-row md:items-end md:justify-between gap-6">
            <div>
              <h1 className="text-4xl font-bold text-white mb-2">Strategy Library</h1>
              <p className="text-slate-400 max-w-2xl">
                Create sports betting strategies, save them, and reuse them in chat backtests.
              </p>
            </div>
            <div className="flex gap-3">
              <button
                onClick={() => void fetchStrategies()}
                className="inline-flex items-center gap-2 rounded-xl border border-white/10 bg-slate-900/50 px-4 py-2 text-white hover:bg-white/5"
                disabled={loading}
              >
                {loading ? <Loader2 className="h-4 w-4 animate-spin" /> : <RefreshCw className="h-4 w-4" />}
                Refresh
              </button>
              <Link
                href="/dashboard"
                className="inline-flex items-center gap-2 rounded-xl bg-blue-600 px-4 py-2 text-white font-semibold hover:bg-blue-500"
              >
                Run backtests in chat
              </Link>
            </div>
          </div>

          <StrategyPlayground />

          {error && (
            <div className="rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
              {error}
            </div>
          )}

          <div className="grid lg:grid-cols-3 gap-6">
            <div className="lg:col-span-1 rounded-2xl border border-white/10 bg-slate-900/50 p-4">
              <div className="flex items-center gap-2 mb-4">
                <Search className="h-4 w-4 text-slate-400" />
                <input
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  placeholder="Search saved strategies…"
                  className="w-full rounded-xl border border-white/10 bg-slate-950/40 px-3 py-2 text-sm text-white placeholder-slate-500"
                />
              </div>

              {loading ? (
                <div className="flex items-center gap-3 text-slate-300">
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Loading strategies…
                </div>
              ) : filtered.length === 0 ? (
                <div className="text-slate-400 text-sm">
                  No saved strategies yet. Use the playground above and enable “Save to Strategy Library”.
                </div>
              ) : (
                <div className="space-y-2">
                  {filtered.map((s) => (
                    <button
                      key={s.id}
                      onClick={() => setSelectedId(s.id)}
                      className={`w-full rounded-xl border px-4 py-3 text-left transition ${
                        selectedId === s.id
                          ? 'border-blue-500/40 bg-blue-500/10'
                          : 'border-white/10 bg-slate-950/30 hover:bg-white/5'
                      }`}
                    >
                      <div className="flex items-center justify-between gap-3">
                        <div className="text-white font-semibold text-sm truncate">{s.name}</div>
                        <div className="text-xs text-slate-400" suppressHydrationWarning>
                          {new Date(s.updatedAt).toLocaleDateString()}
                        </div>
                      </div>
                      <div className="mt-1 text-xs text-slate-400 truncate">
                        {s.description || '—'}
                      </div>
                      <div className="mt-2 flex flex-wrap gap-2">
                        {(s.tags || []).slice(0, 3).map((t: string) => (
                          <span key={t} className="text-xs rounded-full border border-white/10 px-2 py-0.5 text-slate-300">
                            {t}
                          </span>
                        ))}
                        {s.isPublic && (
                          <span className="text-xs rounded-full border border-emerald-500/30 bg-emerald-500/10 px-2 py-0.5 text-emerald-300">
                            Public
                          </span>
                        )}
                        {!s.isPublic && (
                          <span className="text-xs rounded-full border border-slate-500/30 bg-slate-500/10 px-2 py-0.5 text-slate-300">
                            Private
                          </span>
                        )}
              </div>
                    </button>
                  ))}
                </div>
              )}
              </div>

            <div className="lg:col-span-2 rounded-2xl border border-white/10 bg-slate-900/50 p-6">
              {!selected && (
                <div className="text-slate-400">
                  Select a saved strategy on the left to view details.
                </div>
              )}

              {selected && (
                <div className="space-y-5">
                  <div className="flex items-start justify-between gap-4">
                    <div>
                      <div className="text-white text-2xl font-bold">{selected.name}</div>
                      <div className="mt-1 text-slate-400 text-sm">
                        Updated <span suppressHydrationWarning>{new Date(selected.updatedAt).toLocaleString()}</span>
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <button
                        onClick={() => void copyToClipboard(selected.description || '')}
                        className="inline-flex items-center gap-2 rounded-xl border border-white/10 bg-slate-950/30 px-4 py-2 text-white hover:bg-white/5"
                      >
                        <Copy className="h-4 w-4" />
                        Copy prompt
                      </button>
              </div>
            </div>

                  <div className="rounded-xl border border-white/10 bg-slate-950/30 p-4">
                    <div className="flex items-center gap-2 text-slate-200 mb-2">
                      <Shield className="h-4 w-4 text-blue-300" />
                      Strategy Rules (JSON)
                    </div>
                    <pre className="overflow-auto text-xs text-slate-200 whitespace-pre-wrap">
                      {selected.rules ? JSON.stringify(selected.rules, null, 2) : 'No rules stored.'}
                    </pre>
                  </div>

                  <div className="rounded-xl border border-white/10 bg-slate-950/30 p-4">
                    <div className="text-slate-200 mb-2 font-semibold">How to use</div>
                    <p className="text-slate-400 text-sm">
                      Copy the prompt and paste it into the dashboard chat with a season/time period (e.g. “2024 season”)
                      to generate code and run a real backtest.
                    </p>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </ProtectedRoute>
  );
}
