'use client';

import { useState, useRef, Suspense, useCallback } from 'react';
import Link from 'next/link';
import { useSearchParams, useRouter } from 'next/navigation';
import {
  Eye,
  EyeOff,
  Mail,
  Lock,
  User,
  Check,
  ArrowRight,
  Star,
  Zap,
  Crown,
  ArrowLeft,
  Loader2,
  Shield
} from 'lucide-react';
import { StripeProvider } from '@/components/StripeProvider';
import { PaymentForm } from '@/components/PaymentForm';

const pricingTiers = [
  {
    id: 'PRO_TRIAL',
    name: 'Pro Trial',
    price: 1,
    period: 'for 7 days',
    afterTrialPrice: 20,
    description: 'Try Pro risk-free for 7 days',
    features: ['Unlimited Strategy Backtesting', 'All Sports Markets & Leagues', 'Advanced Analytics', 'Priority Email Support'],
    popular: true,
    icon: Star,
    isTrial: true
  },
  {
    id: 'PRO',
    name: 'Pro',
    price: 20,
    period: 'per month',
    description: 'For serious traders and bettors',
    features: ['Unlimited Strategy Backtesting', 'All Sports Markets & Leagues', 'Advanced Analytics', 'Priority Email Support'],
    popular: false,
    icon: Zap,
    isTrial: false
  },
  {
    id: 'ELITE_TRIAL',
    name: 'Elite Trial',
    price: 20,
    period: 'for 7 days',
    afterTrialPrice: 200,
    description: 'Try Elite with full access for 7 days',
    features: ['Everything in Pro +', 'Real-Time Data Feeds', 'Advanced Optimization Tools', 'Genetic Algorithms', 'Dedicated Account Manager'],
    popular: false,
    icon: Crown,
    isTrial: true
  },
  {
    id: 'ELITE',
    name: 'Elite',
    price: 200,
    period: 'per month',
    description: 'For professional funds and institutions',
    features: ['Everything in Pro +', 'Real-Time Data Feeds', 'Advanced Optimization Tools', 'Genetic Algorithms', 'Dedicated Account Manager'],
    popular: false,
    icon: Crown,
    isTrial: false
  }
];

function SignupPageContent() {
  const searchParams = useSearchParams();
  const router = useRouter();
  const tierParam = searchParams.get('tier');
  const trialParam = searchParams.get('trial');

  const getInitialPlan = () => {
    if (tierParam) {
      const tier = tierParam.toLowerCase();
      const isTrial = trialParam === 'true';

      if (tier === 'pro') {
        return isTrial ? 'PRO_TRIAL' : 'PRO';
      }
      if (tier === 'elite') {
        return isTrial ? 'ELITE_TRIAL' : 'ELITE';
      }
    }
    return 'PRO_TRIAL'; // Default to Pro Trial
  };

  const [step, setStep] = useState(1);
  const [selectedPlan, setSelectedPlan] = useState(getInitialPlan());
  const [showApiUpsell, setShowApiUpsell] = useState(false);
  const [apiTokenPack, setApiTokenPack] = useState<string | null>(null);

  // Form refs for uncontrolled inputs
  const firstNameRef = useRef<HTMLInputElement>(null);
  const lastNameRef = useRef<HTMLInputElement>(null);
  const emailRef = useRef<HTMLInputElement>(null);
  const passwordRef = useRef<HTMLInputElement>(null);
  const confirmPasswordRef = useRef<HTMLInputElement>(null);
  const [showPassword, setShowPassword] = useState(false);

  // Payment state
  const [clientSecret, setClientSecret] = useState<string | null>(null);
  const [isCreatingPaymentIntent, setIsCreatingPaymentIntent] = useState(false);
  const [userId, setUserId] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState('');
  const [userFirstName, setUserFirstName] = useState('');
  const [userLastName, setUserLastName] = useState('');

  const [isLoading, setIsLoading] = useState(false);
  const [errors, setErrors] = useState<Record<string, string>>({});

  const selectedTier = pricingTiers.find(t => t.id === selectedPlan);

  const validateStep1 = useCallback(() => {
    const newErrors: Record<string, string> = {};
    const firstName = firstNameRef.current?.value || '';
    const lastName = lastNameRef.current?.value || '';
    const email = emailRef.current?.value || '';
    const password = passwordRef.current?.value || '';
    const confirmPassword = confirmPasswordRef.current?.value || '';

    if (!firstName.trim()) newErrors.firstName = 'First name is required';
    if (!lastName.trim()) newErrors.lastName = 'Last name is required';
    if (!email.trim()) newErrors.email = 'Email is required';
    else if (!/\S+@\S+\.\S+/.test(email)) newErrors.email = 'Email is invalid';
    if (!password) newErrors.password = 'Password is required';
    else if (password.length < 8) newErrors.password = 'Password must be at least 8 characters';
    if (password !== confirmPassword) newErrors.confirmPassword = 'Passwords do not match';

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  }, []);

  const handleStep1Submit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!validateStep1()) {
      return;
    }

    const firstName = firstNameRef.current?.value || '';
    const lastName = lastNameRef.current?.value || '';
    const email = emailRef.current?.value || '';
    const password = passwordRef.current?.value || '';

    // Store for payment step
    setUserEmail(email);
    setUserFirstName(firstName);
    setUserLastName(lastName);

    setIsLoading(true);

    try {
      // First, create the user account
      const signupResponse = await fetch('/api/auth/signup', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          email,
          password,
          firstName,
          lastName,
          subscriptionTier: selectedPlan.toUpperCase(),
        }),
      });

      if (!signupResponse.ok) {
        const errorData = await signupResponse.json();
        throw new Error(errorData.error || 'Signup failed');
      }

      const signupData = await signupResponse.json();
      setUserId(signupData.user?.id);

      // Store token for later use
      if (signupData.token) {
        localStorage.setItem('authToken', signupData.token);
      }

      // Create payment intent for step 2
      setIsCreatingPaymentIntent(true);
      const paymentResponse = await fetch('/api/billing/create-payment-intent', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          tier: selectedPlan,
          email,
          userId: signupData.user?.id,
          firstName,
          lastName,
        }),
      });

      if (!paymentResponse.ok) {
        throw new Error('Failed to initialize payment');
      }

      const paymentData = await paymentResponse.json();

      if (!paymentData.requiresPayment) {
        // Free tier - redirect to welcome
        router.push('/welcome');
        return;
      }

      setClientSecret(paymentData.clientSecret);
      setStep(2);

    } catch (error) {
      console.error('Signup error:', error);
      setErrors({ general: error instanceof Error ? error.message : 'Signup failed. Please try again.' });
    } finally {
      setIsLoading(false);
      setIsCreatingPaymentIntent(false);
    }
  };

  const handlePaymentSuccess = () => {
    // Payment successful - redirect to welcome page
    router.push('/welcome?payment=success');
  };

  const handlePaymentError = (error: string) => {
    setErrors({ payment: error });
  };

  return (
    <div className="min-h-screen bg-slate-950 flex items-center justify-center px-4 py-12">
      <div className="max-w-2xl w-full">
        <div className="text-center mb-8">
          <Link href="/" className="text-3xl font-bold text-white mb-4 block">
            EvenTheOdds<span className="text-blue-500">AI</span>
          </Link>
          <h1 className="text-3xl font-bold text-white mb-2">
            Start Your {selectedTier?.name}
          </h1>
          <p className="text-slate-400">
            Join thousands of successful traders and bettors
          </p>
        </div>

        {/* Plan Selection - Only show on step 1 */}
        {step === 1 && (
          <div className="mb-8">
            <h2 className="text-xl font-semibold text-white mb-4">Choose Your Plan</h2>
            <div className="grid md:grid-cols-3 gap-4">
              {pricingTiers.map((tier) => (
                <div
                  key={tier.id}
                  onClick={() => setSelectedPlan(tier.id)}
                  className={`relative p-4 border rounded-lg cursor-pointer transition-all ${selectedPlan === tier.id
                    ? 'border-blue-500 bg-blue-500/10'
                    : 'border-slate-700 bg-slate-900/50 hover:border-slate-600'
                    }`}
                >
                  {tier.popular && (
                    <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                      <span className="bg-blue-500 text-white px-3 py-1 rounded-full text-xs font-semibold">
                        Most Popular
                      </span>
                    </div>
                  )}

                  <div className="text-center">
                    <h3 className="font-semibold text-white mb-1">{tier.name}</h3>
                    <div className="text-2xl font-bold text-white mb-1">
                      ${tier.price}
                      <span className="text-sm text-slate-400">/{tier.period}</span>
                    </div>
                    <p className="text-sm text-slate-400 mb-3">{tier.description}</p>
                    <ul className="text-xs text-slate-300 space-y-1">
                      {tier.features.slice(0, 3).map((feature, i) => (
                        <li key={i} className="flex items-center gap-2">
                          <Check className="h-3 w-3 text-green-400 flex-shrink-0" />
                          <span>{feature}</span>
                        </li>
                      ))}
                    </ul>
                  </div>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Step Indicator */}
        <div className="flex items-center justify-center mb-8">
          <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${step >= 1 ? 'bg-blue-500 text-white' : 'bg-slate-700 text-slate-400'}`}>
            1
          </div>
          <div className={`w-12 h-1 mx-2 ${step >= 2 ? 'bg-blue-500' : 'bg-slate-700'}`}></div>
          <div className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-semibold ${step >= 2 ? 'bg-blue-500 text-white' : 'bg-slate-700 text-slate-400'}`}>
            2
          </div>
        </div>

        {/* General Error */}
        {errors.general && (
          <div className="mb-6 p-4 bg-red-500/10 border border-red-500/20 rounded-lg">
            <p className="text-red-400 text-sm">{errors.general}</p>
          </div>
        )}

        {/* Step 1: Account Details */}
        {step === 1 && (
          <form onSubmit={handleStep1Submit} className="space-y-6">
            <div className="grid md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-slate-300 mb-2">
                  First Name
                </label>
                <div className="relative">
                  <User className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                  <input
                    ref={firstNameRef}
                    type="text"
                    name="firstName"
                    autoComplete="given-name"
                    className="w-full pl-10 pr-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    placeholder="John"
                  />
                </div>
                {errors.firstName && <p className="text-red-400 text-sm mt-1">{errors.firstName}</p>}
              </div>

              <div>
                <label className="block text-sm font-medium text-slate-300 mb-2">
                  Last Name
                </label>
                <input
                  ref={lastNameRef}
                  type="text"
                  name="lastName"
                  autoComplete="family-name"
                  className="w-full px-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="Doe"
                />
                {errors.lastName && <p className="text-red-400 text-sm mt-1">{errors.lastName}</p>}
              </div>
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Email Address
              </label>
              <div className="relative">
                <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                <input
                  ref={emailRef}
                  type="email"
                  name="email"
                  autoComplete="email"
                  className="w-full pl-10 pr-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="john@example.com"
                />
              </div>
              {errors.email && <p className="text-red-400 text-sm mt-1">{errors.email}</p>}
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Password
              </label>
              <div className="relative">
                <Lock className="absolute left-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-slate-400" />
                <input
                  ref={passwordRef}
                  type={showPassword ? 'text' : 'password'}
                  name="password"
                  autoComplete="new-password"
                  className="w-full pl-10 pr-12 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  placeholder="At least 8 characters"
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-3 top-1/2 transform -translate-y-1/2 text-slate-400 hover:text-slate-300"
                >
                  {showPassword ? <EyeOff className="h-5 w-5" /> : <Eye className="h-5 w-5" />}
                </button>
              </div>
              {errors.password && <p className="text-red-400 text-sm mt-1">{errors.password}</p>}
            </div>

            <div>
              <label className="block text-sm font-medium text-slate-300 mb-2">
                Confirm Password
              </label>
              <input
                ref={confirmPasswordRef}
                type="password"
                name="confirmPassword"
                autoComplete="new-password"
                className="w-full px-4 py-3 bg-slate-900 border border-slate-700 rounded-lg text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                placeholder="Confirm your password"
              />
              {errors.confirmPassword && <p className="text-red-400 text-sm mt-1">{errors.confirmPassword}</p>}
            </div>

            <button
              type="submit"
              disabled={isLoading}
              className="w-full bg-blue-600 hover:bg-blue-500 disabled:bg-slate-600 disabled:cursor-not-allowed text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
            >
              {isLoading ? (
                <>
                  <Loader2 className="h-5 w-5 animate-spin" />
                  Creating Account...
                </>
              ) : (
                <>
                  Continue to Payment
                  <ArrowRight className="h-5 w-5" />
                </>
              )}
            </button>
          </form>
        )}

        {/* Step 2: Payment with Stripe Elements */}
        {step === 2 && clientSecret && (
          <div className="space-y-6">
            {/* Back button */}
            <button
              onClick={() => setStep(1)}
              className="flex items-center gap-2 text-slate-400 hover:text-white transition-colors"
            >
              <ArrowLeft className="h-4 w-4" />
              Back to account details
            </button>

            {/* Order summary */}
            <div className="bg-slate-900/50 border border-slate-700 rounded-lg p-4 mb-6">
              <h3 className="font-semibold text-white mb-3">Order Summary</h3>
              <div className="flex justify-between items-center pb-3 border-b border-slate-700">
                <span className="text-slate-400">{selectedTier?.name}</span>
                <span className="text-white font-semibold">
                  ${selectedTier?.price} {selectedTier?.isTrial ? 'for 7 days' : '/month'}
                </span>
              </div>
              {selectedTier?.isTrial && (
                <p className="text-slate-500 text-xs mt-2">
                  Then ${selectedTier?.afterTrialPrice}/month after trial. Cancel anytime.
                </p>
              )}

              {/* API Token Upsell */}
              <div className="mt-4 pt-4 border-t border-slate-700">
                <div className="flex items-center gap-2 mb-3">
                  <Zap className="h-4 w-4 text-violet-400" />
                  <span className="text-white font-medium text-sm">Maximize Your Workflow</span>
                </div>
                <p className="text-slate-400 text-xs mb-3">
                  Add API tokens to automate data retrieval and build custom integrations.
                </p>
                <div className="grid grid-cols-2 gap-2">
                  <button
                    type="button"
                    onClick={() => setApiTokenPack(apiTokenPack === '100k' ? null : '100k')}
                    className={`p-3 rounded-lg border text-left transition-all ${apiTokenPack === '100k'
                        ? 'border-violet-500 bg-violet-500/10'
                        : 'border-slate-600 hover:border-slate-500'
                      }`}
                  >
                    <div className="text-white font-semibold text-sm">100K Tokens</div>
                    <div className="text-violet-400 text-xs">+$29</div>
                  </button>
                  <button
                    type="button"
                    onClick={() => setApiTokenPack(apiTokenPack === '1m' ? null : '1m')}
                    className={`p-3 rounded-lg border text-left transition-all ${apiTokenPack === '1m'
                        ? 'border-violet-500 bg-violet-500/10'
                        : 'border-slate-600 hover:border-slate-500'
                      }`}
                  >
                    <div className="text-white font-semibold text-sm">1M Tokens</div>
                    <div className="text-violet-400 text-xs">+$199</div>
                  </button>
                </div>
                {apiTokenPack && (
                  <div className="mt-3 flex justify-between items-center text-sm">
                    <span className="text-slate-400">API Token Pack ({apiTokenPack === '100k' ? '100K' : '1M'})</span>
                    <span className="text-violet-400 font-medium">+${apiTokenPack === '100k' ? '29' : '199'}</span>
                  </div>
                )}
              </div>

              {/* Total */}
              {apiTokenPack && (
                <div className="mt-4 pt-3 border-t border-slate-700 flex justify-between items-center">
                  <span className="text-white font-medium">Total Today</span>
                  <span className="text-white font-bold text-lg">
                    ${(selectedTier?.price || 0) + (apiTokenPack === '100k' ? 29 : 199)}
                  </span>
                </div>
              )}
            </div>

            {/* Payment error */}
            {errors.payment && (
              <div className="p-4 bg-red-500/10 border border-red-500/20 rounded-lg">
                <p className="text-red-400 text-sm">{errors.payment}</p>
              </div>
            )}

            {/* Stripe Payment Form */}
            <StripeProvider clientSecret={clientSecret}>
              <PaymentForm
                onSuccess={handlePaymentSuccess}
                onError={handlePaymentError}
                // ⚠️ NOTE: Do NOT use /backtest prefix - production runs without basePath
                returnUrl={`${typeof window !== 'undefined' ? window.location.origin : ''}/welcome?payment=success`}
                planName={selectedTier?.name || 'Subscription'}
                price={selectedTier?.price || 0}
              />
            </StripeProvider>
          </div>
        )}

        {/* Loading state for payment intent creation */}
        {step === 2 && !clientSecret && isCreatingPaymentIntent && (
          <div className="text-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-blue-500 mx-auto mb-4" />
            <p className="text-slate-400">Setting up secure payment...</p>
          </div>
        )}

        <div className="text-center mt-6">
          <p className="text-slate-400">
            Already have an account?{' '}
            <Link href="/login" className="text-blue-400 hover:text-blue-300 font-semibold">
              Sign in
            </Link>
          </p>
        </div>

        <div className="border-t border-slate-800 pt-6 mt-8">
          <div className="flex items-center justify-center gap-4 text-slate-500 text-sm mb-4">
            <Shield className="h-4 w-4" />
            <span>Your payment is secured by Stripe</span>
          </div>
          <div className="text-center text-sm text-slate-500">
            By creating an account, you agree to our{' '}
            <Link href="/terms" className="text-blue-400 hover:text-blue-300">Terms of Service</Link>
            {' '}and{' '}
            <Link href="/privacy" className="text-blue-400 hover:text-blue-300">Privacy Policy</Link>
          </div>
        </div>
      </div>
    </div>
  );
}

export default function SignupPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-slate-950 flex items-center justify-center px-4 py-12">
        <div className="max-w-2xl w-full">
          <div className="text-center mb-8">
            <Link href="/" className="text-3xl font-bold text-white mb-4 block">
              EvenTheOdds<span className="text-blue-500">AI</span>
            </Link>
            <Loader2 className="h-8 w-8 animate-spin text-blue-500 mx-auto" />
            <p className="text-slate-400 mt-4">Loading...</p>
          </div>
        </div>
      </div>
    }>
      <SignupPageContent />
    </Suspense>
  );
}
