'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import {
  User,
  Mail,
  Shield,
  Download,
  Trash2,
  CreditCard,
  Bell,
  Lock,
  AlertTriangle,
  Loader2,
  Check,
  X,
  ArrowLeft,
  ExternalLink
} from 'lucide-react';

interface UserData {
  id: string;
  email: string;
  firstName: string;
  lastName: string;
  subscriptionTier: string;
  subscriptionStatus: string;
  emailVerified: boolean;
}

export default function SettingsPage() {
  const router = useRouter();
  const [user, setUser] = useState<UserData | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [activeSection, setActiveSection] = useState('account');
  
  // Export state
  const [isExporting, setIsExporting] = useState(false);
  const [exportMessage, setExportMessage] = useState('');
  
  // Cancellation state
  const [isCancelling, setIsCancelling] = useState(false);
  const [showCancelModal, setShowCancelModal] = useState(false);
  const [cancelReason, setCancelReason] = useState('');
  const [cancelMessage, setCancelMessage] = useState('');
  
  // Deletion state
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [deleteConfirmation, setDeleteConfirmation] = useState('');
  const [isDeleting, setIsDeleting] = useState(false);
  const [deleteError, setDeleteError] = useState('');

  useEffect(() => {
    fetchUserProfile();
  }, []);

  const fetchUserProfile = async () => {
    try {
      const response = await fetch('/api/auth/profile', {
        credentials: 'include',
      });

      if (!response.ok) {
        router.push('/login');
        return;
      }

      const data = await response.json();
      setUser(data.user);
    } catch (error) {
      console.error('Failed to fetch profile:', error);
      router.push('/login');
    } finally {
      setIsLoading(false);
    }
  };

  const handleExportData = async () => {
    setIsExporting(true);
    setExportMessage('');

    try {
      const response = await fetch('/api/user/export-data', {
        credentials: 'include',
      });

      if (!response.ok) {
        throw new Error('Failed to export data');
      }

      // Trigger download
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `eventheodds-data-export-${new Date().toISOString().split('T')[0]}.json`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      setExportMessage('Your data has been downloaded successfully.');
    } catch (error) {
      setExportMessage('Failed to export data. Please try again.');
    } finally {
      setIsExporting(false);
    }
  };

  const handleCancelSubscription = async () => {
    setIsCancelling(true);
    setCancelMessage('');

    try {
      const response = await fetch('/api/billing/cancel-subscription', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ reason: cancelReason }),
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || 'Failed to cancel subscription');
      }

      setCancelMessage(data.message);
      setShowCancelModal(false);
      fetchUserProfile(); // Refresh user data
    } catch (error) {
      setCancelMessage(error instanceof Error ? error.message : 'Failed to cancel subscription');
    } finally {
      setIsCancelling(false);
    }
  };

  const handleDeleteAccount = async () => {
    if (deleteConfirmation !== 'DELETE_MY_ACCOUNT') {
      setDeleteError('Please type DELETE_MY_ACCOUNT to confirm');
      return;
    }

    setIsDeleting(true);
    setDeleteError('');

    try {
      const response = await fetch('/api/user/delete-account', {
        method: 'DELETE',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ confirmation: deleteConfirmation }),
      });

      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.error || 'Failed to delete account');
      }

      // Clear local storage and redirect
      localStorage.clear();
      router.push('/?deleted=true');
    } catch (error) {
      setDeleteError(error instanceof Error ? error.message : 'Failed to delete account');
    } finally {
      setIsDeleting(false);
    }
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-slate-950 flex items-center justify-center">
        <Loader2 className="h-8 w-8 animate-spin text-blue-500" />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-slate-950 py-12 px-4">
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Link href="/dashboard" className="inline-flex items-center gap-2 text-slate-400 hover:text-white mb-4">
            <ArrowLeft className="h-4 w-4" />
            Back to Dashboard
          </Link>
          <h1 className="text-3xl font-bold text-white">Account Settings</h1>
          <p className="text-slate-400 mt-2">Manage your account, subscription, and data</p>
        </div>

        <div className="grid md:grid-cols-4 gap-6">
          {/* Sidebar */}
          <div className="space-y-2">
            {[
              { id: 'account', icon: User, label: 'Account' },
              { id: 'subscription', icon: CreditCard, label: 'Subscription' },
              { id: 'privacy', icon: Shield, label: 'Privacy & Data' },
              { id: 'danger', icon: AlertTriangle, label: 'Danger Zone' },
            ].map((item) => (
              <button
                key={item.id}
                onClick={() => setActiveSection(item.id)}
                className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg text-left transition-colors ${
                  activeSection === item.id
                    ? 'bg-blue-500/20 text-blue-400 border border-blue-500/30'
                    : 'text-slate-400 hover:bg-slate-800 hover:text-white'
                }`}
              >
                <item.icon className="h-5 w-5" />
                {item.label}
              </button>
            ))}
          </div>

          {/* Content */}
          <div className="md:col-span-3 space-y-6">
            {/* Account Section */}
            {activeSection === 'account' && (
              <div className="bg-slate-900/50 border border-slate-700 rounded-xl p-6">
                <h2 className="text-xl font-semibold text-white mb-6 flex items-center gap-2">
                  <User className="h-5 w-5 text-blue-400" />
                  Account Information
                </h2>

                <div className="space-y-4">
                  <div className="grid md:grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm text-slate-400 mb-1">First Name</label>
                      <p className="text-white">{user?.firstName || '-'}</p>
                    </div>
                    <div>
                      <label className="block text-sm text-slate-400 mb-1">Last Name</label>
                      <p className="text-white">{user?.lastName || '-'}</p>
                    </div>
                  </div>
                  
                  <div>
                    <label className="block text-sm text-slate-400 mb-1">Email</label>
                    <div className="flex items-center gap-2">
                      <p className="text-white">{user?.email}</p>
                      {user?.emailVerified ? (
                        <span className="flex items-center gap-1 text-xs text-green-400 bg-green-500/10 px-2 py-1 rounded">
                          <Check className="h-3 w-3" />
                          Verified
                        </span>
                      ) : (
                        <span className="flex items-center gap-1 text-xs text-yellow-400 bg-yellow-500/10 px-2 py-1 rounded">
                          <X className="h-3 w-3" />
                          Unverified
                        </span>
                      )}
                    </div>
                  </div>
                </div>
              </div>
            )}

            {/* Subscription Section */}
            {activeSection === 'subscription' && (
              <div className="bg-slate-900/50 border border-slate-700 rounded-xl p-6">
                <h2 className="text-xl font-semibold text-white mb-6 flex items-center gap-2">
                  <CreditCard className="h-5 w-5 text-blue-400" />
                  Subscription
                </h2>

                <div className="space-y-4">
                  <div className="flex items-center justify-between p-4 bg-slate-800/50 rounded-lg">
                    <div>
                      <p className="text-white font-medium">{user?.subscriptionTier || 'Free'} Plan</p>
                      <p className="text-slate-400 text-sm capitalize">Status: {user?.subscriptionStatus || 'Active'}</p>
                    </div>
                    <Link
                      href="/pricing"
                      className="text-blue-400 hover:text-blue-300 text-sm flex items-center gap-1"
                    >
                      Upgrade
                      <ExternalLink className="h-4 w-4" />
                    </Link>
                  </div>

                  {cancelMessage && (
                    <div className={`p-4 rounded-lg ${
                      cancelMessage.includes('Failed') ? 'bg-red-500/10 border border-red-500/20' : 'bg-green-500/10 border border-green-500/20'
                    }`}>
                      <p className={cancelMessage.includes('Failed') ? 'text-red-400' : 'text-green-400'}>{cancelMessage}</p>
                    </div>
                  )}

                  {user?.subscriptionTier && user.subscriptionTier !== 'FREE' && (
                    <button
                      onClick={() => setShowCancelModal(true)}
                      className="text-red-400 hover:text-red-300 text-sm"
                    >
                      Cancel Subscription
                    </button>
                  )}
                </div>
              </div>
            )}

            {/* Privacy & Data Section */}
            {activeSection === 'privacy' && (
              <div className="bg-slate-900/50 border border-slate-700 rounded-xl p-6">
                <h2 className="text-xl font-semibold text-white mb-6 flex items-center gap-2">
                  <Shield className="h-5 w-5 text-blue-400" />
                  Privacy & Data
                </h2>

                <div className="space-y-6">
                  <div>
                    <h3 className="text-white font-medium mb-2">Export Your Data</h3>
                    <p className="text-slate-400 text-sm mb-4">
                      Download a copy of all your data including strategies, conversations, and activity log.
                      This is in compliance with GDPR Article 20 (Right to Data Portability).
                    </p>
                    
                    {exportMessage && (
                      <p className={`text-sm mb-4 ${exportMessage.includes('Failed') ? 'text-red-400' : 'text-green-400'}`}>
                        {exportMessage}
                      </p>
                    )}

                    <button
                      onClick={handleExportData}
                      disabled={isExporting}
                      className="flex items-center gap-2 px-4 py-2 bg-slate-700 hover:bg-slate-600 disabled:bg-slate-800 text-white rounded-lg transition-colors"
                    >
                      {isExporting ? (
                        <>
                          <Loader2 className="h-4 w-4 animate-spin" />
                          Exporting...
                        </>
                      ) : (
                        <>
                          <Download className="h-4 w-4" />
                          Export My Data
                        </>
                      )}
                    </button>
                  </div>

                  <div className="border-t border-slate-700 pt-6">
                    <h3 className="text-white font-medium mb-2">Privacy Policy</h3>
                    <p className="text-slate-400 text-sm mb-4">
                      Learn how we collect, use, and protect your data.
                    </p>
                    <Link
                      href="/privacy"
                      className="text-blue-400 hover:text-blue-300 text-sm flex items-center gap-1"
                    >
                      Read Privacy Policy
                      <ExternalLink className="h-4 w-4" />
                    </Link>
                  </div>
                </div>
              </div>
            )}

            {/* Danger Zone Section */}
            {activeSection === 'danger' && (
              <div className="bg-red-500/5 border border-red-500/20 rounded-xl p-6">
                <h2 className="text-xl font-semibold text-red-400 mb-6 flex items-center gap-2">
                  <AlertTriangle className="h-5 w-5" />
                  Danger Zone
                </h2>

                <div className="space-y-4">
                  <div>
                    <h3 className="text-white font-medium mb-2">Delete Account</h3>
                    <p className="text-slate-400 text-sm mb-4">
                      Permanently delete your account and all associated data. This action cannot be undone.
                      Your subscription will be cancelled and you will lose access immediately.
                    </p>
                    <button
                      onClick={() => setShowDeleteModal(true)}
                      className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-500 text-white rounded-lg transition-colors"
                    >
                      <Trash2 className="h-4 w-4" />
                      Delete My Account
                    </button>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Cancel Subscription Modal */}
      {showCancelModal && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-slate-900 border border-slate-700 rounded-xl p-6 max-w-md w-full">
            <h3 className="text-xl font-semibold text-white mb-4">Cancel Subscription</h3>
            <p className="text-slate-400 mb-4">
              Are you sure you want to cancel? You'll keep access until the end of your billing period.
            </p>
            
            <div className="mb-4">
              <label className="block text-sm text-slate-400 mb-2">
                Why are you cancelling? (optional)
              </label>
              <textarea
                value={cancelReason}
                onChange={(e) => setCancelReason(e.target.value)}
                className="w-full px-4 py-2 bg-slate-800 border border-slate-700 rounded-lg text-white placeholder-slate-400"
                placeholder="Help us improve..."
                rows={3}
              />
            </div>

            <div className="flex gap-3">
              <button
                onClick={() => setShowCancelModal(false)}
                className="flex-1 px-4 py-2 bg-slate-700 hover:bg-slate-600 text-white rounded-lg"
              >
                Keep Subscription
              </button>
              <button
                onClick={handleCancelSubscription}
                disabled={isCancelling}
                className="flex-1 px-4 py-2 bg-red-600 hover:bg-red-500 disabled:bg-red-800 text-white rounded-lg"
              >
                {isCancelling ? 'Cancelling...' : 'Confirm Cancel'}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Delete Account Modal */}
      {showDeleteModal && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-slate-900 border border-red-500/30 rounded-xl p-6 max-w-md w-full">
            <div className="flex items-center gap-3 mb-4">
              <div className="p-2 bg-red-500/10 rounded-lg">
                <AlertTriangle className="h-6 w-6 text-red-400" />
              </div>
              <h3 className="text-xl font-semibold text-white">Delete Account</h3>
            </div>
            
            <div className="space-y-4 mb-6">
              <p className="text-slate-400">
                This will permanently delete:
              </p>
              <ul className="text-slate-300 text-sm space-y-1">
                <li>• Your account and profile</li>
                <li>• All saved strategies</li>
                <li>• Conversation history</li>
                <li>• Activity logs</li>
                <li>• Cancel your subscription</li>
              </ul>
              <p className="text-red-400 text-sm font-medium">
                This action cannot be undone.
              </p>
            </div>

            <div className="mb-4">
              <label className="block text-sm text-slate-400 mb-2">
                Type <span className="font-mono text-red-400">DELETE_MY_ACCOUNT</span> to confirm:
              </label>
              <input
                type="text"
                value={deleteConfirmation}
                onChange={(e) => setDeleteConfirmation(e.target.value)}
                className="w-full px-4 py-2 bg-slate-800 border border-slate-700 rounded-lg text-white placeholder-slate-400 font-mono"
                placeholder="DELETE_MY_ACCOUNT"
              />
              {deleteError && (
                <p className="text-red-400 text-sm mt-2">{deleteError}</p>
              )}
            </div>

            <div className="flex gap-3">
              <button
                onClick={() => {
                  setShowDeleteModal(false);
                  setDeleteConfirmation('');
                  setDeleteError('');
                }}
                className="flex-1 px-4 py-2 bg-slate-700 hover:bg-slate-600 text-white rounded-lg"
              >
                Cancel
              </button>
              <button
                onClick={handleDeleteAccount}
                disabled={isDeleting || deleteConfirmation !== 'DELETE_MY_ACCOUNT'}
                className="flex-1 px-4 py-2 bg-red-600 hover:bg-red-500 disabled:bg-slate-700 disabled:text-slate-500 text-white rounded-lg transition-colors"
              >
                {isDeleting ? (
                  <span className="flex items-center justify-center gap-2">
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Deleting...
                  </span>
                ) : (
                  'Delete Account'
                )}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
