'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { ArrowLeft, BadgeCheck, Calendar, History, Mail, Shield, User } from 'lucide-react';
import ProtectedRoute from '@/components/ProtectedRoute';
import { useUser } from '@/lib/hooks/useUser';

export default function ProfilePage() {
  const { user: userProfile, loading, error } = useUser();
  const [betLog, setBetLog] = useState<any[]>([]);
  const [betLogLoading, setBetLogLoading] = useState(false);
  const [betLogError, setBetLogError] = useState<string | null>(null);

  useEffect(() => {
    if (!userProfile) return;
    setBetLogLoading(true);
    setBetLogError(null);

    const token = localStorage.getItem('authToken');
    void fetch('/api/user/bet-log?limit=20', {
      headers: {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
      },
    })
      .then(async (res) => {
        const data = await res.json().catch(() => ({}));
        if (!res.ok) throw new Error(data?.error || 'Failed to load bet log');
        setBetLog(Array.isArray(data?.items) ? data.items : []);
      })
      .catch((e) => setBetLogError(e instanceof Error ? e.message : 'Failed to load bet log'))
      .finally(() => setBetLogLoading(false));
  }, [userProfile]);

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-slate-950 py-12 px-4">
        <div className="max-w-5xl mx-auto">
          <div className="mb-8">
            <Link href="/dashboard" className="inline-flex items-center gap-2 text-slate-400 hover:text-white mb-4">
              <ArrowLeft className="h-4 w-4" />
              Back to Dashboard
            </Link>
            <h1 className="text-3xl font-bold text-white">Profile</h1>
            <p className="text-slate-400 mt-2">Your account details and usage stats</p>
          </div>

          {loading && (
            <div className="glass-panel p-10 rounded-2xl text-center">
              <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-blue-500 mx-auto mb-4"></div>
              <p className="text-slate-400">Loading profile…</p>
            </div>
          )}

          {!loading && (error || !userProfile) && (
            <div className="glass-panel p-10 rounded-2xl text-center">
              <p className="text-slate-200 font-semibold mb-2">Unable to load profile</p>
              <p className="text-slate-400 mb-6">{error || 'Please sign in again.'}</p>
              <div className="flex items-center justify-center gap-3">
                <Link
                  href="/login"
                  className="bg-blue-600 hover:bg-blue-500 text-white font-semibold py-2 px-5 rounded-xl transition"
                >
                  Go to Login
                </Link>
                <Link
                  href="/settings"
                  className="bg-slate-800 hover:bg-slate-700 text-white font-semibold py-2 px-5 rounded-xl transition"
                >
                  Settings
                </Link>
              </div>
            </div>
          )}

          {!loading && userProfile && (
            <div className="space-y-6">
              <div className="grid md:grid-cols-2 gap-6">
                <div className="glass-panel p-6 rounded-2xl border border-white/10">
                  <div className="flex items-start justify-between gap-4">
                    <div>
                      <div className="flex items-center gap-2 text-white font-semibold text-lg">
                        <User className="h-5 w-5 text-blue-400" />
                        {(userProfile.user.firstName || userProfile.user.lastName)
                          ? `${userProfile.user.firstName || ''} ${userProfile.user.lastName || ''}`.trim()
                          : 'Account'}
                      </div>
                      <div className="mt-2 flex items-center gap-2 text-slate-300">
                        <Mail className="h-4 w-4 text-slate-400" />
                        {userProfile.user.email}
                      </div>
                    </div>

                    <div className="text-xs uppercase tracking-wide px-3 py-1 rounded-full border border-blue-500/30 bg-blue-500/10 text-blue-300">
                      {userProfile.user.subscriptionTier}
                    </div>
                  </div>

                  <div className="mt-6 grid grid-cols-1 sm:grid-cols-2 gap-4">
                    <div className="rounded-xl border border-white/10 bg-slate-950/40 p-4">
                      <div className="flex items-center gap-2 text-slate-300 text-sm">
                        <BadgeCheck className="h-4 w-4 text-green-400" />
                        Email Verified
                      </div>
                      <div className="mt-1 text-white font-semibold">
                        {userProfile.user.emailVerified ? 'Yes' : 'No'}
                      </div>
                    </div>
                    <div className="rounded-xl border border-white/10 bg-slate-950/40 p-4">
                      <div className="flex items-center gap-2 text-slate-300 text-sm">
                        <Calendar className="h-4 w-4 text-slate-400" />
                        Member Since
                      </div>
                      <div className="mt-1 text-white font-semibold">
                        {new Date(userProfile.user.createdAt).toLocaleDateString()}
                      </div>
                    </div>
                  </div>

                  <div className="mt-6 flex items-center gap-3">
                    <Link
                      href="/settings"
                      className="inline-flex items-center gap-2 bg-slate-800 hover:bg-slate-700 text-white font-semibold py-2 px-5 rounded-xl transition"
                    >
                      <Shield className="h-4 w-4" />
                      Manage Settings
                    </Link>
                  </div>
                </div>

                <div className="glass-panel p-6 rounded-2xl border border-white/10">
                  <div className="text-white font-semibold text-lg mb-4">Usage Stats</div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="rounded-xl border border-white/10 bg-slate-950/40 p-4">
                      <div className="text-slate-400 text-xs uppercase tracking-wide">AI Generations</div>
                      <div className="mt-1 text-white text-2xl font-bold">{userProfile.stats.aiGenerations}</div>
                    </div>
                    <div className="rounded-xl border border-white/10 bg-slate-950/40 p-4">
                      <div className="text-slate-400 text-xs uppercase tracking-wide">Strategies</div>
                      <div className="mt-1 text-white text-2xl font-bold">{userProfile.stats.strategiesCreated}</div>
                    </div>
                    <Link href="/backtest" className="rounded-xl border border-white/10 bg-slate-950/40 p-4 hover:border-blue-500/30 hover:bg-blue-500/5 transition group">
                      <div className="text-slate-400 text-xs uppercase tracking-wide flex items-center gap-2">
                        <History className="h-3 w-3" />
                        Backtests Run
                      </div>
                      <div className="mt-1 text-white text-2xl font-bold">{userProfile.stats.backtestsRun}</div>
                      <div className="mt-1 text-xs text-blue-400 opacity-0 group-hover:opacity-100 transition">
                        View bet history →
                      </div>
                    </Link>
                    <div className="rounded-xl border border-white/10 bg-slate-950/40 p-4">
                      <div className="text-slate-400 text-xs uppercase tracking-wide">Chat Messages</div>
                      <div className="mt-1 text-white text-2xl font-bold">{userProfile.stats.chatMessages}</div>
                    </div>
                  </div>
                </div>
              </div>

              {/* Recent Bet Log */}
              <div className="glass-panel p-6 rounded-2xl border border-white/10">
                <div className="flex items-center justify-between gap-4 mb-4">
                  <div>
                    <div className="text-white font-semibold text-lg">Recent Bet Log</div>
                    <div className="text-slate-400 text-sm">
                      Your latest saved bets across backtests (for quick access).
                    </div>
                  </div>
                  <Link
                    href="/backtest"
                    className="text-sm text-blue-400 hover:text-blue-300 underline"
                  >
                    View full history →
                  </Link>
                </div>

                {betLogError && (
                  <div className="mb-4 rounded-xl border border-red-500/20 bg-red-500/10 p-3 text-sm text-red-200">
                    {betLogError}
                  </div>
                )}

                {betLogLoading ? (
                  <div className="text-slate-300 text-sm">Loading bet log…</div>
                ) : betLog.length === 0 ? (
                  <div className="text-slate-400 text-sm">
                    No saved bets yet. Run a backtest and keep <strong>“Save bet log to Profile / History”</strong> enabled.
                  </div>
                ) : (
                  <div className="overflow-x-auto max-h-96 overflow-y-auto rounded-xl border border-white/10">
                    <table className="w-full text-sm">
                      <thead className="sticky top-0 bg-slate-900/95 text-slate-400 text-xs uppercase">
                        <tr>
                          <th className="px-3 py-2 text-left">Date</th>
                          <th className="px-3 py-2 text-left">Selection</th>
                          <th className="px-3 py-2 text-left">Market</th>
                          <th className="px-3 py-2 text-center">Outcome</th>
                          <th className="px-3 py-2 text-right">Profit</th>
                          <th className="px-3 py-2 text-right">Open</th>
                        </tr>
                      </thead>
                      <tbody className="divide-y divide-white/5">
                        {betLog.map((row: any) => {
                          const profit = typeof row?.profit === 'number' ? row.profit : Number(row?.profit || 0);
                          const isWin = (row?.outcome || '').toString().toLowerCase() === 'win';
                          return (
                            <tr key={row.id} className="hover:bg-white/5 transition">
                              <td className="px-3 py-2 text-slate-300 whitespace-nowrap">
                                {row?.date ? new Date(row.date).toLocaleDateString() : '—'}
                              </td>
                              <td className="px-3 py-2 text-white max-w-[260px] truncate">
                                {row?.label || '—'}
                              </td>
                              <td className="px-3 py-2 text-slate-300">
                                {(row?.market || 'sports').toString().toUpperCase()}
                              </td>
                              <td className="px-3 py-2 text-center">
                                <span
                                  className={`inline-flex px-2 py-0.5 rounded-full text-xs font-medium ${
                                    isWin ? 'bg-emerald-500/20 text-emerald-300' : 'bg-red-500/20 text-red-300'
                                  }`}
                                >
                                  {isWin ? 'WIN' : 'LOSS'}
                                </span>
                              </td>
                              <td
                                className={`px-3 py-2 text-right font-mono font-medium ${
                                  profit >= 0 ? 'text-emerald-400' : 'text-red-400'
                                }`}
                              >
                                {profit >= 0 ? '+' : ''}{Number.isFinite(profit) ? profit.toFixed(2) : '0.00'}
                              </td>
                              <td className="px-3 py-2 text-right">
                                <Link
                                  href={`/backtest?id=${encodeURIComponent(row.backtestId)}`}
                                  className="text-blue-400 hover:text-blue-300 underline"
                                >
                                  View
                                </Link>
                              </td>
                            </tr>
                          );
                        })}
                      </tbody>
                    </table>
                  </div>
                )}
              </div>
            </div>
          )}
        </div>
      </div>
    </ProtectedRoute>
  );
}


