import React from 'react';
import Link from 'next/link';
import { ArrowLeft, Search, BookOpen, MessageCircle, Zap, TrendingUp, BarChart3, Settings, HelpCircle } from 'lucide-react';

export default function HelpCenter() {
  const categories = [
    {
      icon: <BookOpen className="h-6 w-6" />,
      title: "Getting Started",
      description: "Learn the basics of EventheOdds.ai",
      articles: [
        "Creating your first strategy",
        "Understanding backtesting",
        "Setting up your account"
      ]
    },
    {
      icon: <Zap className="h-6 w-6" />,
      title: "AI Strategy Assistant",
      description: "Master the AI chat and strategy generation",
      articles: [
        "How to describe your betting strategy",
        "Understanding AI responses",
        "Parameter optimization tips"
      ]
    },
    {
      icon: <TrendingUp className="h-6 w-6" />,
      title: "Markets & Data",
      description: "Supported markets and data sources",
      articles: [
        "Supported sports and leagues",
        "Data accuracy and sources",
        "Real-time vs historical data"
      ]
    },
    {
      icon: <BarChart3 className="h-6 w-6" />,
      title: "Backtesting",
      description: "Advanced backtesting features",
      articles: [
        "Walk-forward analysis",
        "Parameter optimization",
        "Trade visualization"
      ]
    },
    {
      icon: <Settings className="h-6 w-6" />,
      title: "Account & Billing",
      description: "Manage your subscription and settings",
      articles: [
        "Subscription plans explained",
        "Payment methods",
        "Account security"
      ]
    }
  ];

  const faqs = [
    {
      question: "How do I create my first betting strategy?",
      answer: "Describe your betting idea in natural language to the AI assistant. For example: 'Bet NBA home underdogs in the 2023-24 season when they have 2+ rest days'. The AI will generate a strategy and run a backtest."
    },
    {
      question: "Can I backtest parlays through the chat?",
      answer: "Yes. You can ask for a parlay backtest in plain English and include the sport + time period. Example: 'Backtest a 3-leg NBA parlay of home moneyline favorites in the 2019 season for $10 per parlay'. We’ll build one parlay per day from qualifying games using real odds/results. (Odds coverage is currently historical for NBA/NFL/NHL; more data will be added as it becomes licensed.)"
    },
    {
      question: "What sports do you support?",
      answer: "We focus on sports betting backtesting (NBA, NFL, MLB, NHL and more as data is available). You can ask in plain English and include the time period you want tested."
    },
    {
      question: "How accurate are the backtests?",
      answer: "Our backtests use real market data and account for transaction costs, slippage, and market impact. While past performance doesn't guarantee future results, our testing provides reliable historical analysis."
    },
    {
      question: "Can I download my backtest reports?",
      answer: "Yes! Pro and Elite subscribers can download detailed CSV and text reports of their backtest results, including trade-by-trade analysis and performance metrics."
    },
    {
      question: "What's the difference between subscription tiers?",
      answer: "Free tier includes basic AI chat. Pro ($20/mo) adds unlimited backtests and parameter optimization. Elite ($200/mo) includes advanced analytics and priority support."
    }
  ];

  return (
    <div className="min-h-screen bg-slate-950">
      {/* Header */}
      <header className="bg-slate-900/50 border-b border-white/10">
        <div className="mx-auto max-w-7xl px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Link href="/dashboard" className="text-slate-400 hover:text-white transition">
                <ArrowLeft className="h-5 w-5" />
              </Link>
              <div>
                <h1 className="text-2xl font-bold text-white">Help Center</h1>
                <p className="text-slate-400 text-sm">Find answers and learn how to use EventheOdds.ai</p>
              </div>
            </div>

            {/* Search Bar */}
            <div className="relative max-w-md w-full">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-slate-400" />
              <input
                type="text"
                placeholder="Search help articles..."
                className="w-full bg-slate-800 border border-slate-600 rounded-lg pl-10 pr-4 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none"
              />
            </div>
          </div>
        </div>
      </header>

      <div className="mx-auto max-w-7xl px-6 py-8">
        {/* Categories */}
        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6 mb-12">
          {categories.map((category, index) => (
            <div key={index} className="bg-slate-900/50 border border-white/10 rounded-xl p-6 hover:border-blue-500/30 transition">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-10 h-10 bg-blue-500/20 rounded-lg flex items-center justify-center text-blue-400">
                  {category.icon}
                </div>
                <h3 className="text-lg font-semibold text-white">{category.title}</h3>
              </div>
              <p className="text-slate-400 text-sm mb-4">{category.description}</p>
              <ul className="space-y-2">
                {category.articles.map((article, articleIndex) => (
                  <li key={articleIndex}>
                    <button className="text-blue-400 hover:text-blue-300 text-sm text-left hover:underline">
                      {article}
                    </button>
                  </li>
                ))}
              </ul>
            </div>
          ))}
        </div>

        {/* FAQ Section */}
        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-8">
          <div className="flex items-center gap-3 mb-8">
            <HelpCircle className="h-6 w-6 text-blue-400" />
            <h2 className="text-2xl font-bold text-white">Frequently Asked Questions</h2>
          </div>

          <div className="space-y-6">
            {faqs.map((faq, index) => (
              <div key={index} className="border-b border-white/10 pb-6 last:border-b-0 last:pb-0">
                <h3 className="text-lg font-semibold text-white mb-2">{faq.question}</h3>
                <p className="text-slate-400 leading-relaxed">{faq.answer}</p>
              </div>
            ))}
          </div>
        </div>

        {/* Contact Support */}
        <div className="mt-12 bg-gradient-to-r from-blue-600/20 to-purple-600/20 border border-blue-500/30 rounded-xl p-8 text-center">
          <MessageCircle className="h-12 w-12 text-blue-400 mx-auto mb-4" />
          <h3 className="text-xl font-bold text-white mb-2">Still need help?</h3>
          <p className="text-slate-400 mb-6">
            Can't find what you're looking for? Our support team is here to help.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link
              href="/dashboard?tab=feedback"
              className="bg-blue-600 hover:bg-blue-500 text-white px-6 py-3 rounded-lg font-semibold transition"
            >
              Submit Feedback
            </Link>
            <Link
              href="/dashboard?tab=updates"
              className="bg-slate-700 hover:bg-slate-600 text-white px-6 py-3 rounded-lg font-semibold transition"
            >
              Check Latest Updates
            </Link>
          </div>
        </div>
      </div>
    </div>
  );
}
