'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';
import {
  MessageSquare,
  BarChart3,
  TrendingUp,
  Settings,
  CreditCard,
  Zap,
  Crown,
  Star,
  Clock,
  CheckCircle,
  AlertCircle,
  ArrowRight,
  MessageCircle,
  Megaphone,
  Send,
  Bell,
  Trophy
} from 'lucide-react';
import { ChatInterface } from '@/components/ChatInterface';
import NBADataViewer from '@/components/NBADataViewer';
import ProtectedRoute from '@/components/ProtectedRoute';
import { useUser } from '@/lib/hooks/useUser';
import dynamic from 'next/dynamic';

// Lazy load heavy components
const StrategyPlayground = dynamic(() => import('@/components/StrategyPlayground').then(mod => ({ default: mod.StrategyPlayground })), {
  loading: () => (
    <div className="glass-panel p-12 rounded-xl text-center">
      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500 mx-auto mb-4"></div>
      <p className="text-slate-400">Loading Strategy Playground...</p>
    </div>
  ),
  ssr: false
});

const tierLimits = {
  free: {
    aiGenerations: 5,
    backtests: 100,
    markets: 1,
    chat: false
  },
  pro: {
    aiGenerations: -1, // unlimited
    backtests: -1,
    markets: 4,
    chat: true
  },
  elite: {
    aiGenerations: -1,
    backtests: -1,
    markets: 4,
    chat: true
  }
};

export default function DashboardPage() {
  const router = useRouter();
  const [activeTab, setActiveTab] = useState('chat');
  const [chatDomain] = useState<'sports'>('sports');
  const { user: userProfile, loading: userLoading, error: userError } = useUser();
  const [timeLeft, setTimeLeft] = useState('');

  // Settings handlers
  const handleUpdatePaymentMethod = () => {
    router.push('/settings');
  };

  const handleUpdateBillingEmail = () => {
    router.push('/settings');
  };

  const handleUpdateProfile = () => {
    router.push('/profile');
  };

  const handleMarkAnnouncementRead = (announcementId: string) => {
    const token = localStorage.getItem('authToken');
    void fetch('/api/announcements/read', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
      },
      body: JSON.stringify({ announcementId }),
    })
      .then(() => fetchAnnouncements())
      .catch((e) => console.error('Failed to mark announcement read', e));
  };

  const user = userProfile ? {
    id: userProfile.user.id,
    name: `${userProfile.user.firstName || ''} ${userProfile.user.lastName || ''}`.trim() || 'User',
    email: userProfile.user.email,
    tier: userProfile.user.subscriptionTier.toLowerCase(),
    subscription: {
      status: (userProfile.user.subscriptionStatus || 'ACTIVE').toString().toLowerCase(),
      currentPeriodEnd: userProfile.user.subscriptionEndsAt
        ? new Date(userProfile.user.subscriptionEndsAt)
        : userProfile.user.trialEndsAt
          ? new Date(userProfile.user.trialEndsAt)
          : null,
      cancelAtPeriodEnd: !!userProfile.user.canceledAt
    },
    usage: {
      // Map API response fields to UI expected fields
      aiGenerations: userProfile.stats?.aiGenerations ?? userProfile.stats?.totalConversations ?? 0,
      strategiesCreated: userProfile.stats?.strategiesCreated ?? userProfile.stats?.totalStrategies ?? 0,
      backtestsRun: userProfile.stats?.backtestsRun ?? userProfile.stats?.totalLikes ?? 0,
      chatMessages: userProfile.stats?.chatMessages ?? userProfile.stats?.totalConversations ?? 0
    },
    trialEndsAt: userProfile.user.trialEndsAt ? new Date(userProfile.user.trialEndsAt) : null
  } : null;

  // Feedback form state
  const [feedbackType, setFeedbackType] = useState('bug');
  const [feedbackTitle, setFeedbackTitle] = useState('');
  const [feedbackDescription, setFeedbackDescription] = useState('');
  const [isSubmittingFeedback, setIsSubmittingFeedback] = useState(false);

  // Announcements state
  const [announcements, setAnnouncements] = useState<any[]>([]);
  const [announcementsLoading, setAnnouncementsLoading] = useState(true);
  const [announcementsUnreadCount, setAnnouncementsUnreadCount] = useState(0);

  // Analytics state
  const [analyticsData, setAnalyticsData] = useState<{
    stats: {
      strategiesCreated: number;
      aiGenerations: number;
      backtestsRun: number;
      totalTrades: number;
      totalWins: number;
      avgWinRate: number;
      totalProfit: number;
    };
    recentActivity: Array<{
      id: string;
      type: string;
      title: string;
      description: string;
      sport: string;
      winRate: number;
      createdAt: string;
      data: any;
    }>;
    backtestHistory: Array<{
      id: string;
      strategyName: string;
      market: string;
      status: string;
      createdAt: string;
      results: any;
      performance: any;
    }>;
  } | null>(null);
  const [analyticsLoading, setAnalyticsLoading] = useState(true);
  const [selectedBacktest, setSelectedBacktest] = useState<any>(null);

  async function fetchAnnouncements() {
    setAnnouncementsLoading(true);
    try {
      const token = localStorage.getItem('authToken');
      const response = await fetch('/api/announcements', {
        headers: {
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
      });
        if (response.ok) {
          const data = await response.json();
        setAnnouncements(data.announcements || []);
        setAnnouncementsUnreadCount(Number(data.unreadCount || 0));
        }
      } catch (error) {
        console.error('Failed to fetch announcements:', error);
      } finally {
        setAnnouncementsLoading(false);
      }
  }

  async function fetchAnalytics() {
    setAnalyticsLoading(true);
    try {
      const token = localStorage.getItem('authToken');
      const response = await fetch('/api/user/analytics', {
        headers: {
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
      });
      if (response.ok) {
        const data = await response.json();
        setAnalyticsData(data);
      }
    } catch (error) {
      console.error('Failed to fetch analytics:', error);
    } finally {
      setAnalyticsLoading(false);
    }
  }

  // Helper to format time ago
  function formatTimeAgo(dateString: string) {
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now.getTime() - date.getTime();
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffMins < 1) return 'Just now';
    if (diffMins < 60) return `${diffMins} minute${diffMins > 1 ? 's' : ''} ago`;
    if (diffHours < 24) return `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
    if (diffDays < 7) return `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;
    return date.toLocaleDateString();
  }

  // Fetch announcements and analytics
  useEffect(() => {
    void fetchAnnouncements();
    void fetchAnalytics();
  }, []);

  useEffect(() => {
    if (user && user.tier === 'free' && user.trialEndsAt) {
      const updateTimeLeft = () => {
        const now = new Date();
        const diff = user.trialEndsAt!.getTime() - now.getTime();

        if (diff <= 0) {
          setTimeLeft('Expired');
        } else {
          const days = Math.floor(diff / (1000 * 60 * 60 * 24));
          const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
          setTimeLeft(`${days}d ${hours}h`);
        }
      };

      updateTimeLeft();
      const interval = setInterval(updateTimeLeft, 60000); // Update every minute
      return () => clearInterval(interval);
    }

    setTimeLeft('');
  }, [user]);

  // Show loading state while fetching user data
  if (userLoading) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-slate-950 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto mb-4"></div>
            <p className="text-slate-400">Loading your dashboard...</p>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  // Show error state if user data failed to load
  if (userError || !user) {
    return (
      <ProtectedRoute>
        <div className="min-h-screen bg-slate-950 flex items-center justify-center">
          <div className="text-center">
            <div className="text-red-400 mb-4">Failed to load user data</div>
            <p className="text-slate-400">Please try refreshing the page or contact support.</p>
          </div>
        </div>
      </ProtectedRoute>
    );
  }

  const limits = user ? tierLimits[user.tier as keyof typeof tierLimits] : tierLimits.pro;
  const isFreeTrial = user ? user.tier === 'free' : false;

  const getTierIcon = () => {
    if (!user) return <Star className="h-5 w-5 text-gray-400" />;
    switch (user.tier) {
      case 'free': return <Star className="h-5 w-5 text-yellow-400" />;
      case 'pro': return <Zap className="h-5 w-5 text-blue-400" />;
      case 'elite': return <Crown className="h-5 w-5 text-purple-400" />;
      default: return <Star className="h-5 w-5 text-gray-400" />;
    }
  };

  const getTierColor = () => {
    if (!user) return 'text-gray-400';
    switch (user.tier) {
      case 'free': return 'text-yellow-400';
      case 'pro': return 'text-blue-400';
      case 'elite': return 'text-purple-400';
      default: return 'text-gray-400';
    }
  };

  const handleFeedbackSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!feedbackTitle.trim() || !feedbackDescription.trim()) {
      alert('Please fill in all fields');
      return;
    }

    setIsSubmittingFeedback(true);

    try {
      const token = localStorage.getItem('authToken');
      const response = await fetch('/api/feedback', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
        body: JSON.stringify({
          type: feedbackType,
          title: feedbackTitle,
          description: feedbackDescription,
        }),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to submit feedback');
      }

      const data = await response.json();

      // Reset form
      setFeedbackType('bug');
      setFeedbackTitle('');
      setFeedbackDescription('');

      alert(`Thank you for your feedback! We'll review it and get back to you within ${data.estimatedResponse}.`);

    } catch (error) {
      console.error('Feedback submission failed:', error);
      alert('Failed to submit feedback. Please try again.');
    } finally {
      setIsSubmittingFeedback(false);
    }
  };

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-slate-950">
        {/* Header */}
        <header className="border-b border-white/10 bg-slate-950/50 backdrop-blur">
          <div className="mx-auto max-w-7xl px-6 py-4">
            <div className="flex items-center justify-between">
              <Link href="/" className="text-2xl font-bold text-white">
                LG<span className="text-blue-500">Backtest</span>
              </Link>

              <div className="flex items-center gap-6">
                {/* User Info */}
                <div className="flex items-center gap-3">
                  <div className="text-right">
                    <div className="text-sm font-medium text-white">{user.name}</div>
                    <div className={`text-xs flex items-center gap-1 ${getTierColor()}`}>
                      {getTierIcon()}
                      {user.tier.charAt(0).toUpperCase() + user.tier.slice(1)}
                      {isFreeTrial && (
                        <span className="text-slate-400">• {timeLeft} left</span>
                      )}
                    </div>
                  </div>
                  <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center text-white font-semibold">
                    {user.name.split(' ').map(n => n[0]).join('')}
                  </div>
                </div>

                {/* Upgrade Button */}
                {user.tier === 'free' && (
                  <Link
                    href="/signup?plan=pro"
                    className="bg-gradient-to-r from-blue-600 to-blue-500 hover:from-blue-500 hover:to-blue-400 text-white px-4 py-2 rounded-lg font-semibold text-sm transition"
                  >
                    Upgrade to Pro
                  </Link>
                )}
              </div>
            </div>
          </div>
        </header>

        <div className="mx-auto max-w-7xl px-6 py-8">
          <div className="grid lg:grid-cols-4 gap-8">
            {/* Sidebar */}
            <div className="lg:col-span-1">
              <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6 space-y-4">
                <nav className="space-y-2">
                  {limits.chat && (
                    <button
                      onClick={() => setActiveTab('chat')}
                      className={`w-full flex items-start gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'chat'
                        ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                        : 'text-slate-300 hover:bg-white/5 hover:text-white'
                        }`}
                    >
                      <span className="text-lg mt-0.5">🏆</span>
                      <div>
                        <span className="block">AI Sport Guru</span>
                        <span className="text-xs text-slate-500 block">Stats & analysis</span>
                      </div>
                    </button>
                  )}

                  <button
                    onClick={() => setActiveTab('playground')}
                    className={`w-full flex items-start gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'playground'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <BarChart3 className="h-5 w-5 mt-0.5" />
                    <div>
                      <span className="block">Strategy Playground</span>
                      <span className="text-xs text-slate-500 block">Run backtests</span>
                    </div>
                  </button>

                  <button
                    onClick={() => setActiveTab('analytics')}
                    className={`w-full flex items-center gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'analytics'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <TrendingUp className="h-5 w-5" />
                    Analytics
                  </button>

                  <button
                    onClick={() => setActiveTab('nba')}
                    className={`w-full flex items-center gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'nba'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <Trophy className="h-5 w-5" />
                    NBA Data
                  </button>

                  <button
                    onClick={() => setActiveTab('updates')}
                    className={`w-full flex items-center gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'updates'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <Megaphone className="h-5 w-5" />
                    Updates
                    {announcementsUnreadCount > 0 && (
                      <span className="ml-auto bg-red-500 text-white text-xs px-2 py-1 rounded-full">
                        {announcementsUnreadCount}
                      </span>
                    )}
                  </button>

                  <button
                    onClick={() => setActiveTab('feedback')}
                    className={`w-full flex items-center gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'feedback'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <MessageCircle className="h-5 w-5" />
                    Feedback
                  </button>

                  <button
                    onClick={() => setActiveTab('settings')}
                    className={`w-full flex items-center gap-3 px-3 py-2 rounded-lg text-left transition ${activeTab === 'settings'
                      ? 'bg-blue-500/10 text-blue-400 border border-blue-500/20'
                      : 'text-slate-300 hover:bg-white/5 hover:text-white'
                      }`}
                  >
                    <Settings className="h-5 w-5" />
                    Settings
                  </button>
                </nav>

                {/* Usage Stats */}
                <div className="border-t border-white/10 pt-4">
                  <h4 className="text-sm font-semibold text-white mb-3">Usage This Month</h4>
                  <div className="space-y-3">
                    <div>
                      <div className="flex justify-between text-xs text-slate-400 mb-1">
                        <span>AI Generations</span>
                        <span>{user.usage.aiGenerations}/{limits.aiGenerations === -1 ? '∞' : limits.aiGenerations}</span>
                      </div>
                      <div className="w-full bg-slate-700 rounded-full h-2">
                        <div
                          className="bg-blue-500 h-2 rounded-full"
                          style={{
                            width: limits.aiGenerations === -1
                              ? '30%'
                              : `${Math.min((user.usage.aiGenerations / limits.aiGenerations) * 100, 100)}%`
                          }}
                        ></div>
                      </div>
                    </div>

                    <div>
                      <div className="flex justify-between text-xs text-slate-400 mb-1">
                        <span>Backtests Run</span>
                        <span>{user.usage.backtestsRun}/{limits.backtests === -1 ? '∞' : limits.backtests}</span>
                      </div>
                      <div className="w-full bg-slate-700 rounded-full h-2">
                        <div
                          className="bg-green-500 h-2 rounded-full"
                          style={{
                            width: limits.backtests === -1
                              ? '45%'
                              : `${Math.min((user.usage.backtestsRun / limits.backtests) * 100, 100)}%`
                          }}
                        ></div>
                      </div>
                    </div>

                    {limits.chat && (
                      <div>
                        <div className="flex justify-between text-xs text-slate-400 mb-1">
                          <span>Chat Messages</span>
                          <span>{user.usage.chatMessages}/∞</span>
                        </div>
                        <div className="w-full bg-slate-700 rounded-full h-2">
                          <div className="bg-purple-500 h-2 rounded-full w-1/3"></div>
                        </div>
                      </div>
                    )}
                  </div>
                </div>

                {/* Upgrade Prompt */}
                {user.tier === 'free' && (
                  <div className="border-t border-white/10 pt-4">
                    <div className="bg-gradient-to-r from-blue-500/10 to-purple-500/10 border border-blue-500/20 rounded-lg p-4">
                      <div className="flex items-start gap-3">
                        <Zap className="h-5 w-5 text-blue-400 mt-0.5" />
                        <div>
                          <h4 className="font-semibold text-white mb-1">Upgrade to Pro</h4>
                          <p className="text-slate-300 text-sm mb-3">
                            Unlock unlimited AI generations and all markets.
                          </p>
                          <Link
                            href="/signup?plan=pro"
                            className="bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded text-sm font-semibold transition"
                          >
                            Upgrade Now
                          </Link>
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* Main Content */}
            <div className="lg:col-span-3">
              {activeTab === 'chat' && limits.chat && (
                <div className="space-y-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                    <h1 className="text-2xl font-bold text-white flex items-center gap-2">
                      <span className="text-2xl">🏆</span> AI Sport Guru
                    </h1>
                      <div className="flex bg-slate-900/50 border border-white/10 rounded-lg p-1">
                        <span className="px-3 py-1 rounded-md text-xs font-medium bg-blue-600 text-white">
                          Sports Betting
                        </span>
                      </div>
                    </div>
                    {isFreeTrial && (
                      <div className="flex items-center gap-2 bg-yellow-500/10 border border-yellow-500/20 rounded-lg px-3 py-2">
                        <Clock className="h-4 w-4 text-yellow-400" />
                        <span className="text-yellow-400 text-sm">{timeLeft} trial remaining</span>
                      </div>
                    )}
                  </div>
                  <ChatInterface domain={chatDomain} />
                </div>
              )}

              {activeTab === 'chat' && !limits.chat && (
                <div className="bg-slate-900/50 border border-white/10 rounded-xl p-12 text-center">
                  <div className="max-w-md mx-auto">
                    <span className="text-6xl mb-6 block">🏆</span>
                    <h2 className="text-2xl font-bold text-white mb-4">AI Sport Guru - Pro Feature</h2>
                    <p className="text-slate-400 mb-6">
                      Upgrade to Pro to access AI Sport Guru – your intelligent assistant for sports statistics, analysis, and strategy backtesting.
                    </p>
                    <Link
                      href="/signup?plan=pro"
                      className="bg-blue-600 hover:bg-blue-500 text-white px-6 py-3 rounded-lg font-semibold transition inline-flex items-center gap-2"
                    >
                      Upgrade to Pro
                      <ArrowRight className="h-5 w-5" />
                    </Link>
                  </div>
                </div>
              )}

              {activeTab === 'playground' && (
                <div className="space-y-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <h1 className="text-2xl font-bold text-white flex items-center gap-2">
                        <BarChart3 className="h-6 w-6 text-blue-400" />
                        Strategy Playground
                      </h1>
                      <p className="text-slate-400 text-sm mt-1">
                        Test betting strategies against real historical data
                      </p>
                    </div>
                    {isFreeTrial && (
                      <div className="bg-slate-900/50 border border-white/10 rounded-lg px-4 py-2">
                        <span className="text-slate-400 text-sm">
                          {limits.aiGenerations - user.usage.aiGenerations} AI generations remaining
                        </span>
                      </div>
                    )}
                  </div>
                  
                  {/* Beginner Guide */}
                  <div className="bg-gradient-to-r from-green-500/10 to-blue-500/10 border border-green-500/20 rounded-xl p-4">
                    <h3 className="font-semibold text-white mb-3 flex items-center gap-2">
                      <span>🎯</span> Quick Start Guide
                    </h3>
                    <div className="grid md:grid-cols-3 gap-4 text-sm">
                      <div className="flex items-start gap-2">
                        <span className="bg-blue-500 text-white text-xs w-5 h-5 rounded-full flex items-center justify-center flex-shrink-0">1</span>
                        <div>
                          <p className="text-white font-medium">Describe a Strategy</p>
                          <p className="text-slate-400">Type a betting rule in plain English or pick a preset</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-2">
                        <span className="bg-blue-500 text-white text-xs w-5 h-5 rounded-full flex items-center justify-center flex-shrink-0">2</span>
                        <div>
                          <p className="text-white font-medium">Run Backtest</p>
                          <p className="text-slate-400">We test against real sportsbook odds & results</p>
                        </div>
                      </div>
                      <div className="flex items-start gap-2">
                        <span className="bg-blue-500 text-white text-xs w-5 h-5 rounded-full flex items-center justify-center flex-shrink-0">3</span>
                        <div>
                          <p className="text-white font-medium">Analyze Results</p>
                          <p className="text-slate-400">See win rate, ROI, and detailed trade history</p>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  <StrategyPlayground />
                </div>
              )}

              {activeTab === 'nba' && (
                <div className="space-y-6">
                  <h1 className="text-2xl font-bold text-white">NBA Data Explorer</h1>
                  <NBADataViewer />
                </div>
              )}

              {activeTab === 'analytics' && (
                <div className="space-y-6">
                  <div className="flex items-center justify-between">
                    <h1 className="text-2xl font-bold text-white">Analytics Dashboard</h1>
                    <button
                      onClick={() => fetchAnalytics()}
                      className="text-sm text-blue-400 hover:text-blue-300 flex items-center gap-1"
                    >
                      <TrendingUp className="h-4 w-4" />
                      Refresh
                    </button>
                  </div>

                  {analyticsLoading ? (
                    <div className="flex items-center justify-center py-12">
                      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-400"></div>
                    </div>
                  ) : (
                    <>
                      {/* Stats Grid */}
                      <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
                        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                          <div className="flex items-center justify-between">
                            <div>
                              <p className="text-sm text-slate-400">Strategies Created</p>
                              <p className="text-2xl font-bold text-white">
                                {analyticsData?.stats.strategiesCreated || user.usage.strategiesCreated}
                              </p>
                            </div>
                            <BarChart3 className="h-8 w-8 text-blue-400" />
                          </div>
                        </div>

                        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                          <div className="flex items-center justify-between">
                            <div>
                              <p className="text-sm text-slate-400">AI Generations</p>
                              <p className="text-2xl font-bold text-white">
                                {analyticsData?.stats.aiGenerations || user.usage.aiGenerations}
                              </p>
                            </div>
                            <Zap className="h-8 w-8 text-yellow-400" />
                          </div>
                        </div>

                        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                          <div className="flex items-center justify-between">
                            <div>
                              <p className="text-sm text-slate-400">Backtests Run</p>
                              <p className="text-2xl font-bold text-white">
                                {analyticsData?.stats.backtestsRun || user.usage.backtestsRun}
                              </p>
                            </div>
                            <TrendingUp className="h-8 w-8 text-green-400" />
                          </div>
                        </div>

                        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                          <div className="flex items-center justify-between">
                            <div>
                              <p className="text-sm text-slate-400">Avg Win Rate</p>
                              <p className={`text-2xl font-bold ${(analyticsData?.stats.avgWinRate || 0) >= 50 ? 'text-green-400' : 'text-white'}`}>
                                {analyticsData?.stats.avgWinRate || 0}%
                              </p>
                            </div>
                            <CheckCircle className="h-8 w-8 text-emerald-400" />
                          </div>
                        </div>
                      </div>

                      {/* Recent Activity - Clickable */}
                      <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                        <h3 className="text-lg font-semibold text-white mb-4 flex items-center gap-2">
                          <Clock className="h-5 w-5 text-blue-400" />
                          Recent Activity
                        </h3>
                        {analyticsData?.recentActivity && analyticsData.recentActivity.length > 0 ? (
                          <div className="space-y-3">
                            {analyticsData.recentActivity.map((activity) => (
                              <button
                                key={activity.id}
                                onClick={() => setSelectedBacktest(activity)}
                                className="w-full flex items-start gap-3 p-3 rounded-lg hover:bg-slate-800/50 transition text-left group"
                              >
                                <div className={`w-2 h-2 rounded-full mt-2 ${
                                  activity.type === 'backtest' ? 'bg-green-400' : 'bg-blue-400'
                                }`}></div>
                                <div className="flex-1">
                                  <p className="text-white text-sm group-hover:text-blue-300 transition">{activity.title}</p>
                                  <p className="text-slate-400 text-xs">{activity.description}</p>
                                  <p className="text-slate-500 text-xs mt-1">{formatTimeAgo(activity.createdAt)}</p>
                                </div>
                                <div className="text-slate-400 group-hover:text-blue-400">
                                  <ArrowRight className="h-4 w-4" />
                                </div>
                              </button>
                            ))}
                          </div>
                        ) : (
                          <div className="text-center py-8">
                            <BarChart3 className="h-12 w-12 text-slate-600 mx-auto mb-3" />
                            <p className="text-slate-400">No activity yet</p>
                            <p className="text-slate-500 text-sm">Run a backtest to see your history here</p>
                          </div>
                        )}
                      </div>

                      {/* Backtest History */}
                      {analyticsData?.backtestHistory && analyticsData.backtestHistory.length > 0 && (
                        <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                          <h3 className="text-lg font-semibold text-white mb-4 flex items-center gap-2">
                            <Trophy className="h-5 w-5 text-yellow-400" />
                            Backtest History
                          </h3>
                          <div className="overflow-x-auto">
                            <table className="w-full text-sm">
                              <thead>
                                <tr className="border-b border-slate-700">
                                  <th className="text-left py-2 px-3 text-slate-400 font-medium">Strategy</th>
                                  <th className="text-left py-2 px-3 text-slate-400 font-medium">Sport</th>
                                  <th className="text-right py-2 px-3 text-slate-400 font-medium">Win Rate</th>
                                  <th className="text-right py-2 px-3 text-slate-400 font-medium">Trades</th>
                                  <th className="text-right py-2 px-3 text-slate-400 font-medium">Profit</th>
                                  <th className="text-left py-2 px-3 text-slate-400 font-medium">Date</th>
                                  <th className="text-center py-2 px-3 text-slate-400 font-medium">Action</th>
                                </tr>
                              </thead>
                              <tbody>
                                {analyticsData.backtestHistory.map((bt) => {
                                  const winRate = bt.results?.win_rate || bt.results?.results?.win_rate || 0;
                                  const totalTrades = bt.results?.total_trades || bt.results?.results?.total_trades || 0;
                                  const profit = bt.results?.total_profit || bt.results?.results?.total_profit || 0;
                                  
                                  return (
                                    <tr key={bt.id} className="border-b border-slate-700/50 hover:bg-slate-800/30">
                                      <td className="py-3 px-3 text-white font-medium">{bt.strategyName || 'Unnamed'}</td>
                                      <td className="py-3 px-3">
                                        <span className="bg-blue-500/20 text-blue-300 px-2 py-1 rounded text-xs uppercase">
                                          {bt.market || 'Sports'}
                                        </span>
                                      </td>
                                      <td className={`py-3 px-3 text-right font-semibold ${winRate >= 50 ? 'text-green-400' : 'text-red-400'}`}>
                                        {winRate.toFixed(1)}%
                                      </td>
                                      <td className="py-3 px-3 text-right text-slate-300">{totalTrades}</td>
                                      <td className={`py-3 px-3 text-right font-semibold ${profit >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                        ${profit.toFixed(2)}
                                      </td>
                                      <td className="py-3 px-3 text-slate-400">{formatTimeAgo(bt.createdAt)}</td>
                                      <td className="py-3 px-3 text-center">
                                        <button
                                          onClick={() => setSelectedBacktest({ ...bt, type: 'backtest' })}
                                          className="text-blue-400 hover:text-blue-300 text-sm"
                                        >
                                          View →
                                        </button>
                                      </td>
                                    </tr>
                                  );
                                })}
                              </tbody>
                            </table>
                          </div>
                        </div>
                      )}
                    </>
                  )}

                  {/* Selected Backtest Modal */}
                  {selectedBacktest && (
                    <div className="fixed inset-0 bg-black/70 flex items-center justify-center z-50 p-4">
                      <div className="bg-slate-900 border border-slate-700 rounded-xl max-w-2xl w-full max-h-[80vh] overflow-y-auto">
                        <div className="flex items-center justify-between p-4 border-b border-slate-700">
                          <h3 className="text-lg font-semibold text-white">
                            {selectedBacktest.title || selectedBacktest.strategyName || 'Backtest Details'}
                          </h3>
                          <button
                            onClick={() => setSelectedBacktest(null)}
                            className="text-slate-400 hover:text-white"
                          >
                            ✕
                          </button>
                        </div>
                        <div className="p-4 space-y-4">
                          {/* Summary Stats */}
                          <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                            <div className="bg-slate-800 rounded-lg p-3 text-center">
                              <p className="text-2xl font-bold text-white">
                                {selectedBacktest.results?.total_trades || selectedBacktest.data?.results?.total_trades || 0}
                              </p>
                              <p className="text-xs text-slate-400">Total Trades</p>
                            </div>
                            <div className="bg-slate-800 rounded-lg p-3 text-center">
                              <p className="text-2xl font-bold text-green-400">
                                {(selectedBacktest.winRate || selectedBacktest.results?.win_rate || 0).toFixed(1)}%
                              </p>
                              <p className="text-xs text-slate-400">Win Rate</p>
                            </div>
                            <div className="bg-slate-800 rounded-lg p-3 text-center">
                              <p className={`text-2xl font-bold ${(selectedBacktest.results?.total_profit || 0) >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                                ${(selectedBacktest.results?.total_profit || 0).toFixed(2)}
                              </p>
                              <p className="text-xs text-slate-400">Profit/Loss</p>
                            </div>
                            <div className="bg-slate-800 rounded-lg p-3 text-center">
                              <p className="text-2xl font-bold text-blue-400 uppercase">
                                {selectedBacktest.market || selectedBacktest.sport || 'Sports'}
                              </p>
                              <p className="text-xs text-slate-400">Sport</p>
                            </div>
                          </div>

                          {/* Strategy Name */}
                          <div className="bg-slate-800/50 rounded-lg p-4">
                            <p className="text-sm text-slate-400 mb-1">Strategy</p>
                            <p className="text-white">{selectedBacktest.strategyName || selectedBacktest.title || 'Custom Strategy'}</p>
                          </div>

                          {/* Date */}
                          <div className="bg-slate-800/50 rounded-lg p-4">
                            <p className="text-sm text-slate-400 mb-1">Run Date</p>
                            <p className="text-white">{new Date(selectedBacktest.createdAt).toLocaleString()}</p>
                          </div>

                          {/* Additional Performance Data */}
                          {selectedBacktest.performance && Object.keys(selectedBacktest.performance).length > 0 && (
                            <div className="bg-slate-800/50 rounded-lg p-4">
                              <p className="text-sm text-slate-400 mb-2">Performance Metrics</p>
                              <div className="grid grid-cols-2 gap-2 text-sm">
                                {Object.entries(selectedBacktest.performance).slice(0, 6).map(([key, value]) => (
                                  <div key={key} className="flex justify-between">
                                    <span className="text-slate-400 capitalize">{key.replace(/_/g, ' ')}:</span>
                                    <span className="text-white">{typeof value === 'number' ? value.toFixed(2) : String(value)}</span>
                                  </div>
                                ))}
                              </div>
                            </div>
                          )}
                        </div>
                        <div className="p-4 border-t border-slate-700">
                          <button
                            onClick={() => setSelectedBacktest(null)}
                            className="w-full bg-blue-600 hover:bg-blue-500 text-white py-2 rounded-lg font-semibold transition"
                          >
                            Close
                          </button>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              )}

              {activeTab === 'settings' && (
                <div className="space-y-6">
                  <h1 className="text-2xl font-bold text-white">Account Settings</h1>

                  <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                    <h3 className="text-lg font-semibold text-white mb-2">Manage your account</h3>
                    <p className="text-slate-400 text-sm mb-4">
                      Subscription, billing, privacy/export, and account deletion live in the dedicated Settings page.
                    </p>

                    <div className="flex flex-wrap items-center justify-between gap-3 rounded-lg bg-slate-800/50 p-4">
                      <div>
                        <p className="font-semibold text-white capitalize">{user.tier} Plan</p>
                        <p className="text-slate-400 text-sm">
                          Status: {user.subscription.status}
                          {user.subscription.currentPeriodEnd ? (
                            <span className="text-slate-500"> • ends {user.subscription.currentPeriodEnd.toLocaleDateString()}</span>
                          ) : null}
                        </p>
                      </div>

                          <Link
                        href="/settings"
                            className="bg-blue-600 hover:bg-blue-500 text-white px-4 py-2 rounded-lg font-semibold text-sm transition"
                          >
                        Open Settings →
                          </Link>
                    </div>
                  </div>
                </div>
              )}

              {activeTab === 'updates' && (
                <div className="space-y-6">
                  <div className="flex items-center justify-between">
                    <h1 className="text-2xl font-bold text-white">Platform Updates & Announcements</h1>
                    <div className="flex items-center gap-2 bg-blue-500/10 border border-blue-500/20 rounded-lg px-3 py-2">
                      <Bell className="h-4 w-4 text-blue-400" />
                      <span className="text-blue-400 text-sm">
                        {announcementsLoading ? 'Loading...' : `${announcements.length} updates`}
                      </span>
                    </div>
                  </div>

                  {announcementsLoading ? (
                    <div className="flex items-center justify-center py-12">
                      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-400"></div>
                    </div>
                  ) : announcements.length === 0 ? (
                    <div className="text-center py-12">
                      <Bell className="h-12 w-12 text-slate-600 mx-auto mb-4" />
                      <h3 className="text-lg font-semibold text-slate-400 mb-2">No announcements yet</h3>
                      <p className="text-slate-500">Check back later for platform updates and news.</p>
                    </div>
                  ) : (
                    <div className="space-y-4">
                      {announcements.map((announcement) => (
                        <div
                          key={announcement.id}
                          className={`bg-slate-900/50 border rounded-xl p-6 ${announcement.important
                            ? 'border-red-500/30 bg-red-500/5'
                            : 'border-white/10'
                            }`}
                        >
                          <div className="flex items-start justify-between mb-3">
                            <div className="flex items-center gap-3">
                              <div className={`w-3 h-3 rounded-full ${announcement.type === 'feature' ? 'bg-green-400' :
                                announcement.type === 'security' ? 'bg-red-400' :
                                  announcement.type === 'performance' ? 'bg-blue-400' :
                                    'bg-yellow-400'
                                }`}></div>
                              <h3 className="text-lg font-semibold text-white">{announcement.title}</h3>
                              {announcement.important && (
                                <span className="bg-red-500/20 text-red-400 text-xs px-2 py-1 rounded-full font-semibold">
                                  IMPORTANT
                                </span>
                              )}
                            </div>
                            <span className="text-slate-400 text-sm">{announcement.date}</span>
                          </div>

                          <p className="text-slate-300 leading-relaxed mb-4">{announcement.content}</p>

                          <div className="flex items-center justify-between">
                            <span className={`text-xs px-3 py-1 rounded-full font-semibold ${announcement.type === 'feature' ? 'bg-green-500/20 text-green-400' :
                              announcement.type === 'security' ? 'bg-red-500/20 text-red-400' :
                                announcement.type === 'performance' ? 'bg-blue-500/20 text-blue-400' :
                                  'bg-yellow-500/20 text-yellow-400'
                              }`}>
                              {announcement.type.charAt(0).toUpperCase() + announcement.type.slice(1)} Update
                            </span>
                            <button
                              onClick={() => handleMarkAnnouncementRead(announcement.id)}
                              className="text-blue-400 hover:text-blue-300 text-sm"
                            >
                              Mark as read →
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}

              {activeTab === 'feedback' && (
                <div className="space-y-6">
                  <div className="flex items-center justify-between">
                    <h1 className="text-2xl font-bold text-white">Feedback & Support</h1>
                    <div className="text-sm text-slate-400">
                      Help us improve EventTheOdds
                    </div>
                  </div>

                  <div className="grid md:grid-cols-2 gap-6">
                    {/* Feedback Form */}
                    <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                      <h3 className="text-lg font-semibold text-white mb-4 flex items-center gap-2">
                        <MessageCircle className="h-5 w-5 text-blue-400" />
                        Submit Feedback
                      </h3>

                      <form onSubmit={handleFeedbackSubmit} className="space-y-4">
                        <div>
                          <label className="block text-sm font-medium text-slate-300 mb-2">
                            Type of Feedback
                          </label>
                          <select
                            value={feedbackType}
                            onChange={(e) => setFeedbackType(e.target.value)}
                            className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white focus:border-blue-500 focus:outline-none"
                          >
                            <option value="bug">🐛 Bug Report</option>
                            <option value="feature">✨ Feature Request</option>
                            <option value="improvement">🔧 Improvement Suggestion</option>
                            <option value="question">❓ Question</option>
                            <option value="other">📝 Other</option>
                          </select>
                        </div>

                        <div>
                          <label className="block text-sm font-medium text-slate-300 mb-2">
                            Title
                          </label>
                          <input
                            type="text"
                            required
                            value={feedbackTitle}
                            onChange={(e) => setFeedbackTitle(e.target.value)}
                            placeholder="Brief summary of your feedback"
                            className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none"
                          />
                        </div>

                        <div>
                          <label className="block text-sm font-medium text-slate-300 mb-2">
                            Description
                          </label>
                          <textarea
                            required
                            value={feedbackDescription}
                            onChange={(e) => setFeedbackDescription(e.target.value)}
                            placeholder="Please provide detailed information about your feedback, including steps to reproduce if it's a bug..."
                            rows={6}
                            className="w-full bg-slate-800 border border-slate-600 rounded-lg px-3 py-2 text-white placeholder-slate-400 focus:border-blue-500 focus:outline-none resize-none"
                          />
                        </div>

                        <button
                          type="submit"
                          disabled={isSubmittingFeedback}
                          className="w-full bg-blue-600 hover:bg-blue-500 disabled:opacity-50 disabled:cursor-not-allowed text-white py-3 px-4 rounded-lg font-semibold transition flex items-center justify-center gap-2"
                        >
                          {isSubmittingFeedback ? (
                            <>
                              <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                              Submitting...
                            </>
                          ) : (
                            <>
                              <Send className="h-4 w-4" />
                              Submit Feedback
                            </>
                          )}
                        </button>
                      </form>
                    </div>

                    {/* Support Options */}
                    <div className="space-y-6">
                      <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                        <h3 className="text-lg font-semibold text-white mb-4">Quick Support</h3>
                        <div className="space-y-3">
                          <div className="flex items-start gap-3 p-3 bg-slate-800/50 rounded-lg">
                            <div className="w-8 h-8 bg-blue-500/20 rounded-lg flex items-center justify-center">
                              <MessageSquare className="h-4 w-4 text-blue-400" />
                            </div>
                            <div>
                              <h4 className="font-semibold text-white">Live Chat</h4>
                              <p className="text-slate-400 text-sm">Get instant help from our support team</p>
                              <Link href="/support" className="text-blue-400 hover:text-blue-300 text-sm mt-1 inline-block">
                                Start Chat →
                              </Link>
                            </div>
                          </div>

                          <div className="flex items-start gap-3 p-3 bg-slate-800/50 rounded-lg">
                            <div className="w-8 h-8 bg-green-500/20 rounded-lg flex items-center justify-center">
                              <CheckCircle className="h-4 w-4 text-green-400" />
                            </div>
                            <div>
                              <h4 className="font-semibold text-white">Help Center</h4>
                              <p className="text-slate-400 text-sm">Browse FAQs and tutorials</p>
                              <Link href="/help" className="text-blue-400 hover:text-blue-300 text-sm mt-1 inline-block">
                                Visit Help Center →
                              </Link>
                            </div>
                          </div>

                          <div className="flex items-start gap-3 p-3 bg-slate-800/50 rounded-lg">
                            <div className="w-8 h-8 bg-purple-500/20 rounded-lg flex items-center justify-center">
                              <Megaphone className="h-4 w-4 text-purple-400" />
                            </div>
                            <div>
                              <h4 className="font-semibold text-white">Community Forum</h4>
                              <p className="text-slate-400 text-sm">Connect with other traders</p>
                              <Link href="/community" className="text-blue-400 hover:text-blue-300 text-sm mt-1 inline-block">
                                Join Community →
                              </Link>
                            </div>
                          </div>
                        </div>
                      </div>

                      <div className="bg-slate-900/50 border border-white/10 rounded-xl p-6">
                        <h3 className="text-lg font-semibold text-white mb-4">Response Time</h3>
                        <div className="space-y-2">
                          <div className="flex justify-between text-sm">
                            <span className="text-slate-400">Bug Reports:</span>
                            <span className="text-green-400">24-48 hours</span>
                          </div>
                          <div className="flex justify-between text-sm">
                            <span className="text-slate-400">Feature Requests:</span>
                            <span className="text-blue-400">1-2 weeks</span>
                          </div>
                          <div className="flex justify-between text-sm">
                            <span className="text-slate-400">General Questions:</span>
                            <span className="text-purple-400">Same day</span>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>

      </div>
    </ProtectedRoute>
  );
}
