'use client';

import { useEffect, useMemo, useState } from 'react';
import Link from 'next/link';
import { useSearchParams } from 'next/navigation';
import { BarChart3, Download, Loader2, RefreshCw } from 'lucide-react';
import ProtectedRoute from '@/components/ProtectedRoute';
import BacktestChart from '@/components/BacktestChart';

export default function BacktestPage() {
  const searchParams = useSearchParams();
  const preselectId = searchParams.get('id');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [backtests, setBacktests] = useState<any[]>([]);
  const [selectedId, setSelectedId] = useState<string | null>(null);
  const [selected, setSelected] = useState<any | null>(null);
  const [detailsLoading, setDetailsLoading] = useState(false);
  const [search, setSearch] = useState('');

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    if (!q) return backtests;
    return backtests.filter((b) =>
      (b.market || '').toLowerCase().includes(q) ||
      (b.strategyName || '').toLowerCase().includes(q)
    );
  }, [backtests, search]);

  async function fetchBacktests() {
    setLoading(true);
    setError(null);
    try {
      const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
      const res = await fetch('/api/backtests?limit=50', {
        headers: {
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
      });
      const data = await res.json();
      if (!res.ok) throw new Error(data?.error || 'Failed to load backtests');
      setBacktests(data.backtests || []);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to load backtests');
    } finally {
      setLoading(false);
    }
  }

  async function fetchBacktestDetails(id: string) {
    setDetailsLoading(true);
    setSelected(null);
    try {
      const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
      const res = await fetch(`/api/backtests/${id}`, {
        headers: {
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
      });
      const data = await res.json();
      if (!res.ok) throw new Error(data?.error || 'Failed to load backtest');
      setSelected(data);
    } catch (e) {
      setError(e instanceof Error ? e.message : 'Failed to load backtest');
    } finally {
      setDetailsLoading(false);
    }
  }

  async function downloadReport(format: 'csv' | 'txt') {
    if (!selected?.results) return;

    const token = typeof window !== 'undefined' ? localStorage.getItem('authToken') : null;
    const res = await fetch('/api/reports/generate', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
      },
      body: JSON.stringify({
        backtestResults: selected.results,
        strategyName: selected.strategyName || 'Backtest',
        market: (selected.market || 'SPORTS').toString().toUpperCase(),
        timestamp: selected.createdAt || new Date().toISOString(),
        format,
      }),
    });

    if (!res.ok) {
      const data = await res.json().catch(() => ({}));
      throw new Error(data?.error || 'Failed to generate report');
    }

    const blob = await res.blob();
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.style.display = 'none';
    a.href = url;
    a.download = `eventheodds-backtest-${selected.market || 'sports'}-${selected.id}.${format === 'csv' ? 'csv' : 'txt'}`;
    document.body.appendChild(a);
    a.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(a);
  }

  useEffect(() => {
    void fetchBacktests();
  }, []);

  useEffect(() => {
    if (!preselectId) return;
    setSelectedId(preselectId);
    void fetchBacktestDetails(preselectId);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [preselectId]);

  return (
    <ProtectedRoute>
      <div className="min-h-screen bg-slate-950">
        <div className="max-w-7xl mx-auto px-6 py-12">
          <div className="flex flex-col md:flex-row md:items-end md:justify-between gap-6 mb-10">
            <div>
              <h1 className="text-4xl font-bold text-white mb-2">Backtest History</h1>
              <p className="text-slate-400 max-w-2xl">
                Review your past backtests, open charts, and download reports.
              </p>
            </div>
            <div className="flex gap-3">
              <button
                onClick={() => void fetchBacktests()}
                className="inline-flex items-center gap-2 rounded-xl border border-white/10 bg-slate-900/50 px-4 py-2 text-white hover:bg-white/5"
                disabled={loading}
              >
                {loading ? <Loader2 className="h-4 w-4 animate-spin" /> : <RefreshCw className="h-4 w-4" />}
                Refresh
              </button>
              <Link
                href="/dashboard"
                className="inline-flex items-center gap-2 rounded-xl bg-blue-600 px-4 py-2 text-white font-semibold hover:bg-blue-500"
              >
                Run a new backtest
              </Link>
            </div>
          </div>

          {error && (
            <div className="mb-6 rounded-xl border border-red-500/20 bg-red-500/10 p-4 text-red-200">
              {error}
            </div>
          )}

          <div className="grid lg:grid-cols-3 gap-6">
            <div className="lg:col-span-1 rounded-2xl border border-white/10 bg-slate-900/50 p-4">
              <div className="flex items-center justify-between gap-3 mb-4">
                <input
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  placeholder="Search market or strategy..."
                  className="w-full rounded-xl border border-white/10 bg-slate-950/40 px-3 py-2 text-sm text-white placeholder-slate-500"
                />
              </div>

              {loading ? (
                <div className="flex items-center gap-3 text-slate-300">
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Loading backtests…
                </div>
              ) : filtered.length === 0 ? (
                <div className="text-slate-400 text-sm">
                  No backtests yet. Run one from the dashboard chat.
                </div>
              ) : (
                <div className="space-y-2">
                  {filtered.map((bt) => (
                    (() => {
                      const profitRaw = bt.summary?.totalProfit;
                      const profitNum = profitRaw != null ? Number(profitRaw) : null;
                      const hasProfit = profitNum != null && Number.isFinite(profitNum);
                      const profitClass = !hasProfit
                        ? 'text-slate-400'
                        : profitNum >= 0
                          ? 'text-emerald-300'
                          : 'text-rose-300';
                      const profitText = !hasProfit
                        ? '—'
                        : `${profitNum >= 0 ? '+' : '-'}$${Math.abs(profitNum).toFixed(2)}`;

                      return (
                    <button
                      key={bt.id}
                      onClick={() => {
                        setSelectedId(bt.id);
                        void fetchBacktestDetails(bt.id);
                      }}
                      className={`w-full rounded-xl border px-4 py-3 text-left transition ${
                        selectedId === bt.id
                          ? 'border-blue-500/40 bg-blue-500/10'
                          : 'border-white/10 bg-slate-950/30 hover:bg-white/5'
                      }`}
                    >
                      <div className="flex items-center justify-between gap-3">
                        <div className="text-white font-semibold text-sm">
                          {(bt.market || 'SPORTS').toString().toUpperCase()}
                        </div>
                        <div className="text-xs text-slate-400" suppressHydrationWarning>
                          {new Date(bt.createdAt).toLocaleDateString()}
                        </div>
                      </div>
                      <div className="mt-1 text-slate-300 text-sm truncate">{bt.strategyName}</div>
                      <div className="mt-2 flex items-center gap-4 text-xs text-slate-400">
                        <span>{bt.summary?.totalTrades ?? 0} trades</span>
                        <span>{bt.summary?.winRate != null ? `${Number(bt.summary.winRate).toFixed(1)}%` : '—'} win</span>
                        <span className={profitClass}>{profitText}</span>
              </div>
                    </button>
                      );
                    })()
                  ))}
                </div>
              )}
              </div>

            <div className="lg:col-span-2 rounded-2xl border border-white/10 bg-slate-900/50 p-6">
              {!selectedId && (
                <div className="text-slate-400">
                  Select a backtest on the left to view details.
                </div>
              )}

              {selectedId && detailsLoading && (
                <div className="flex items-center gap-3 text-slate-300">
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Loading backtest details…
                </div>
              )}

              {selected && (
                <div className="space-y-6">
                  <div className="flex items-start justify-between gap-4">
                    <div>
                      <div className="text-white text-xl font-bold">{selected.strategyName}</div>
                      <div className="text-slate-400 text-sm">
                        Market: {(selected.market || 'sports').toString().toUpperCase()} •{' '}
                        {selected.status} •{' '}
                        <span suppressHydrationWarning>{new Date(selected.createdAt).toLocaleString()}</span>
                      </div>
                      {(() => {
                        const profitRaw =
                          selected?.results?.results?.total_profit ??
                          selected?.results?.total_profit ??
                          null;
                        const profitNum = profitRaw != null ? Number(profitRaw) : null;
                        if (profitNum == null || !Number.isFinite(profitNum)) return null;

                        const isProfit = profitNum >= 0;
                        return (
                          <div
                            className={`mt-3 inline-flex items-center gap-2 rounded-full border px-3 py-1 text-xs font-semibold ${
                              isProfit
                                ? 'border-emerald-500/30 bg-emerald-500/10 text-emerald-200'
                                : 'border-rose-500/30 bg-rose-500/10 text-rose-200'
                            }`}
                          >
                            <span className="uppercase tracking-widest">{isProfit ? 'Profit' : 'Loss'}</span>
                            <span className="font-mono">
                              {isProfit ? '+' : '-'}${Math.abs(profitNum).toFixed(2)}
                            </span>
                          </div>
                        );
                      })()}
                    </div>
                    <div className="flex gap-2">
                      <button
                        onClick={() => void downloadReport('csv').catch((e) => setError(e.message))}
                        className="inline-flex items-center gap-2 rounded-xl bg-green-600 px-4 py-2 text-white font-semibold hover:bg-green-500"
                      >
                        <Download className="h-4 w-4" />
                        CSV
                      </button>
                      <button
                        onClick={() => void downloadReport('txt').catch((e) => setError(e.message))}
                        className="inline-flex items-center gap-2 rounded-xl bg-blue-600 px-4 py-2 text-white font-semibold hover:bg-blue-500"
                      >
                        <Download className="h-4 w-4" />
                        Text
                      </button>
              </div>
            </div>

                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <Stat label="Trades" value={String(selected.results?.total_trades ?? selected.trades?.length ?? 0)} />
                    <Stat label="Win Rate" value={selected.results?.results?.win_rate != null ? `${Number(selected.results.results.win_rate).toFixed(1)}%` : '—'} />
                    <Stat
                      label="Total P&L"
                      value={
                        selected.results?.results?.total_profit != null
                          ? `${Number(selected.results.results.total_profit) >= 0 ? '+$' : '-$'}${Math.abs(Number(selected.results.results.total_profit)).toFixed(2)}`
                          : '—'
                      }
                      valueClassName={
                        selected.results?.results?.total_profit != null
                          ? Number(selected.results.results.total_profit) >= 0
                            ? 'text-emerald-400'
                            : 'text-rose-400'
                          : undefined
                      }
                    />
                    <Stat label="Max Drawdown" value={selected.results?.results?.max_drawdown != null ? `$${Number(selected.results.results.max_drawdown).toFixed(2)}` : '—'} />
                  </div>

                  {Array.isArray(selected.trades) && selected.trades.length > 0 ? (
                    <>
                      <div className="rounded-2xl border border-white/10 bg-slate-950/30 p-4">
                        <div className="mb-3 flex items-center gap-2 text-slate-200">
                          <BarChart3 className="h-4 w-4 text-blue-300" />
                          Trade Profit (Cumulative)
                        </div>
                        <BacktestChart
                          trades={selected.trades.map((t: any, i: number) => ({
                            id: t?.id || `${selected.id}-${i}`,
                            date: t?.date || new Date().toISOString(),
                            outcome: (t?.outcome || 'loss').toString().toLowerCase() === 'win' ? 'win' : 'loss',
                            profit: typeof t?.profit === 'number' ? t.profit : Number(t?.profit || 0),
                            label: t?.game || t?.selection || t?.reason || `Trade ${i + 1}`,
                            context: t,
                          }))}
                          market={(selected.market || 'sports').toString().toUpperCase()}
                          chartType="profit"
                          height={360}
                        />
                      </div>

                      {/* Bet Log Table */}
                      <div className="rounded-2xl border border-white/10 bg-slate-950/30 p-4">
                        <div className="mb-4 flex items-center justify-between">
                          <div className="flex items-center gap-2 text-slate-200">
                            <svg className="h-4 w-4 text-emerald-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
                            </svg>
                            Bet Log ({selected.trades.length} trades)
                          </div>
                        </div>
                        <div className="overflow-x-auto max-h-96 overflow-y-auto">
                          <table className="w-full text-sm">
                            <thead className="sticky top-0 bg-slate-900/95 text-slate-400 text-xs uppercase">
                              <tr>
                                <th className="px-3 py-2 text-left">#</th>
                                <th className="px-3 py-2 text-left">Date</th>
                                <th className="px-3 py-2 text-left">Game/Selection</th>
                                <th className="px-3 py-2 text-left">Bet Type</th>
                                <th className="px-3 py-2 text-center">Odds</th>
                                <th className="px-3 py-2 text-center">Outcome</th>
                                <th className="px-3 py-2 text-right">Profit</th>
                              </tr>
                            </thead>
                            <tbody className="divide-y divide-white/5">
                              {selected.trades.map((trade: any, idx: number) => {
                                const isWin = (trade?.outcome || '').toLowerCase() === 'win';
                                const profit = typeof trade?.profit === 'number' ? trade.profit : Number(trade?.profit || 0);
                                return (
                                  <tr key={trade?.id || idx} className="hover:bg-white/5 transition">
                                    <td className="px-3 py-2 text-slate-500">{idx + 1}</td>
                                    <td className="px-3 py-2 text-slate-300 whitespace-nowrap">
                                      {trade?.date ? new Date(trade.date).toLocaleDateString() : '—'}
                                    </td>
                                    <td className="px-3 py-2 text-white max-w-[200px] truncate">
                                      {trade?.game || trade?.selection || trade?.reason || `Trade ${idx + 1}`}
                                    </td>
                                    <td className="px-3 py-2 text-slate-400">
                                      {trade?.bet_type || trade?.betType || 'Moneyline'}
                                    </td>
                                    <td className="px-3 py-2 text-center text-slate-300">
                                      {trade?.odds ? (typeof trade.odds === 'number' ? trade.odds.toFixed(2) : trade.odds) : '—'}
                                    </td>
                                    <td className="px-3 py-2 text-center">
                                      <span className={`inline-flex px-2 py-0.5 rounded-full text-xs font-medium ${
                                        isWin 
                                          ? 'bg-emerald-500/20 text-emerald-300' 
                                          : 'bg-red-500/20 text-red-300'
                                      }`}>
                                        {isWin ? 'WIN' : 'LOSS'}
                                      </span>
                                    </td>
                                    <td className={`px-3 py-2 text-right font-mono font-medium ${
                                      profit >= 0 ? 'text-emerald-400' : 'text-red-400'
                                    }`}>
                                      {profit >= 0 ? '+' : ''}{profit.toFixed(2)}
                                    </td>
                                  </tr>
                                );
                              })}
                            </tbody>
                          </table>
                        </div>
                      </div>
                    </>
                  ) : (
                    <div className="text-slate-400 text-sm">No trades found for this backtest.</div>
                  )}
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </ProtectedRoute>
  );
}

function Stat({
  label,
  value,
  valueClassName,
}: {
  label: string;
  value: string;
  valueClassName?: string;
}) {
  return (
    <div className="rounded-xl border border-white/10 bg-slate-950/30 p-4">
      <div className="text-xs uppercase tracking-widest text-slate-400">{label}</div>
      <div className={`mt-1 text-2xl font-bold ${valueClassName ?? 'text-white'}`}>{value}</div>
    </div>
  );
}
