/**
 * Stripe Webhook Handler
 * POST /api/webhooks/stripe
 * Handles Stripe webhook events for subscription and payment updates
 */

import { NextRequest } from 'next/server';
import { StripeService } from '@/lib/stripe';
import { logger, ErrorHandler } from '@/lib/error-handler';
import { headers } from 'next/headers';

const WEBHOOK_SECRET = process.env.STRIPE_WEBHOOK_SECRET;

export async function POST(request: NextRequest) {
  const requestId = `wh_${Date.now()}_${Math.random().toString(36).substring(2, 8)}`;

  try {
    if (!WEBHOOK_SECRET) {
      logger.error('Stripe webhook secret not configured');
      return new Response('Webhook secret not configured', { status: 500 });
    }

    // Get the request body and signature
    const body = await request.text();
    const headersList = await headers();
    const sig = headersList.get('stripe-signature');

    if (!sig) {
      logger.warn('Missing Stripe signature', { requestId });
      return new Response('Missing signature', { status: 400 });
    }

    // Construct the event
    const event = await StripeService.constructEvent(body, sig, WEBHOOK_SECRET);

    logger.info('Received Stripe webhook', {
      type: event.type,
      id: event.id,
      requestId,
    });

    // Handle the event
    await StripeService.handleWebhook(event);

    // Return success response
    return new Response(JSON.stringify({ received: true }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' },
    });

  } catch (error) {
    logger.error('Stripe webhook error', {
      error: (error as Error).message,
      requestId,
    });

    await ErrorHandler.handleError(error as Error, {
      requestId,
      url: request.url,
      method: request.method,
    });

    // Return error response
    return new Response(
      JSON.stringify({ error: 'Webhook processing failed' }),
      {
        status: 400,
        headers: { 'Content-Type': 'application/json' },
      }
    );
  }
}

// Handle preflight requests for CORS
export async function OPTIONS() {
  return new Response(null, {
    status: 200,
    headers: {
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Methods': 'POST, OPTIONS',
      'Access-Control-Allow-Headers': 'Content-Type, stripe-signature',
    },
  });
}
