/**
 * Visualization Generation API Endpoint
 * POST /api/visualization/generate - Generate charts for backtest results
 */

import { NextRequest } from 'next/server';
import { authenticateAndAuthorize } from '@/lib/auth';
import { SubscriptionTier } from '@/lib/types';
import { handleApiError } from '@/lib/error-handler';
import { DatabaseService } from '@/lib/database';
import { spawn } from 'child_process';
import path from 'path';

export async function POST(request: NextRequest) {
  try {
    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE
    ]);
    if (authResult.status !== 200) {
      return authResult.response;
    }

    const { chartType, trades, market, chartFormat = 'plotly' } = await request.json();

    if (!chartType || !trades) {
      return new Response(JSON.stringify({
        error: 'chartType and trades are required'
      }), {
        status: 400,
        headers: { 'Content-Type': 'application/json' }
      });
    }

    console.log(`📊 Generating ${chartFormat} chart: ${chartType} for ${market || 'general'} market`);

    // Call Python visualization service
    const visualizationData = await callVisualizationService(chartType, {
      trades,
      market: market || 'general',
      chart_type: chartFormat
    });

    if (!visualizationData) {
      return new Response(JSON.stringify({
        error: 'Failed to generate visualization'
      }), {
        status: 500,
        headers: { 'Content-Type': 'application/json' }
      });
    }

    // Record analytics
    if (authResult.data?.user) {
      await DatabaseService.recordUserEvent(authResult.data.user.id, 'VISUALIZATION_GENERATED', {
        chart_type: chartType,
        market: market || 'general',
        format: chartFormat,
        trades_count: trades.length
      });
    }

    return new Response(JSON.stringify({
      success: true,
      data: visualizationData,
      meta: {
        chartType,
        market: market || 'general',
        format: chartFormat,
        tradesCount: trades.length,
        generatedAt: new Date().toISOString()
      }
    }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' }
    });

  } catch (error) {
    console.error('Visualization API error:', error);
    return handleApiError(error as Error, request);
  }
}

async function callVisualizationService(chartType: string, params: any = {}): Promise<any> {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(process.cwd(), 'visualization_service.py');

    const args = [
      scriptPath,
      chartType,
      JSON.stringify(params)
    ];

    const pythonProcess = spawn('python3', args, {
      cwd: process.cwd(),
      env: {
        ...process.env,
        PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages')
      }
    });

    let stdout = '';
    let stderr = '';

    pythonProcess.stdout.on('data', (data) => {
      stdout += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      stderr += data.toString();
    });

    pythonProcess.on('close', (code) => {
      if (code !== 0) {
        console.error(`Python visualization process failed with code ${code}:`, stderr);
        reject(new Error(`Visualization service failed: ${stderr}`));
        return;
      }

      try {
        const result = JSON.parse(stdout.trim());
        resolve(result);
      } catch (parseError) {
        console.error('Failed to parse visualization service response:', stdout);
        reject(new Error('Invalid response from visualization service'));
      }
    });

    pythonProcess.on('error', (error) => {
      console.error('Failed to start visualization service:', error);
      reject(error);
    });
  });
}
