import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';

function safeParseJson<T>(value: any, fallback: T): T {
  if (!value) return fallback;
  if (typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;
    const url = new URL(request.url);
    const limit = Math.min(100, Math.max(1, Number(url.searchParams.get('limit') || 50)));

    const strategies = await DatabaseService.findStrategiesByUser(user.id, limit);

    const items = strategies.map((s) => ({
      id: s.id,
      name: s.name,
      description: s.description,
      domain: s.domain,
      isPublic: s.isPublic,
      tags: safeParseJson<string[]>(s.tags, []),
      rules: safeParseJson<any>(s.rules, null),
      totalTrades: s.totalTrades,
      winRate: s.winRate,
      totalPnL: s.totalPnL,
      likes: s.likes,
      forks: s.forks,
      views: s.views,
      createdAt: s.createdAt,
      updatedAt: s.updatedAt,
    }));

    return NextResponse.json({ strategies: items, total: items.length });
  } catch (error) {
    console.error('[api/user/strategies] GET failed', error);
    return NextResponse.json({ error: 'Failed to load strategies' }, { status: 500 });
  }
}


