/**
 * Data Export API Route (GDPR Compliance)
 * GET /api/user/export-data
 * Exports all user data in JSON format
 */

import { NextRequest, NextResponse } from 'next/server';
import { authenticateAndAuthorize, DatabaseService } from '@/lib/auth';
import { SubscriptionTier } from '@/lib/types';
import { logger } from '@/lib/error-handler';
import { prisma } from '@/lib/database';

function safeParseJson<T>(value: any, fallback: T): T {
  if (!value || typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

export async function GET(request: NextRequest) {
  try {
    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);

    if (authResult.status !== 200) {
      return authResult.response;
    }

    const user = authResult.data?.user;
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    // Get full user data from database
    const dbUser = await DatabaseService.findUserById(user.id);
    
    if (!dbUser) {
      return NextResponse.json(
        { error: 'User not found' },
        { status: 404 }
      );
    }

    // Pull full related data directly for export
    const [strategies, conversations, events, backtests, feedback] = await Promise.all([
      prisma.strategy.findMany({
        where: { userId: user.id },
        orderBy: { createdAt: 'desc' },
      }),
      prisma.conversation.findMany({
        where: { userId: user.id },
        include: { messages: true },
        orderBy: { createdAt: 'desc' },
      }),
      prisma.userAnalytics.findMany({
        where: { userId: user.id },
        orderBy: { createdAt: 'desc' },
      }),
      prisma.backtestResult.findMany({
        where: { userId: user.id },
        orderBy: { createdAt: 'desc' },
      }),
      prisma.feedback.findMany({
        where: { userId: user.id },
        orderBy: { createdAt: 'desc' },
      }),
    ]);

    // Compile all user data
    const exportData = {
      exportDate: new Date().toISOString(),
      exportType: 'GDPR_DATA_EXPORT',
      user: {
        id: dbUser.id,
        email: dbUser.email,
        firstName: dbUser.firstName,
        lastName: dbUser.lastName,
        createdAt: dbUser.createdAt,
        lastLoginAt: dbUser.lastLoginAt,
        emailVerified: dbUser.emailVerified,
        emailVerifiedAt: dbUser.emailVerifiedAt,
        subscriptionTier: dbUser.subscriptionTier,
        role: dbUser.role,
      },
      subscription: {
        tier: dbUser.subscriptionTier,
        subscriptionId: dbUser.subscriptionId,
        stripeCustomerId: dbUser.stripeCustomerId,
        trialEndsAt: dbUser.trialEndsAt,
        subscriptionEndsAt: dbUser.subscriptionEndsAt,
        canceledAt: dbUser.canceledAt,
      },
      strategies: strategies.map((s: any) => ({
        id: s.id,
        name: s.name,
        description: s.description,
        domain: s.domain,
        createdAt: s.createdAt,
        updatedAt: s.updatedAt,
        isPublic: s.isPublic,
        tags: safeParseJson<string[]>(s.tags, []),
        rules: safeParseJson<any>(s.rules, null),
        metrics: {
          totalTrades: s.totalTrades,
          winRate: s.winRate,
          totalPnL: s.totalPnL,
          maxDrawdown: s.maxDrawdown,
          sharpeRatio: s.sharpeRatio,
          likes: s.likes,
          forks: s.forks,
          views: s.views,
        },
      })),
      conversations: conversations.map((c: any) => ({
        id: c.id,
        domain: c.domain,
        createdAt: c.createdAt,
        updatedAt: c.updatedAt,
        messages: c.messages,
      })),
      backtests: backtests.map((b: any) => ({
        id: b.id,
        jobId: b.jobId,
        market: b.market,
        strategyName: b.strategyName,
        status: b.status,
        createdAt: b.createdAt,
        startedAt: b.startedAt,
        completedAt: b.completedAt,
        executionTime: b.executionTime,
        errorMessage: b.errorMessage,
        parameters: safeParseJson<any>(b.parameters, {}),
        results: safeParseJson<any>(b.results, {}),
        trades: safeParseJson<any[]>(b.trades, []),
      })),
      feedback: feedback.map((f: any) => ({
        id: f.id,
        feedbackType: f.feedbackType,
        subject: f.subject,
        message: f.message,
        priority: f.priority,
        status: f.status,
        response: f.response,
        respondedAt: f.respondedAt,
        respondedBy: f.respondedBy,
        createdAt: f.createdAt,
        updatedAt: f.updatedAt,
      })),
      activityLog: events.map((e: any) => ({
        eventType: e.eventType,
        timestamp: e.createdAt,
        eventData: safeParseJson<any>(e.eventData, {}),
        metadata: safeParseJson<any>(e.metadata, {}),
      })),
    };

    // Record the export event
    await DatabaseService.recordUserEvent(user.id, 'DATA_EXPORT', {
      exportFormat: 'json',
      dataCategories: ['user', 'strategies', 'conversations', 'activityLog'],
    });

    logger.info('User data exported (GDPR)', { userId: user.id });

    // Return as downloadable JSON file
    const jsonString = JSON.stringify(exportData, null, 2);
    
    return new NextResponse(jsonString, {
      status: 200,
      headers: {
        'Content-Type': 'application/json',
        'Content-Disposition': `attachment; filename="eventheodds-data-export-${new Date().toISOString().split('T')[0]}.json"`,
      },
    });

  } catch (error) {
    logger.error('Data export error', { error });

    return NextResponse.json(
      { error: 'Failed to export data. Please try again or contact support.' },
      { status: 500 }
    );
  }
}
