/**
 * Account Deletion API Route (GDPR Compliance)
 * DELETE /api/user/delete-account
 * Permanently deletes user account and all associated data
 */

import { NextRequest, NextResponse } from 'next/server';
import { authenticateAndAuthorize, DatabaseService, AuthService } from '@/lib/auth';
import { StripeService } from '@/lib/stripe';
import { SubscriptionTier } from '@/lib/types';
import { logger } from '@/lib/error-handler';
import { prisma } from '@/lib/database';

export async function DELETE(request: NextRequest) {
  try {
    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);

    if (authResult.status !== 200) {
      return authResult.response;
    }

    const user = authResult.data?.user;
    if (!user) {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }
    const body = await request.json().catch(() => ({}));
    const { confirmation, password, reason } = body;

    // Require explicit confirmation
    if (confirmation !== 'DELETE_MY_ACCOUNT') {
      return NextResponse.json(
        { error: 'Please type DELETE_MY_ACCOUNT to confirm deletion' },
        { status: 400 }
      );
    }

    // Get user from database
    const dbUser = await DatabaseService.findUserById(user.id);
    
    if (!dbUser) {
      return NextResponse.json(
        { error: 'User not found' },
        { status: 404 }
      );
    }

    // Verify password if provided (extra security)
    if (password) {
      const isValidPassword = await AuthService.verifyPassword(password, dbUser.passwordHash);
      if (!isValidPassword) {
        return NextResponse.json(
          { error: 'Invalid password' },
          { status: 401 }
        );
      }
    }

    // Cancel any active subscriptions first
    if (dbUser.subscriptionId) {
      try {
        await StripeService.cancelSubscription(dbUser.subscriptionId, user.id);
        logger.info('Cancelled subscription before account deletion', {
          userId: user.id,
          subscriptionId: dbUser.subscriptionId,
        });
      } catch (stripeError) {
        logger.warn('Failed to cancel subscription during account deletion', {
          userId: user.id,
          error: stripeError,
        });
        // Continue with deletion even if Stripe cancellation fails
      }
    }

    // Record deletion reason before deleting
    await DatabaseService.recordUserEvent(user.id, 'ACCOUNT_DELETION_INITIATED', {
      reason: reason || 'not_provided',
      email: dbUser.email,
    });

    // Delete all user data
    // Order matters: delete dependent records first
    
    // Backtests are not related via FK (userId is optional), so delete explicitly
    await prisma.backtestResult.deleteMany({
      where: { userId: user.id },
    });
    
    // 1. Delete user's conversations
    await DatabaseService.deleteUserConversations(user.id);
    
    // 2. Delete user's strategies
    await DatabaseService.deleteUserStrategies(user.id);
    
    // 3. Delete user's events/activity log
    await DatabaseService.deleteUserEvents(user.id);
    
    // 4. Finally, delete the user record
    await DatabaseService.deleteUser(user.id);

    logger.info('User account deleted (GDPR)', {
      userId: user.id,
      email: dbUser.email,
      reason,
    });

    // Create response that clears the auth cookie
    const response = NextResponse.json({
      success: true,
      message: 'Your account and all associated data have been permanently deleted.',
    });

    // Clear the auth cookie
    response.cookies.set('auth-token', '', {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'lax',
      maxAge: 0,
      path: '/',
    });

    return response;

  } catch (error) {
    logger.error('Account deletion error', { error });

    return NextResponse.json(
      { error: 'Failed to delete account. Please try again or contact support.' },
      { status: 500 }
    );
  }
}

// Also support POST for compatibility
export async function POST(request: NextRequest) {
  return DELETE(request);
}
