import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';

function safeJsonParse<T>(value: string | null | undefined, fallback: T): T {
  if (!value) return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

function toNumber(value: any, fallback = 0): number {
  const n = typeof value === 'number' ? value : Number(value);
  return Number.isFinite(n) ? n : fallback;
}

function normalizeOutcome(raw: any): 'win' | 'loss' | 'push' | 'void' {
  const o = (raw ?? '').toString().toLowerCase().trim();
  if (o === 'win' || o === 'won' || o === 'w') return 'win';
  if (o === 'push' || o === 'tie' || o === 'draw') return 'push';
  if (o === 'void' || o === 'cancelled' || o === 'canceled') return 'void';
  return 'loss';
}

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;
    const url = new URL(request.url);
    const limit = Math.min(200, Math.max(1, Number(url.searchParams.get('limit') || 25)));

    // Pull enough backtests to satisfy the requested trade limit.
    const backtests = await DatabaseService.findBacktestsByUser(user.id, 50);

    const items: any[] = [];
    for (const bt of backtests) {
      const trades = safeJsonParse<any[]>(bt.trades, []);
      if (!Array.isArray(trades) || trades.length === 0) continue;

      for (let idx = 0; idx < trades.length; idx++) {
        const t = trades[idx] ?? {};

        const date =
          t?.date ||
          t?.timestamp ||
          t?.game_date ||
          t?.gameDate ||
          null;
        const label =
          t?.game ||
          t?.selection ||
          t?.label ||
          t?.reason ||
          `Trade ${idx + 1}`;

        items.push({
          id: t?.id || `${bt.id}-${idx}`,
          backtestId: bt.id,
          market: bt.market,
          strategyName: bt.strategyName,
          backtestCreatedAt: bt.createdAt,
          date,
          label,
          betType: t?.bet_type || t?.betType || t?.type || null,
          odds: t?.odds ?? t?.price ?? null,
          outcome: normalizeOutcome(t?.outcome || t?.result),
          profit: toNumber(t?.profit, 0),
        });

        if (items.length >= limit * 3) break; // safety cap before sorting
      }
    }

    // Sort: use trade date when available, else backtest createdAt
    items.sort((a, b) => {
      const aTime = a.date ? Date.parse(a.date) : new Date(a.backtestCreatedAt).getTime();
      const bTime = b.date ? Date.parse(b.date) : new Date(b.backtestCreatedAt).getTime();
      return (Number.isFinite(bTime) ? bTime : 0) - (Number.isFinite(aTime) ? aTime : 0);
    });

    const sliced = items.slice(0, limit);

    return NextResponse.json({
      items: sliced,
      total: sliced.length,
    });
  } catch (error) {
    console.error('[api/user/bet-log] GET failed', error);
    return NextResponse.json({ error: 'Failed to load bet log' }, { status: 500 });
  }
}


