import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware } from '@/lib/auth';
import { prisma } from '@/lib/database';

export async function GET(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;

    // Get user's strategies count
    const strategiesCount = await prisma.strategy.count({
      where: { userId: user.id }
    });

    // Get user's backtest results
    const backtestResults = await prisma.backtestResult.findMany({
      where: { userId: user.id },
      orderBy: { createdAt: 'desc' },
      take: 100 // Get last 100 for calculating stats
    });

    // Get conversations count (AI generations)
    const conversationsCount = await prisma.conversation.count({
      where: { userId: user.id }
    });

    // Calculate aggregate stats from backtests
    let totalTrades = 0;
    let totalWins = 0;
    let totalProfit = 0;

    for (const backtest of backtestResults) {
      try {
        const results = JSON.parse(backtest.results || '{}');
        const performance = JSON.parse(backtest.performance || '{}');
        
        totalTrades += results.total_trades || performance.total_trades || 0;
        totalWins += results.wins || performance.wins || Math.round((results.win_rate || 0) * (results.total_trades || 0) / 100);
        totalProfit += results.total_profit || performance.total_profit || 0;
      } catch (e) {
        // Skip malformed data
      }
    }

    const avgWinRate = totalTrades > 0 ? Math.round((totalWins / totalTrades) * 100) : 0;

    // Get recent activity (last 10 events)
    const recentBacktests = await prisma.backtestResult.findMany({
      where: { userId: user.id },
      orderBy: { createdAt: 'desc' },
      take: 10,
      select: {
        id: true,
        strategyName: true,
        market: true,
        results: true,
        performance: true,
        createdAt: true,
        status: true
      }
    });

    const recentStrategies = await prisma.strategy.findMany({
      where: { userId: user.id },
      orderBy: { createdAt: 'desc' },
      take: 5,
      select: {
        id: true,
        name: true,
        domain: true,
        winRate: true,
        totalTrades: true,
        createdAt: true
      }
    });

    // Build recent activity feed
    const recentActivity = [
      ...recentBacktests.map(bt => {
        let winRate = 0;
        try {
          const results = JSON.parse(bt.results || '{}');
          winRate = results.win_rate || results.results?.win_rate || 0;
        } catch (e) {}

        return {
          id: bt.id,
          type: 'backtest',
          title: `Backtest: ${bt.strategyName || 'Unnamed Strategy'}`,
          description: `${bt.market?.toUpperCase() || 'Sports'} - ${winRate.toFixed(1)}% win rate`,
          sport: bt.market,
          winRate: winRate,
          createdAt: bt.createdAt,
          data: {
            strategyName: bt.strategyName,
            market: bt.market,
            results: bt.results,
            performance: bt.performance
          }
        };
      }),
      ...recentStrategies.map(st => ({
        id: st.id,
        type: 'strategy',
        title: `Strategy saved: ${st.name}`,
        description: `${st.domain?.toUpperCase() || 'Sports'} - ${st.totalTrades || 0} trades`,
        sport: st.domain,
        winRate: st.winRate || 0,
        createdAt: st.createdAt,
        data: {
          name: st.name,
          domain: st.domain,
          winRate: st.winRate,
          totalTrades: st.totalTrades
        }
      }))
    ].sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())
    .slice(0, 10);

    return NextResponse.json({
      stats: {
        strategiesCreated: strategiesCount,
        aiGenerations: conversationsCount,
        backtestsRun: backtestResults.length,
        totalTrades: totalTrades,
        totalWins: totalWins,
        avgWinRate: avgWinRate,
        totalProfit: totalProfit
      },
      recentActivity: recentActivity,
      backtestHistory: recentBacktests.map(bt => {
        let parsedResults = {};
        let parsedPerformance = {};
        try {
          parsedResults = JSON.parse(bt.results || '{}');
          parsedPerformance = JSON.parse(bt.performance || '{}');
        } catch (e) {}

        return {
          id: bt.id,
          strategyName: bt.strategyName,
          market: bt.market,
          status: bt.status,
          createdAt: bt.createdAt,
          results: parsedResults,
          performance: parsedPerformance
        };
      })
    });
  } catch (error) {
    console.error('[api/user/analytics] GET failed', error);
    return NextResponse.json({ error: 'Failed to load analytics' }, { status: 500 });
  }
}
