/**
 * Live Chat Support API
 * POST /api/support/chat
 * Sends chat messages to support team via email
 */

import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import nodemailer from 'nodemailer';
import { AuthMiddleware, createSecureResponse, DatabaseService } from '@/lib/auth';

const chatSchema = z.object({
  message: z.string().min(1, 'Message is required').max(2000, 'Message is too long'),
  sessionId: z.string().optional(),
  userEmail: z.string().email().optional(),
  userName: z.string().optional(),
});

export async function POST(request: NextRequest) {
  try {
    // Try to get authenticated user, but allow anonymous chats too
    let user: any = null;
    try {
      const authResponse = await AuthMiddleware.requireAuth(request);
      if (!authResponse) {
        user = (request as any).user;
      }
    } catch {
      // Anonymous chat allowed
    }

    const body = await request.json();
    const { message, sessionId, userEmail, userName } = chatSchema.parse(body);

    // Generate or use existing session ID
    const chatSessionId = sessionId || `chat_${Date.now()}_${Math.random().toString(36).substring(7)}`;
    
    // Get user info
    const senderEmail = user?.email || userEmail || 'Anonymous';
    const senderName = user?.firstName || userName || 'Website Visitor';
    const userId = user?.id || 'system'; // Use system user for anonymous chats

    // Store chat message in database
    try {
      const { PrismaClient } = require('@prisma/client');
      const prisma = new PrismaClient();
      await prisma.feedback.create({
        data: {
          userId: userId,
          feedbackType: 'question',
          subject: `Live Chat: ${message.substring(0, 50)}${message.length > 50 ? '...' : ''}`,
          message: message,
          priority: 'HIGH',
          status: 'OPEN',
          metadata: JSON.stringify({
            type: 'live_chat',
            sessionId: chatSessionId,
            userName: senderName,
            userEmail: senderEmail,
            userAgent: request.headers.get('user-agent') || undefined,
            timestamp: new Date().toISOString(),
          }),
        },
      });
      await prisma.$disconnect();
    } catch (dbError) {
      console.warn('Failed to store chat message in DB:', dbError);
    }

    // Send email notification to support team
    try {
      const transporter = nodemailer.createTransport({
        host: process.env.SMTP_HOST || 'localhost',
        port: parseInt(process.env.SMTP_PORT || '25'),
        secure: false,
        tls: { rejectUnauthorized: false },
      });

      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; background: #1e293b; padding: 20px; border-radius: 12px;">
          <div style="background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%); padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <h1 style="color: white; margin: 0; font-size: 24px;">💬 New Live Chat Message</h1>
            <p style="color: rgba(255,255,255,0.8); margin: 5px 0 0 0;">Someone needs help on EventheOdds.ai</p>
          </div>
          
          <div style="background: #334155; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <table style="width: 100%; border-collapse: collapse;">
              <tr>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: #94a3b8; width: 100px;"><strong>From:</strong></td>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: white;">${senderName}</td>
              </tr>
              <tr>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: #94a3b8;"><strong>Email:</strong></td>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: white;">
                  <a href="mailto:${senderEmail}" style="color: #60a5fa;">${senderEmail}</a>
                </td>
              </tr>
              <tr>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: #94a3b8;"><strong>Session:</strong></td>
                <td style="padding: 10px 0; border-bottom: 1px solid #475569; color: #94a3b8; font-size: 12px;">${chatSessionId}</td>
              </tr>
              <tr>
                <td style="padding: 10px 0; color: #94a3b8;"><strong>Time:</strong></td>
                <td style="padding: 10px 0; color: white;">${new Date().toLocaleString()}</td>
              </tr>
            </table>
          </div>

          <div style="background: #334155; padding: 20px; border-radius: 8px;">
            <h3 style="color: #60a5fa; margin: 0 0 15px 0; font-size: 14px; text-transform: uppercase;">Message:</h3>
            <div style="background: #1e293b; padding: 15px; border-radius: 6px; border-left: 4px solid #3b82f6;">
              <p style="color: white; margin: 0; white-space: pre-wrap; line-height: 1.6;">${message}</p>
            </div>
          </div>

          <div style="margin-top: 20px; text-align: center;">
            <a href="${process.env.NEXT_PUBLIC_BASE_URL || 'https://eventheodds.ai'}/admin/chat" 
               style="display: inline-block; background: #10b981; color: white; padding: 12px 24px; border-radius: 6px; text-decoration: none; font-weight: bold; margin-right: 10px;">
              💬 Open Live Chat Panel
            </a>
            <a href="mailto:${senderEmail}?subject=Re: Your EventheOdds Support Request" 
               style="display: inline-block; background: #3b82f6; color: white; padding: 12px 24px; border-radius: 6px; text-decoration: none; font-weight: bold;">
              📧 Reply via Email
            </a>
          </div>

          <p style="color: #64748b; font-size: 12px; text-align: center; margin-top: 20px;">
            This message was sent from the Live Chat on eventheodds.ai
          </p>
        </div>
      `;

      await transporter.sendMail({
        from: `"EventheOdds Live Chat" <${process.env.EMAIL_FROM || 'noreply@eventheodds.ai'}>`,
        to: process.env.SUPPORT_EMAIL || 'admin@eventheodds.ai',
        replyTo: senderEmail !== 'Anonymous' ? senderEmail : undefined,
        subject: `💬 Live Chat: ${senderName} - ${message.substring(0, 40)}${message.length > 40 ? '...' : ''}`,
        html: emailHtml,
        text: `New Live Chat Message\n\nFrom: ${senderName}\nEmail: ${senderEmail}\nTime: ${new Date().toLocaleString()}\n\nMessage:\n${message}\n\nSession ID: ${chatSessionId}`,
      });

      console.log('Live chat email sent to support team');
    } catch (emailError) {
      console.error('Failed to send live chat email:', emailError);
    }

    return NextResponse.json({
      success: true,
      sessionId: chatSessionId,
      message: "Your message has been sent to our support team. We'll respond as soon as possible!",
    });

  } catch (error) {
    console.error('Live chat error:', error);

    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Invalid message', details: error.issues },
        { status: 400 }
      );
    }

    return NextResponse.json(
      { error: 'Failed to send message. Please try again.' },
      { status: 500 }
    );
  }
}
