import { NextRequest } from 'next/server';
import { z } from 'zod';

import { getSnapshotOverview, runPromptSimulation } from '@/lib/strategyEngine';
import { Domain } from '@/lib/types';
import { AuthMiddleware, SecurityMiddleware, createSecureResponse, DatabaseService, checkFeatureAccess } from '@/lib/auth';
import { ErrorHandler, ErrorFactory, logger, performanceMonitor } from '@/lib/error-handler';
import { PerformanceMiddleware, APICacheStrategies, CacheInvalidation } from '@/lib/performance-middleware';

const createStrategySchema = z.object({
  prompt: z.string().min(8, 'Prompt is too short').max(1000),
  domain: z.enum(['sports', 'crypto', 'stocks', 'forex']),
  saveStrategy: z.boolean().optional().default(false),
  saveBacktest: z.boolean().optional().default(true),
  strategyName: z.string().max(100).optional(),
  isPublic: z.boolean().optional().default(false),
});

export async function POST(request: NextRequest) {
  return PerformanceMiddleware.trackPerformance(request, async () => {
    try {
      // Optional authentication (allow anonymous for basic simulations)
      await AuthMiddleware.optionalAuth(request);
      const user = (request as any).user;

      // Rate limiting
      const rateLimitResponse = SecurityMiddleware.rateLimit(request, 20, 60 * 60 * 1000); // 20 simulations per hour
      if (rateLimitResponse) return rateLimitResponse;

      // Parse and validate request
      const json = await request.json();
      const validation = SecurityMiddleware.validateInput(json, {
        required: ['prompt', 'domain'],
        stringFields: ['prompt', 'domain', 'strategyName'],
        maxLength: { prompt: 1000, strategyName: 100 }
      });

      if (!validation.valid) {
        return createSecureResponse(
          { error: 'Validation failed', details: validation.errors },
          400,
          user
        );
      }

      const { prompt, domain, saveStrategy, saveBacktest, strategyName, isPublic } = createStrategySchema.parse(json);

      // Sanitize input
      const sanitizedPrompt = SecurityMiddleware.sanitizeInput(prompt);

      // Check subscription limits for saved strategies
      if (saveStrategy && user) {
        const userProfile = await DatabaseService.findUserById(user.id);
        if (!userProfile) {
          return createSecureResponse({ error: 'User not found' }, 404, user);
        }

        // FREE tier users have full Pro access (they're on a trial subscription)
        const effectiveTier = userProfile.subscriptionTier === 'FREE' ? 'PRO' : userProfile.subscriptionTier;
        const isTrialActive = userProfile.subscriptionTier === 'FREE' || userProfile.subscriptionTier === 'TRIAL';
        const userStats = await DatabaseService.getUserStats(user.id);
        const maxStrategiesFeature = checkFeatureAccess(effectiveTier, 'maxStrategies');
        const maxStrategies = typeof maxStrategiesFeature === 'number' ? maxStrategiesFeature : (maxStrategiesFeature ? -1 : 5);

        if (maxStrategies !== -1 && userStats.totalStrategies >= maxStrategies) {
          const tierName = isTrialActive ? 'trial' : userProfile.subscriptionTier.toLowerCase();
          return createSecureResponse(
            {
              error: 'Strategy limit reached',
              message: `Your ${tierName} plan allows ${maxStrategies} strategies.${isTrialActive ? 'Upgrade to Pro to continue saving strategies.' : 'Upgrade to save more.'} `,
              currentUsage: userStats.totalStrategies,
              limit: maxStrategies,
              isTrialActive,
              trialEndsAt: userProfile.trialEndsAt
            },
            403,
            user
          );
        }
      }

      // Run simulation
      const simulationResult = await runPromptSimulation(sanitizedPrompt, domain as Domain);

      let savedStrategy = null;
      if (saveStrategy && user) {
        // Save strategy to database
        savedStrategy = await DatabaseService.createStrategy({
          name: strategyName || sanitizedPrompt.substring(0, 50) + '...',
          description: sanitizedPrompt,
          domain: domain as Domain,
          rules: (simulationResult as any).rules || {},
          isPublic,
          userId: user.id,
        });

        // Invalidate user cache
        await CacheInvalidation.invalidateUserCache(user.id);

        // Record analytics
        await DatabaseService.recordUserEvent(user.id, 'STRATEGY_CREATED', {
          strategyId: savedStrategy.id,
          domain,
          isPublic,
          simulationSuccess: !!(simulationResult as any).results
        });

        await DatabaseService.recordStrategyEvent(savedStrategy.id, 'STRATEGY_CREATED', {
          userId: user.id,
          domain,
          performance: (simulationResult as any).results?.length || 0
        });
      }

      // Persist backtest + bet log for authenticated users so they can access it later from Profile/Backtest History.
      // This is separate from saving the strategy to the library.
      let savedBacktest: { id: string; createdAt: Date } | null = null;
      if (saveBacktest && user) {
        try {
          const summary = simulationResult.summary;
          const market =
            domain === 'sports'
              ? ((simulationResult as any).sportFilter || 'sports').toString()
              : domain.toString();
          const inferredName =
            (strategyName || '').trim() ||
            (sanitizedPrompt.length > 60 ? sanitizedPrompt.slice(0, 60) + '…' : sanitizedPrompt);

          // Shape results similar to the Python backtest engine outputs so existing UI + analytics keep working.
          const winRatePct = Number(((summary.winRate || 0) * 100).toFixed(2));
          const totalProfit = Number((summary.netProfit || 0).toFixed(6));

          const bt = {
            market,
            strategy_name: inferredName,
            total_trades: summary.totalTrades ?? summary.trades?.length ?? 0,
            wins: summary.wins ?? 0,
            win_rate: winRatePct,
            total_profit: totalProfit,
            results: {
              total_trades: summary.totalTrades ?? summary.trades?.length ?? 0,
              wins: summary.wins ?? 0,
              win_rate: winRatePct,
              total_profit: totalProfit,
              avg_return: summary.avgReturn ?? null,
              roi_percent: summary.roiPercent ?? null,
              profit_factor: summary.profitFactor ?? null,
              sharpe_ratio: summary.sharpeRatio ?? null,
              max_drawdown: summary.maxDrawdown ?? null,
              max_drawdown_pct: summary.maxDrawdownPct ?? null,
              ats_record: summary.atsRecord ?? null,
            },
            all_trades: summary.trades || [],
            detailed_statistics: {
              notes: summary.notes || [],
              rule: summary.rule || null,
              datasetSize: (simulationResult as any).datasetSize ?? null,
              sportFilter: (simulationResult as any).sportFilter ?? null,
            },
          };

          const stored = await DatabaseService.createBacktestResult({
            userId: user.id,
            market,
            strategyName: inferredName,
            strategyCode: null,
            parameters: {
              prompt: sanitizedPrompt,
              domain,
              sportFilter: (simulationResult as any).sportFilter ?? null,
              rule: summary.rule ?? null,
            },
            results: bt,
            trades: summary.trades || [],
            performance: bt.results || {},
            riskMetrics: bt.detailed_statistics || {},
            chartData: null,
            status: 'COMPLETED',
            completedAt: new Date(),
          });

          savedBacktest = { id: stored.id, createdAt: stored.createdAt };

          await DatabaseService.recordUserEvent(user.id, 'BACKTEST_RUN', {
            backtestId: stored.id,
            market,
            strategyName: inferredName,
            totalTrades: bt.total_trades,
            winRate: bt.win_rate,
            totalProfit: bt.total_profit,
          });
        } catch (e) {
          console.warn('[api/strategies] Failed to persist backtest result', e);
        }
      }

      return createSecureResponse({
        ...simulationResult,
        savedStrategy: savedStrategy ? {
          id: savedStrategy.id,
          name: savedStrategy.name,
          createdAt: savedStrategy.createdAt,
        } : null,
        savedBacktest,
      }, 200, user);

    } catch (error) {
      console.error('[api/strategies] failed', error);

      if (error instanceof z.ZodError) {
        return createSecureResponse(
          { error: 'Validation failed', issues: error.issues },
          422
        );
      }

      // Record error analytics
      const user = (request as any).user;
      if (user) {
        await DatabaseService.recordUserEvent(user.id, 'STRATEGY_SIMULATION_ERROR', {
          error: (error as Error).message,
          domain: (await request.json().catch(() => ({}))).domain
        });
      }

      return createSecureResponse(
        { error: 'Unable to run simulation' },
        500
      );
    }
  });
}

export async function GET(request: NextRequest) {
  return PerformanceMiddleware.cacheAPIResponse(
    'strategies_overview',
    request,
    async () => {
      try {
        // Optional authentication
        await AuthMiddleware.optionalAuth(request);
        const user = (request as any).user;

        // Get dataset overview
        const overview = await getSnapshotOverview();

        // Add user-specific strategy counts if authenticated
        if (user) {
          const userStats = await DatabaseService.getUserStats(user.id);
          (overview as any).userStats = userStats;
        }

        return createSecureResponse(overview, 200, user);

      } catch (error) {
        console.error('[api/strategies] GET failed', error);
        return createSecureResponse(
          { error: 'Unable to fetch strategy overview' },
          500
        );
      }
    },
    {
      ttl: 10 * 60 * 1000, // 10 minutes
      tags: ['strategies', 'datasets'],
    }
  );
}

