import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';
import { prisma } from '@/lib/database';

const createFromTemplateSchema = z.object({
  templateId: z.string(),
  customizations: z.any().optional(), // Allow any object for customizations
});

function safeParseJson<T>(value: any, fallback: T): T {
  if (!value || typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

// GET /api/strategies/library/templates - Get strategy templates
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const domain = searchParams.get('domain') as 'sports' | 'crypto' | 'stocks' | 'forex' | undefined;
    const templates = await prisma.strategy.findMany({
      where: {
        isTemplate: true,
        ...(domain ? { domain } : {}),
      },
      orderBy: { createdAt: 'desc' },
      take: 100,
    });

    return NextResponse.json({
      templates: templates.map((t) => ({
        id: t.id,
        name: t.name,
        description: t.description,
        domain: t.domain,
        tags: safeParseJson<string[]>(t.tags, []),
        rules: safeParseJson<any>(t.rules, null),
        createdAt: t.createdAt,
        updatedAt: t.updatedAt,
      })),
      total: templates.length,
    });
  } catch (error) {
    console.error('[api/strategies/library/templates] GET error:', error);
    return NextResponse.json(
      { error: 'Failed to get templates' },
      { status: 500 },
    );
  }
}

// POST /api/strategies/library/templates - Create strategy from template
export async function POST(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;
    const user = (request as any).user;

    const json = await request.json();
    const { templateId, customizations } = createFromTemplateSchema.parse(json);

    const template = await prisma.strategy.findFirst({
      where: { id: templateId, isTemplate: true },
    });
    if (!template) {
      return NextResponse.json({ error: 'Template not found' }, { status: 404 });
    }

    const templateRules = safeParseJson<any>(template.rules, {});
    const mergedRules = customizations ? { ...templateRules, customizations } : templateRules;

    const created = await DatabaseService.createStrategy({
      userId: user.id,
      name: `${template.name} (Custom)`,
      description: template.description || undefined,
      domain: template.domain as any,
      rules: mergedRules,
      tags: safeParseJson<string[]>(template.tags, []),
      isPublic: false,
    });

    await DatabaseService.recordUserEvent(user.id, 'STRATEGY_CREATED', {
      strategyId: created.id,
      source: 'template',
      templateId,
    });

    return NextResponse.json({ success: true, strategy: created }, { status: 201 });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Invalid template data', issues: error.issues },
        { status: 422 },
      );
    }

    console.error('[api/strategies/library/templates] POST error:', error);
    return NextResponse.json(
      { error: 'Failed to create from template' },
      { status: 500 },
    );
  }
}
