import { NextResponse } from 'next/server';
import { prisma } from '@/lib/database';

function safeParseJsonArray(value: any): string[] {
  if (!value || typeof value !== 'string') return [];
  try {
    const parsed = JSON.parse(value);
    return Array.isArray(parsed) ? parsed.map(String) : [];
  } catch {
    return [];
  }
}

// GET /api/strategies/library/stats - Get library statistics
export async function GET() {
  try {
    const [totalStrategies, publicStrategies, recentStrategies] = await Promise.all([
      prisma.strategy.count(),
      prisma.strategy.count({ where: { isPublic: true } }),
      prisma.strategy.findMany({
        orderBy: { createdAt: 'desc' },
        take: 5,
        include: { user: { select: { id: true, firstName: true, lastName: true } } },
      }),
    ]);

    const domains = await prisma.strategy.groupBy({
      by: ['domain'],
      _count: { _all: true },
    });
    const domainDistribution: Record<string, number> = {};
    domains.forEach((d) => {
      domainDistribution[d.domain] = d._count._all;
    });

    // Tag popularity (sample recent strategies to keep it fast)
    const tagRows = await prisma.strategy.findMany({
      select: { tags: true },
      orderBy: { createdAt: 'desc' },
      take: 500,
    });
    const tagCounts: Record<string, number> = {};
    tagRows.forEach((r) => {
      const tags = safeParseJsonArray(r.tags);
      tags.forEach((t) => {
        tagCounts[t] = (tagCounts[t] || 0) + 1;
      });
    });
    const popularTags = Object.entries(tagCounts)
      .sort(([, a], [, b]) => b - a)
      .slice(0, 10)
      .map(([tag, count]) => ({ tag, count }));

    const totalUsers = await prisma.strategy
      .findMany({ select: { userId: true } })
      .then((rows) => new Set(rows.map((r) => r.userId)).size);

    const recentActivity = recentStrategies.map((s) => ({
      type: 'created' as const,
      strategyId: s.id,
      strategyName: s.name,
      userId: s.userId,
      userName: s.user
        ? `${s.user.firstName || ''} ${s.user.lastName || ''}`.trim() || s.user.id
        : s.userId,
      timestamp: s.createdAt,
    }));

    return NextResponse.json({
      totalStrategies,
      publicStrategies,
      totalUsers,
      popularTags,
      domainDistribution,
      recentActivity,
    });
  } catch (error) {
    console.error('[api/strategies/library/stats] GET error:', error);
    return NextResponse.json(
      { error: 'Failed to get statistics' },
      { status: 500 },
    );
  }
}
