import { NextRequest, NextResponse } from 'next/server';
import { z } from 'zod';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';
import { prisma } from '@/lib/database';

const createStrategySchema = z.object({
  name: z.string().min(1, 'Name is required').max(100),
  description: z.string().min(1, 'Description is required').max(2000),
  domain: z.enum(['sports', 'crypto', 'stocks', 'forex']).default('sports'),
  rules: z.any().optional(),
  tags: z.array(z.string()).optional(),
  isPublic: z.boolean().optional().default(false),
});

function safeParseJson<T>(value: any, fallback: T): T {
  if (!value || typeof value !== 'string') return fallback;
  try {
    return JSON.parse(value) as T;
  } catch {
    return fallback;
  }
}

// GET /api/strategies/library - Search strategies
export async function GET(request: NextRequest) {
  try {
    await AuthMiddleware.optionalAuth(request);
    const user = (request as any).user;
    const { searchParams } = new URL(request.url);

    const domains = (searchParams.get('domain') || '')
      .split(',')
      .map((d) => d.trim())
      .filter(Boolean);
    const tags = (searchParams.get('tags') || '')
      .split(',')
      .map((t) => t.trim())
      .filter(Boolean);
    const isPublicParam = searchParams.get('isPublic');
    const isPublic = isPublicParam === null ? true : isPublicParam === 'true';
    const limit = Math.min(100, Math.max(1, Number(searchParams.get('limit') || 20)));

    if (!isPublic && !user) {
      return NextResponse.json({ error: 'Authentication required' }, { status: 401 });
    }

    const where: any = {
      ...(domains.length ? { domain: { in: domains } } : {}),
      ...(isPublic ? { isPublic: true } : { isPublic: false, userId: user.id }),
    };

    const rows = await prisma.strategy.findMany({
      where,
      include: {
        user: { select: { id: true, firstName: true, lastName: true } },
      },
      orderBy: { createdAt: 'desc' },
      take: limit,
    });

    const strategies = rows
      .map((s) => ({
        id: s.id,
        name: s.name,
        description: s.description,
        domain: s.domain,
        isPublic: s.isPublic,
        tags: safeParseJson<string[]>(s.tags, []),
        rules: safeParseJson<any>(s.rules, null),
        likes: s.likes,
        forks: s.forks,
        views: s.views,
        createdAt: s.createdAt,
        updatedAt: s.updatedAt,
        author: s.user ? `${s.user.firstName || ''} ${s.user.lastName || ''}`.trim() || s.user.id : 'Unknown',
      }))
      .filter((s) => (tags.length ? tags.some((t) => s.tags.includes(t)) : true));

    return NextResponse.json({ strategies, total: strategies.length });
  } catch (error) {
    console.error('[api/strategies/library] GET error:', error);
    return NextResponse.json(
      { error: 'Failed to search strategies' },
      { status: 500 },
    );
  }
}

// POST /api/strategies/library - Create new strategy
export async function POST(request: NextRequest) {
  try {
    const authResponse = await AuthMiddleware.requireAuth(request);
    if (authResponse) return authResponse;

    const user = (request as any).user;
    const json = await request.json();
    const validatedData = createStrategySchema.parse(json);

    const strategy = await DatabaseService.createStrategy({
      name: validatedData.name,
      description: validatedData.description,
      domain: validatedData.domain as any,
      rules: validatedData.rules || {},
      tags: validatedData.tags || [],
      isPublic: validatedData.isPublic,
      userId: user.id,
    });

    await DatabaseService.recordUserEvent(user.id, 'STRATEGY_CREATED', {
      strategyId: strategy.id,
      domain: strategy.domain,
      isPublic: strategy.isPublic,
      source: 'library_api',
    });

    return NextResponse.json({ success: true, strategy }, { status: 201 });
  } catch (error) {
    if (error instanceof z.ZodError) {
      return NextResponse.json(
        { error: 'Invalid strategy data', issues: error.issues },
        { status: 422 },
      );
    }

    console.error('[api/strategies/library] POST error:', error);
    return NextResponse.json(
      { error: 'Failed to create strategy' },
      { status: 500 },
    );
  }
}
