/**
 * BallDontLie API Route
 * GET /api/sportsdata - Fetch sports data (BallDontLie)
 */

import { NextRequest, NextResponse } from 'next/server';
import { BallDontLieService } from '@/services/BallDontLieService';
import { AuthMiddleware, SecurityMiddleware } from '@/lib/auth';

// Valid leagues whitelist for input validation
const VALID_LEAGUES = [
  'nba',
  'nfl',
  'mlb',
  'nhl',
  'wnba',
  'epl',
  'ncaaf',
  'ncaab',
  'la_liga',
  'serie_a',
  'bundesliga',
  'ligue_1',
  'champions_league',
] as const;

// Circuit breaker state for external API resilience
const circuitBreaker = {
    failures: 0,
    lastFailure: 0,
    isOpen: false,
    threshold: 5,        // Open circuit after 5 failures
    resetTime: 60000,    // Reset after 60 seconds
};

function validateLeague(league: string): boolean {
  return VALID_LEAGUES.includes(league.toLowerCase() as typeof VALID_LEAGUES[number]);
}

function checkCircuitBreaker(): boolean {
    if (circuitBreaker.isOpen) {
        const timeSinceFailure = Date.now() - circuitBreaker.lastFailure;
        if (timeSinceFailure > circuitBreaker.resetTime) {
            // Reset circuit breaker (half-open state)
            circuitBreaker.isOpen = false;
            circuitBreaker.failures = 0;
            return true;
        }
        return false; // Circuit is open, reject request
    }
    return true;
}

function recordFailure(): void {
    circuitBreaker.failures++;
    circuitBreaker.lastFailure = Date.now();
    if (circuitBreaker.failures >= circuitBreaker.threshold) {
        circuitBreaker.isOpen = true;
        console.warn('🔴 Circuit breaker OPEN - BallDontLie API failures exceeded threshold');
    }
}

function recordSuccess(): void {
    circuitBreaker.failures = 0;
    circuitBreaker.isOpen = false;
}

export async function GET(request: NextRequest) {
    try {
        // Require authentication
        const authResponse = await AuthMiddleware.requireAuth(request);
        if (authResponse) return authResponse;

        // Rate limiting
        const rateLimitResponse = SecurityMiddleware.rateLimit(request, 100, 60 * 60 * 1000); // 100 requests per hour
        if (rateLimitResponse) return rateLimitResponse;

        // Circuit breaker check
        if (!checkCircuitBreaker()) {
            return NextResponse.json(
                {
                    error: 'Service temporarily unavailable',
                    message: 'BallDontLie API is experiencing issues. Please try again in 60 seconds.',
                    retryAfter: Math.ceil((circuitBreaker.resetTime - (Date.now() - circuitBreaker.lastFailure)) / 1000)
                },
                { status: 503 }
            );
        }

        // Parse query parameters
        const { searchParams } = new URL(request.url);
        const league = searchParams.get('sport') || searchParams.get('league'); // keep backward-compat with `sport`
        const resource = (searchParams.get('resource') || searchParams.get('endpoint') || 'team').toLowerCase();

        if (!league) {
            return NextResponse.json(
                { error: 'League parameter is required (use ?sport=nba|nfl|mlb|nhl|...)' },
                { status: 400 }
            );
        }

        // Validate league against whitelist
        if (!validateLeague(league)) {
            return NextResponse.json(
                {
                    error: 'Invalid league parameter',
                    validLeagues: VALID_LEAGUES
                },
                { status: 400 }
            );
        }

        const bdl = BallDontLieService.getInstance();

        // Forward unknown query params to BallDontLie
        const forwardedParams: Record<string, any> = {};
        for (const [k, v] of searchParams.entries()) {
          if (['sport', 'league', 'resource', 'endpoint'].includes(k)) continue;
          forwardedParams[k] = v;
        }

        let data: any;
        if (resource.startsWith('team')) {
          data = await bdl.getTeams(league, forwardedParams);
        } else if (resource.startsWith('player')) {
          data = await bdl.getPlayers(league, forwardedParams);
        } else if (resource.startsWith('game')) {
          data = await bdl.getGames(league, forwardedParams);
        } else if (resource.startsWith('odd')) {
          data = await bdl.getOdds(league, forwardedParams);
        } else {
          return NextResponse.json(
            { error: 'Invalid resource. Use resource=team|player|game|odds' },
            { status: 400 }
          );
        }

        // Record success for circuit breaker
        recordSuccess();

        return NextResponse.json({
            success: true,
            league: league.toUpperCase(),
            resource,
            data,
            count: Array.isArray(data)
              ? data.length
              : Array.isArray((data as any)?.data)
                ? (data as any).data.length
                : 1,
            timestamp: new Date().toISOString(),
        });

    } catch (error) {
        // Record failure for circuit breaker
        recordFailure();

        console.error('BallDontLie API error:', error);
        return NextResponse.json(
            {
                error: 'Failed to fetch sports data',
                details: process.env.NODE_ENV === 'development'
                    ? (error instanceof Error ? error.message : 'Unknown error')
                    : 'An internal error occurred'
            },
            { status: 500 }
        );
    }
}

