import { NextRequest, NextResponse } from 'next/server';
import { AuthMiddleware, DatabaseService } from '@/lib/auth';

function toSafeUpper(value: unknown, fallback: string) {
  if (typeof value === 'string' && value.trim()) return value.toUpperCase();
  return fallback.toUpperCase();
}

function toSafeDateString(value: unknown): string {
  try {
    const d = value ? new Date(value as any) : new Date();
    if (Number.isNaN(d.getTime())) return new Date().toLocaleString();
    return d.toLocaleString();
  } catch {
    return new Date().toLocaleString();
  }
}

function asNumber(value: unknown, fallback = 0): number {
  const n = typeof value === 'number' ? value : Number(value);
  return Number.isFinite(n) ? n : fallback;
}

function asString(value: unknown, fallback = ''): string {
  if (typeof value === 'string') return value;
  if (value == null) return fallback;
  return String(value);
}

function csvCell(value: unknown): string {
  const s = asString(value, '');
  // Escape quotes and wrap in quotes if needed (commas/newlines/quotes)
  const needsQuotes = /[",\n\r]/.test(s);
  const escaped = s.replace(/"/g, '""');
  return needsQuotes ? `"${escaped}"` : escaped;
}

function getTrades(backtestResults: any): any[] {
  if (Array.isArray(backtestResults?.all_trades)) return backtestResults.all_trades;
  if (Array.isArray(backtestResults?.trades)) return backtestResults.trades;
  return [];
}

function computeTradeStatsFromTrades(trades: any[]) {
  const profits = trades.map(t => asNumber(t?.profit, 0));
  const wins = trades.filter(t => (t?.outcome || '').toString().toLowerCase() === 'win').length;
  const losses = trades.length - wins;
  const winRate = trades.length ? (wins / trades.length) * 100 : 0;
  const totalProfit = profits.reduce((sum, p) => sum + p, 0);

  const winProfits = profits.filter(p => p > 0);
  const lossProfits = profits.filter(p => p < 0);

  const biggestWin = winProfits.length ? Math.max(...winProfits) : 0;
  const biggestLoss = lossProfits.length ? Math.min(...lossProfits) : 0;
  const avgWin = winProfits.length ? winProfits.reduce((s, p) => s + p, 0) / winProfits.length : 0;
  const avgLoss = lossProfits.length ? lossProfits.reduce((s, p) => s + p, 0) / lossProfits.length : 0;

  // Simple volatility: std dev of profits
  const mean = profits.length ? totalProfit / profits.length : 0;
  const variance = profits.length
    ? profits.reduce((acc, p) => acc + (p - mean) ** 2, 0) / profits.length
    : 0;
  const profitVolatility = Math.sqrt(variance);

  return {
    totalTrades: trades.length,
    wins,
    losses,
    winRate,
    totalProfit,
    biggestWin,
    biggestLoss,
    avgWin,
    avgLoss,
    profitVolatility,
  };
}

export async function POST(request: NextRequest) {
  try {
    // Optional auth: allow downloads, but track them for signed-in users
    await AuthMiddleware.optionalAuth(request);
    const user = (request as any).user;

    const body = await request.json();
    const { backtestResults, strategyName, market, timestamp, format = 'csv' } = body;

    if (!backtestResults) {
      return NextResponse.json(
        { error: 'No backtest results provided' },
        { status: 400 }
      );
    }

    if (format === 'csv') {
      // Generate CSV report
      const csvContent = generateCSVReport(backtestResults, strategyName, market, timestamp);

      // Track report download
      if (user) {
        try {
          await DatabaseService.recordUserEvent(user.id, 'REPORT_DOWNLOADED', {
            format: 'csv',
            strategyName,
            market,
            totalTrades: backtestResults?.total_trades ?? backtestResults?.results?.total_trades ?? null,
          });
        } catch (e) {
          console.warn('Failed to record REPORT_DOWNLOADED', e);
        }
      }

      const response = new NextResponse(csvContent, {
        status: 200,
        headers: {
          'Content-Type': 'text/csv',
          'Content-Disposition': `attachment; filename="eventheodds-backtest-${strategyName}-${market}-${new Date().toISOString().split('T')[0]}.csv"`
        }
      });

      return response;
    } else {
      // Generate text report
      const textContent = generateTextReport(backtestResults, strategyName, market, timestamp);

      // Track report download
      if (user) {
        try {
          await DatabaseService.recordUserEvent(user.id, 'REPORT_DOWNLOADED', {
            format: 'txt',
            strategyName,
            market,
            totalTrades: backtestResults?.total_trades ?? backtestResults?.results?.total_trades ?? null,
          });
        } catch (e) {
          console.warn('Failed to record REPORT_DOWNLOADED', e);
        }
      }

      const response = new NextResponse(textContent, {
        status: 200,
        headers: {
          'Content-Type': 'text/plain',
          'Content-Disposition': `attachment; filename="eventheodds-backtest-${strategyName}-${market}-${new Date().toISOString().split('T')[0]}.txt"`
        }
      });

      return response;
    }

  } catch (error) {
    console.error('Error generating report:', error);
    return NextResponse.json(
      { error: 'Failed to generate report' },
      { status: 500 }
    );
  }
}

function generateCSVReport(
  backtestResults: any,
  strategyName: string,
  market: string,
  timestamp: string
): string {
  let csv = '';
  const safeStrategyName = asString(strategyName, 'AI Generated Strategy');
  const safeMarket = toSafeUpper(market, 'SPORTS');
  const generatedAt = toSafeDateString(timestamp);
  const results = backtestResults?.results || backtestResults || {};
  const stats = backtestResults?.detailed_statistics || {};
  const trades = getTrades(backtestResults);
  const computed = computeTradeStatsFromTrades(trades);

  // Report Header
  csv += 'EventheOdds.ai Backtest Report\n';
  csv += `Strategy: ${safeStrategyName}\n`;
  csv += `Market: ${safeMarket}\n`;
  csv += `Generated: ${generatedAt}\n`;
  csv += '\n';

  // Summary Statistics
  csv += 'SUMMARY STATISTICS\n';
  csv += `Win Rate %,${asNumber(results.win_rate, computed.winRate).toFixed(2)}\n`;
  csv += `Total Profit,$${asNumber(results.total_profit, computed.totalProfit).toFixed(2)}\n`;
  csv += `Profit Factor,${asNumber(results.profit_factor, 0).toFixed(2)}\n`;
  csv += `Sharpe Ratio,${asNumber(results.sharpe_ratio, 0).toFixed(2)}\n`;
  csv += `Max Drawdown,$${asNumber(results.max_drawdown, 0).toFixed(2)}\n`;
  csv += `Total Trades,${asNumber(backtestResults?.total_trades, computed.totalTrades)}\n`;
  csv += '\n';

  // Detailed Statistics
  const tradeAnalysis = stats?.trade_analysis || {};
  const profitLossAnalysis = stats?.profit_loss_analysis || {};
  const riskMetrics = stats?.risk_metrics || {};

  // Only include these sections if we have either nested stats or trades to compute from
  if (Object.keys(stats || {}).length > 0 || trades.length > 0) {
    csv += 'TRADE ANALYSIS\n';
    csv += `Total Trades,${asNumber(tradeAnalysis.total_trades, computed.totalTrades)}\n`;
    csv += `Winning Trades,${asNumber(tradeAnalysis.winning_trades, computed.wins)}\n`;
    csv += `Losing Trades,${asNumber(tradeAnalysis.losing_trades, computed.losses)}\n`;
    csv += `Win Rate %,${asNumber(tradeAnalysis.win_rate_percent, asNumber(tradeAnalysis.win_rate, computed.winRate)).toFixed(2)}\n`;
    csv += '\n';

    csv += 'PROFIT/LOSS ANALYSIS\n';
    csv += `Total Profit,$${asNumber(profitLossAnalysis.total_profit, asNumber(stats.total_profit, computed.totalProfit)).toFixed(2)}\n`;
    csv += `Biggest Win,$${asNumber(profitLossAnalysis.biggest_win, computed.biggestWin).toFixed(2)}\n`;
    csv += `Biggest Loss,$${asNumber(profitLossAnalysis.biggest_loss, computed.biggestLoss).toFixed(2)}\n`;
    csv += `Average Win,$${asNumber(profitLossAnalysis.average_win, computed.avgWin).toFixed(2)}\n`;
    csv += `Average Loss,$${asNumber(profitLossAnalysis.average_loss, computed.avgLoss).toFixed(2)}\n`;
    csv += '\n';

    csv += 'RISK METRICS\n';
    csv += `Sharpe Ratio,${asNumber(riskMetrics.sharpe_ratio, asNumber(stats.sharpe_ratio, asNumber(results.sharpe_ratio, 0))).toFixed(2)}\n`;
    csv += `Sortino Ratio,${asNumber(riskMetrics.sortino_ratio, 0).toFixed(2)}\n`;
    csv += `Max Drawdown,$${asNumber(riskMetrics.max_drawdown, asNumber(stats.max_drawdown, asNumber(results.max_drawdown, 0))).toFixed(2)}\n`;
    csv += `Profit Volatility,$${asNumber(riskMetrics.profit_volatility, computed.profitVolatility).toFixed(2)}\n`;
    csv += '\n';
  }

  // Strategy Equation
  if (backtestResults.strategyEquation) {
    csv += 'STRATEGY EQUATION\n';
    csv += `"${backtestResults.strategyEquation}"\n`;
    csv += '\n';
  }

  // Trade Data
  if (trades.length > 0) {
    csv += 'TRADE DATA\n';
    csv += 'Date,Action,Outcome,Profit ($),Stake ($)\n';

    for (const trade of trades) {
      csv += `${csvCell(trade?.date || 'N/A')},${csvCell(trade?.action || 'N/A')},${csvCell(trade?.outcome || 'N/A')},${asNumber(trade?.profit, 0).toFixed(2)},${asNumber(trade?.stake, 0).toFixed(2)}\n`;
    }
    csv += '\n';
  }

  // Walk Forward Analysis
  if (backtestResults.walk_forward_analysis) {
    const wf = backtestResults.walk_forward_analysis;
    csv += 'WALK FORWARD ANALYSIS\n';

    if (wf.overall_statistics) {
      const overall = wf.overall_statistics;
      csv += 'Overall Statistics\n';
      csv += `Total Periods,${overall.total_periods}\n`;
      csv += `Average Win Rate %,${overall.average_win_rate}\n`;
      csv += `Average Monthly P&L,$${overall.average_monthly_pnl}\n`;
      csv += `Total P&L,$${overall.total_pnl}\n`;
      csv += `Profitable Periods %,${overall.profitable_periods_percent}\n`;
      csv += '\n';
    }

    if (wf.periods) {
      csv += 'Period-by-Period Results\n';
      csv += 'Period,In-Sample Trades,In-Sample Win %,In-Sample P&L,Out-of-Sample Trades,Out-of-Sample Win %,Out-of-Sample P&L,Overfitting Risk\n';

      for (const period of wf.periods) {
        const ins = period.in_sample_period.metrics;
        const outs = period.out_of_sample_period.metrics;
        const risk = period.overfitting_analysis.overfitting_risk_level;

        csv += `${period.period_number},${ins.total_trades},${ins.win_rate},${ins.total_pnl},${outs.total_trades},${outs.win_rate},${outs.total_pnl},${risk}\n`;
      }
    }
  }

  return csv;
}

function generateTextReport(
  backtestResults: any,
  strategyName: string,
  market: string,
  timestamp: string
): string {
  let report = '';
  const safeStrategyName = asString(strategyName, 'AI Generated Strategy');
  const safeMarket = toSafeUpper(market, 'SPORTS');
  const generatedAt = toSafeDateString(timestamp);
  const results = backtestResults?.results || backtestResults || {};
  const stats = backtestResults?.detailed_statistics || {};
  const trades = getTrades(backtestResults);
  const computed = computeTradeStatsFromTrades(trades);

  report += '='.repeat(80) + '\n';
  report += '                    EventheOdds.ai Backtest Report\n';
  report += '='.repeat(80) + '\n\n';

  report += `Strategy: ${safeStrategyName}\n`;
  report += `Market: ${safeMarket}\n`;
  report += `Generated: ${generatedAt}\n\n`;

  // Executive Summary
  report += 'EXECUTIVE SUMMARY\n';
  report += '-'.repeat(50) + '\n';
  report += `Win Rate:              ${asNumber(results.win_rate, computed.winRate).toFixed(2)}%\n`;
  report += `Total Profit:          $${asNumber(results.total_profit, computed.totalProfit).toFixed(2)}\n`;
  report += `Profit Factor:         ${asNumber(results.profit_factor, 0).toFixed(2)}\n`;
  report += `Sharpe Ratio:          ${asNumber(results.sharpe_ratio, 0).toFixed(2)}\n`;
  report += `Max Drawdown:          $${asNumber(results.max_drawdown, 0).toFixed(2)}\n`;
  report += `Total Trades:          ${asNumber(backtestResults?.total_trades, computed.totalTrades)}\n`;
  report += '\n';

  // Detailed Statistics
  const tradeAnalysis = stats?.trade_analysis || {};
  const profitLossAnalysis = stats?.profit_loss_analysis || {};
  const riskMetrics = stats?.risk_metrics || {};

  if (Object.keys(stats || {}).length > 0 || trades.length > 0) {
    report += 'TRADE ANALYSIS\n';
    report += '-'.repeat(50) + '\n';
    report += `Total Trades:          ${asNumber(tradeAnalysis.total_trades, computed.totalTrades)}\n`;
    report += `Winning Trades:        ${asNumber(tradeAnalysis.winning_trades, computed.wins)}\n`;
    report += `Losing Trades:         ${asNumber(tradeAnalysis.losing_trades, computed.losses)}\n`;
    report += `Win Rate:              ${asNumber(tradeAnalysis.win_rate_percent, asNumber(tradeAnalysis.win_rate, computed.winRate)).toFixed(2)}%\n\n`;

    report += 'PROFIT/LOSS ANALYSIS\n';
    report += '-'.repeat(50) + '\n';
    report += `Total Profit:          $${asNumber(profitLossAnalysis.total_profit, asNumber(stats.total_profit, computed.totalProfit)).toFixed(2)}\n`;
    report += `Biggest Win:           $${asNumber(profitLossAnalysis.biggest_win, computed.biggestWin).toFixed(2)}\n`;
    report += `Biggest Loss:          $${asNumber(profitLossAnalysis.biggest_loss, computed.biggestLoss).toFixed(2)}\n`;
    report += `Average Win:           $${asNumber(profitLossAnalysis.average_win, computed.avgWin).toFixed(2)}\n`;
    report += `Average Loss:          $${asNumber(profitLossAnalysis.average_loss, computed.avgLoss).toFixed(2)}\n\n`;

    report += 'RISK METRICS\n';
    report += '-'.repeat(50) + '\n';
    report += `Sharpe Ratio:          ${asNumber(riskMetrics.sharpe_ratio, asNumber(stats.sharpe_ratio, asNumber(results.sharpe_ratio, 0))).toFixed(2)}\n`;
    report += `Sortino Ratio:         ${asNumber(riskMetrics.sortino_ratio, 0).toFixed(2)}\n`;
    report += `Max Drawdown:          $${asNumber(riskMetrics.max_drawdown, asNumber(stats.max_drawdown, asNumber(results.max_drawdown, 0))).toFixed(2)}\n`;
    report += `Profit Volatility:     $${asNumber(riskMetrics.profit_volatility, computed.profitVolatility).toFixed(2)}\n\n`;
  }

  // Strategy Equation
  if (backtestResults.strategyEquation) {
    report += 'STRATEGY DETAILS\n';
    report += '-'.repeat(50) + '\n';
    report += 'Strategy Equation:\n';
    report += `${backtestResults.strategyEquation}\n\n`;
  }

  // Recent Trades
  if (trades.length > 0) {
    report += 'RECENT TRADES\n';
    report += '-'.repeat(50) + '\n';
    report += 'Date              Action    Outcome    Profit ($)\n';
    report += '-'.repeat(50) + '\n';

    const recentTrades = trades.slice(-20); // Last 20 trades
    for (const trade of recentTrades) {
      const date = asString(trade?.date || 'N/A', 'N/A').substring(0, 10);
      const action = asString(trade?.action || 'N/A', 'N/A').padEnd(8);
      const outcome = asString(trade?.outcome || 'N/A', 'N/A').padEnd(8);
      const profit = asNumber(trade?.profit, 0).toFixed(2).padStart(10);
      report += `${date}  ${action}  ${outcome}  ${profit}\n`;
    }
    report += '\n';
  }

  // Walk Forward Analysis
  if (backtestResults.walk_forward_analysis) {
    const wf = backtestResults.walk_forward_analysis;
    report += 'WALK FORWARD ANALYSIS\n';
    report += '-'.repeat(50) + '\n';

    if (wf.overall_statistics) {
      const overall = wf.overall_statistics;
      report += 'Overall Performance:\n';
      report += `  Periods Tested:      ${overall.total_periods}\n`;
      report += `  Average Win Rate:    ${overall.average_win_rate}%\n`;
      report += `  Average Monthly P&L: $${overall.average_monthly_pnl}\n`;
      report += `  Profitable Periods:  ${overall.profitable_periods_percent}%\n\n`;
    }

    report += 'Benefits of Walk Forward Analysis:\n';
    report += '• Prevents overfitting by testing on unseen future data\n';
    report += '• Validates strategy robustness across different market conditions\n';
    report += '• Provides realistic performance expectations\n';
    report += '• Identifies parameter stability and reliability\n\n';
  }

  // Footer
  report += '='.repeat(80) + '\n';
  report += 'Report generated by EventheOdds.ai - Sports Betting Backtesting Platform\n';
  report += 'For questions or support, visit eventheodds.ai\n';
  report += '='.repeat(80) + '\n';

  return report;
}
