/**
 * NBA Teams API Endpoint
 * GET /api/nba/teams - Get NBA teams data
 */

import { NextRequest } from 'next/server';
import { authenticateAndAuthorize } from '@/lib/auth';
import { SubscriptionTier } from '@/lib/types';
import { handleApiError } from '@/lib/error-handler';
import { DatabaseService } from '@/lib/database';
import path from 'path';
import { spawn } from 'child_process';

export async function GET(request: NextRequest) {
  try {
    const url = new URL(request.url);
    const limit = parseInt(url.searchParams.get('limit') || '30');
    const abbreviation = url.searchParams.get('abbreviation');
    const useCache = url.searchParams.get('cache') !== 'false';

    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);
    if (authResult.status !== 200) return authResult.response;

    console.log(`🔍 NBA Teams API: limit=${limit}, abbreviation=${abbreviation}`);

    let teamsData = await callNBAService('teams', { use_cache: useCache });

    if (!Array.isArray(teamsData)) {
      return new Response(JSON.stringify({ error: 'Failed to fetch NBA teams' }), {
        status: 500,
        headers: { 'Content-Type': 'application/json' },
      });
    }

    // Filter by abbreviation if specified
    if (abbreviation) {
      teamsData = teamsData.filter((team: any) =>
        String(team.abbreviation || team.Abbreviation || '').toLowerCase() === abbreviation.toLowerCase()
      );
    }

    // Apply limit
    teamsData = teamsData.slice(0, limit);

    // Record analytics
    if (authResult.data?.user) {
      await DatabaseService.recordUserEvent(authResult.data.user.id, 'NBA_DATA_ACCESSED', {
        endpoint: 'teams',
        cached: useCache,
        abbreviation,
        returned: teamsData.length,
      });
    }

    return new Response(JSON.stringify({
      success: true,
      data: teamsData,
      meta: {
        total: teamsData.length,
        filtered: teamsData.length,
        limit,
        abbreviation,
        lastUpdated: new Date().toISOString()
      }
    }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' }
    });

  } catch (error) {
    console.error('NBA Teams API error:', error);
    return handleApiError(error as Error, request);
  }
}

async function callNBAService(endpoint: string, params: any = {}): Promise<any> {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(process.cwd(), 'nba_data_service.py');

    const args = [
      scriptPath,
      endpoint,
      JSON.stringify(params)
    ];

    const pythonProcess = spawn('python3', args, {
      cwd: process.cwd(),
      env: {
        ...process.env,
        PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages')
      }
    });

    let stdout = '';
    let stderr = '';

    pythonProcess.stdout.on('data', (data) => {
      stdout += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      stderr += data.toString();
    });

    pythonProcess.on('close', (code) => {
      if (code !== 0) {
        console.error(`Python process failed with code ${code}:`, stderr);
        reject(new Error(`NBA service failed: ${stderr}`));
        return;
      }

      try {
        const result = JSON.parse(stdout.trim());
        resolve(result);
      } catch (parseError) {
        console.error('Failed to parse NBA service response:', stdout);
        reject(new Error('Invalid response from NBA service'));
      }
    });

    pythonProcess.on('error', (error) => {
      console.error('Failed to start NBA service:', error);
      reject(error);
    });
  });
}
