/**
 * NBA Players API Endpoint
 * GET /api/nba/players - Get NBA players data
 */

import { NextRequest } from 'next/server';
import { authenticateAndAuthorize } from '@/lib/auth';
import { SubscriptionTier } from '@/lib/types';
import { handleApiError } from '@/lib/error-handler';
import { DatabaseService } from '@/lib/database';
import { spawn } from 'child_process';
import path from 'path';

export async function GET(request: NextRequest) {
  try {
    const url = new URL(request.url);
    const limit = parseInt(url.searchParams.get('limit') || '50');
    const offset = parseInt(url.searchParams.get('offset') || '0');
    const useCache = url.searchParams.get('cache') !== 'false';
    const activeOnly = url.searchParams.get('activeOnly') !== 'false';

    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);
    if (authResult.status !== 200) return authResult.response;

    console.log(`🔍 NBA Players API: limit=${limit}, offset=${offset}`);

    const playersData = await callNBAService('players', {
      use_cache: useCache,
      active_only: activeOnly,
      limit,
      offset,
    });

    if (!Array.isArray(playersData)) {
      return new Response(JSON.stringify({ error: 'Failed to fetch NBA players' }), {
        status: 500,
        headers: { 'Content-Type': 'application/json' },
      });
    }

    // Record analytics
    if (authResult.data?.user) {
      await DatabaseService.recordUserEvent(authResult.data.user.id, 'NBA_DATA_ACCESSED', {
        endpoint: 'players',
        cached: useCache,
        activeOnly,
        returned: playersData.length,
      });
    }

    return new Response(JSON.stringify({
      success: true,
      data: playersData,
      meta: {
        total: playersData.length,
        limit,
        offset,
        lastUpdated: new Date().toISOString()
      }
    }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' }
    });

  } catch (error) {
    console.error('NBA Players API error:', error);
    return handleApiError(error as Error, request);
  }
}

async function callNBAService(endpoint: string, params: any = {}): Promise<any> {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(process.cwd(), 'nba_data_service.py');

    const args = [
      scriptPath,
      endpoint,
      JSON.stringify(params),
    ];

    const pythonProcess = spawn('python3', args, {
      cwd: process.cwd(),
      env: {
        ...process.env,
        PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages'),
      },
    });

    let stdout = '';
    let stderr = '';

    pythonProcess.stdout.on('data', (data) => {
      stdout += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      stderr += data.toString();
    });

    pythonProcess.on('close', (code) => {
      if (code !== 0) {
        console.error(`Python process failed with code ${code}:`, stderr);
        reject(new Error(`NBA service failed: ${stderr}`));
        return;
      }

      try {
        const result = JSON.parse(stdout.trim());
        resolve(result);
      } catch (parseError) {
        console.error('Failed to parse NBA service response:', stdout);
        reject(new Error('Invalid response from NBA service'));
      }
    });

    pythonProcess.on('error', (err) => {
      console.error('Failed to start NBA service:', err);
      reject(err);
    });
  });
}
