/**
 * NBA Games API Endpoint
 * GET /api/nba/games - Get NBA games data
 * GET /api/nba/games/live - Get live games data
 */

import { NextRequest } from 'next/server';
import { authenticateAndAuthorize } from '@/lib/auth';
import { SubscriptionTier } from '@/lib/types';
import { handleApiError } from '@/lib/error-handler';
import { DatabaseService } from '@/lib/database';
import { spawn } from 'child_process';
import path from 'path';

export async function GET(request: NextRequest) {
  try {
    // Require authentication
    const authResult = await authenticateAndAuthorize(request, [
      SubscriptionTier.FREE,
      SubscriptionTier.TRIAL,
      SubscriptionTier.PRO,
      SubscriptionTier.ELITE,
    ]);
    if (authResult.status !== 200) return authResult.response;

    const url = new URL(request.url);
    const pathname = url.pathname;

    // Check if this is a live games request
    const isLive = pathname.includes('/live');

    const limit = parseInt(url.searchParams.get('limit') || '20');
    const offset = parseInt(url.searchParams.get('offset') || '0');
    const season = url.searchParams.get('season') || '2023-24';
    const seasonType = url.searchParams.get('seasonType') || 'Regular Season';
    const useCache = url.searchParams.get('cache') !== 'false';

    console.log(`🔍 NBA Games API: live=${isLive}, limit=${limit}, offset=${offset}`);

    let gamesData: any;

      if (isLive) {
      // Get live games from NBA service (always fresh)
      gamesData = await callNBAService('live_games', { use_cache: false });
      if (!gamesData || gamesData.error) {
        return new Response(JSON.stringify({ error: gamesData?.error || 'Failed to fetch live games' }), {
          status: 500,
          headers: { 'Content-Type': 'application/json' },
        });
      }

      // Record analytics
      if (authResult.data?.user) {
        await DatabaseService.recordUserEvent(authResult.data.user.id, 'NBA_DATA_ACCESSED', {
          endpoint: 'live_games',
          cached: false,
          returned: gamesData.games?.length || 0,
        });
      }
      } else {
      const allGames = await callNBAService('games', {
        use_cache: useCache,
        season,
        season_type: seasonType,
      });

      if (!Array.isArray(allGames)) {
        return new Response(JSON.stringify({ error: 'Failed to fetch NBA games' }), {
          status: 500,
          headers: { 'Content-Type': 'application/json' },
        });
      }

      gamesData = allGames.slice(offset, offset + limit);

      // Record analytics
      if (authResult.data?.user) {
        await DatabaseService.recordUserEvent(authResult.data.user.id, 'NBA_DATA_ACCESSED', {
          endpoint: 'games',
          cached: useCache,
          season,
          seasonType,
          returned: gamesData.length,
        });
      }
    }

    return new Response(JSON.stringify({
      success: true,
      data: gamesData,
      meta: {
        total: isLive ? (gamesData?.games?.length || 0) : gamesData.length,
        limit,
        offset,
        isLive,
        lastUpdated: new Date().toISOString()
      }
    }), {
      status: 200,
      headers: { 'Content-Type': 'application/json' }
    });

  } catch (error) {
    console.error('NBA Games API error:', error);
    return handleApiError(error as Error, request);
  }
}

async function callNBAService(endpoint: string, params: any = {}): Promise<any> {
  return new Promise((resolve, reject) => {
    const scriptPath = path.join(process.cwd(), 'nba_data_service.py');

    const args = [
      scriptPath,
      endpoint,
      JSON.stringify(params)
    ];

    const pythonProcess = spawn('python3', args, {
      cwd: process.cwd(),
      env: {
        ...process.env,
        PYTHONPATH: path.join(process.cwd(), 'nba_api_env', 'lib', 'python3.13', 'site-packages')
      }
    });

    let stdout = '';
    let stderr = '';

    pythonProcess.stdout.on('data', (data) => {
      stdout += data.toString();
    });

    pythonProcess.stderr.on('data', (data) => {
      stderr += data.toString();
    });

    pythonProcess.on('close', (code) => {
      if (code !== 0) {
        console.error(`Python process failed with code ${code}:`, stderr);
        reject(new Error(`NBA service failed: ${stderr}`));
        return;
      }

      try {
        const result = JSON.parse(stdout.trim());
        resolve(result);
      } catch (parseError) {
        console.error('Failed to parse NBA service response:', stdout);
        reject(new Error('Invalid response from NBA service'));
      }
    });

    pythonProcess.on('error', (error) => {
      console.error('Failed to start NBA service:', error);
      reject(error);
    });
  });
}
